/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.math.BigInteger;
import java.util.Map;
import java.util.Observable;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_ERROR;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByEPOSController;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * A general model for a controller EPOS.
 *
 * @author virieux
 */
public class CanOpenEPOS extends CanOpenDevice implements EPOSController {
    
    private boolean turnedOff;
    
    /**
     * The value of current that is actually sent to the motor. UNIT= mA given
     * in decimal mode
     */
    private int actualCurrent;


    private int profileVelocity;

    /**
     * used in checkParameters and initializeAndCheckHardware methods
     */
    private boolean parametersOK;



    /* Mode of Operation */
    protected EposMode mode;
    private String modeInString;

    /* Parameters */
    private final Map<String, Integer> paramsForHoming;
    private final Map<String, Integer> paramsForProfilePosition;
    private final Map<String, Integer> paramsForCurrent;

    /**
     * Used to publish on the status bus for the GUI*
     */
    private boolean enabledToPublish;
    private boolean inError;
    private String[] errorHistory;
    private String errorRegister;

    /**
     * Construction of a CanOpenEPOS.
     * @param aName
     * @param aTickMillis
     * @param nodeID
     * @param serialNB
     * @param paramsForCurrent
     * @param paramsForProfilePosition
     * @param paramsForHoming 
     */
    public CanOpenEPOS(String aName, int aTickMillis,
            String nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent,
            Map<String, Integer> paramsForProfilePosition,
            Map<String, Integer> paramsForHoming) {
        super(aName, aTickMillis, nodeID, serialNB);
        this.paramsForCurrent = paramsForCurrent;
        this.paramsForProfilePosition = paramsForProfilePosition;
        this.paramsForHoming = paramsForHoming;
        this.actualCurrent = 0;
        turnedOff = false;
        parametersOK = true;
        enabledToPublish = false;
        inError = false;
        errorRegister = NO_ERROR;
        errorHistory = new String[0];
    }

    @Override
    public boolean isEnabledToPublish() {
        return enabledToPublish;
    }

    @Override
    public String getModeInString() {
        return modeInString;
    }

    @Override
    public boolean isTurnedOff() {
        return turnedOff;
    }
    
    
    
    /**
     * 
     * @return true if parameters in the CPU of the controllers
     * have the same values than in configuration system.
     */
    @Override
    public boolean isParametersOK() {
        return parametersOK;
    }

    public void setParametersOK(boolean parametersOK) {
        this.parametersOK = parametersOK;
    }

    /**
     * For the GUI
     *
     * @return a list of Errors - can be empty
     */
    @Override
    public String[] getErrorHistory() {
        if (errorHistory == null) {
            return new String[0];
        } else {
            return errorHistory.clone();
        }
    }

    /**
     * For the GUI
     *
     * @return
     */
    @Override
    public String getErrorRegister() {
        return errorRegister;
    }

    /**
     * For the GUI
     *
     * @return
     */
    @Override
    public boolean isInError() {
        return inError;
    }

    @Override
    public Map<String, Integer> getParamsForHoming() {
        return paramsForHoming;
    }

    @Override
    public Map<String, Integer> getParamsForProfilePosition() {
        return paramsForProfilePosition;
    }

    @Override
    public Map<String, Integer> getParamsForCurrent() {
        return paramsForCurrent;
    }

    @Override
    public int getMaxCurrent() throws FcsHardwareException {
        try {
            return readParameter(Parameter.OutputCurrentLimit);
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            FCSLOG.error(name + ":Error in reading Parameter ");
            FCSLOG.error(ex);
        }
        return 0;
    }

    @Override
    public int getMinCurrent() throws FcsHardwareException {
        try {
            return readParameter(Parameter.OutputCurrentLimit);
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            FCSLOG.error(name + ":Error in reading Parameter ");
            FCSLOG.error(ex);
        }
        return 0;
    }

    @Override
    public int getMaxSpeed() throws FcsHardwareException {
        try {
            return readParameter(Parameter.MaxProfileVelocity);
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            FCSLOG.error(name + ":Error in reading Parameter ");
            FCSLOG.error(ex);
        }
        return 0;
    }

    @Override
    public int getMaxPosition() throws FcsHardwareException {
        try {
            return readParameter(Parameter.MaxPositionLimit);
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            FCSLOG.error(name + ":Error in reading Parameter ");
            FCSLOG.error(ex);
        }
        return 0;
    }

    @Override
    public int getMinPosition() throws FcsHardwareException {
        try {
            return readParameter(Parameter.MinPositionLimit);
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            FCSLOG.error(name + ":Error in reading Parameter ");
            FCSLOG.error(ex);
        }
        return 0;
    }








    /**
     * Return the value configured for a motor parameter.
     * Doesn't read the CPU of the controller.
     * @param parameterName
     * @return
     * @throws EPOSConfigurationException 
     */
    public int getMotorParameterValue(String parameterName) throws EPOSConfigurationException {
        if (this.paramsForCurrent.containsKey(parameterName)) {
            return this.paramsForCurrent.get(parameterName);
        } else {
            throw new EPOSConfigurationException(parameterName + " :undefined", getName());
        }
    }

    /**
     * Print the configured parameters for this EPOS for a mode name given in
     * argument. This methods doesnt't read the CPU of the EPOS. So if no
 writeParameters(modeIS) command has been done before, the values in
 the EPOS CPU can be different.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print the configured parameters for this EPOS for this mode."
            + "This command doesn't read the controlle.")
    public String printParameters() throws FcsHardwareException {
        if (mode == null) {
            mode = readMode();
        }
        return printConfigurationParameters(this.mode.toString());
    }

    /**
     * Print the configured parameters for this EPOS for a mode name given in
     * argument. This methods doesnt't read the CPU of the EPOS. So if no
 writeParameters(modeIS) command has been done before, the values in
 the EPOS CPU can be different.
     *
     * @param modeInString
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print the configured parameters for this EPOS for a mode name "
            + "given in argument. This command doesn't read the controller.")
    public String printConfigurationParameters(String modeInString) {
        EposMode aMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        Map<String, Integer> paramsMap = null;
        switch (aMode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;
                
            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
                
            default:
                throw new IllegalArgumentException(modeInString +"  is not a valid EPOS mode");
        }
        return printParameters(paramsMap);
    }

    public String printParameters(Map<String, Integer> paramMap) {
        StringBuilder sb = new StringBuilder(name);
        sb.append("Parameters in this map in configuration file:\n");
        if (paramMap == null) {
            sb.append("NO parameter found in configuration file.");
        } else {
            for (Map.Entry<String, Integer> entry : paramMap.entrySet()) {
                String paramName = entry.getKey();
                int value = entry.getValue();
                sb.append("key=");
                sb.append(paramName);
                sb.append("/value=");
                sb.append(value);
                sb.append("\n");
            }
        }
        return sb.toString();
    }

    /**
     * Reads the EPOS mode in the CPU of the EPOS device, updates the field mode
     * and returns the mode as a String.
     *
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads the EPOS mode in the CPU of the EPOS device and "
            + "updates the field mode.")
    @Override
    public EposMode readMode() throws SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        String modeInHexa;
        modeInHexa = this.readSDO("6061", "0").trim().toUpperCase();
        EposMode readMode = EposMode.getMode(modeInHexa);
        this.mode = readMode;
        this.modeInString = readMode.toString();
        return readMode;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Print the Epos controller mode without reading controller CPU.")
    public String printMode() {
        return mode.toString();
    }

    /**
     * Changes the mode to the new mode given as a String argument.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3, 
            description = "Change EPOS mode .")
    public String changeMode(String modeInString) throws FcsHardwareException {
        EposMode newMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        return changeMode(newMode);
    }

    /**
     * This methods changes the mode to the new mode given as an argument. It
     * writes the new mode in the CPU of the EPOS controller and updates the
     * field mode.
     *
     * @param newMode
     * @return
     * @throws SDORequestException
     */
    @Override
    public String changeMode(EposMode newMode) throws FcsHardwareException {
        //TODO check if the mode is already newMode don't change anything.
        this.writeSDO("6060", "0", "1", newMode.getModeInHexa());
        this.mode = newMode;
        this.modeInString = newMode.toString();
        publishData();
        return getName() + " mode has been changed to: " + newMode.toString()
                + ". Please check parameters associated with this mode.";
    }

    /**
     * Displays the list of parameters that we have to define for a given mode.
     *
     * @param modeInString
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Displays the list of parameters that we have to define for a given mode.")
    public static String displayListParameters(String modeInString) {
        StringBuilder sb = new StringBuilder("List of parameters for mode: ");
        sb.append(modeInString);
        sb.append("\n");
        Parameter[] params = EposMode.valueOf(modeInString).getParameters();
        for (Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Displays the list of parameters.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Displays the list of parameters.")
    public static String displayListParameters() {
        StringBuilder sb = new StringBuilder("List of parameters : ");
        sb.append("\n");
        Parameter[] params = Parameter.values();
        for (Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * This methods reads in the CPU of the EPOS the values of the parameters
     * for a given mode.
     *
     * @param modeInString
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "This methods reads in the CPU of the EPOS the values "
                    + "of the parameters for a given mode.")
    public String readParameters(String modeInString) throws FcsHardwareException {
        return readParameters(EposMode.valueOf(modeInString));
    }

    @Override
    public String readParameters(EposMode aMode) throws FcsHardwareException {
        String modeIS = aMode.toString();
        StringBuilder sb = new StringBuilder(name + " parameters VALUES in decimal format for mode ");
        sb.append(modeIS);
        sb.append("\n");
        Parameter[] params = aMode.getParameters();
        for (Parameter param : params) {
            sb.append(param.toString());
            sb.append("=");
            int valueInt = readParameter(param);
            sb.append(valueInt);
            sb.append("\n");
        }
        FCSLOG.debug(name + sb.toString());
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the parameters for the actual mode.")
    public String readParameters() throws FcsHardwareException {
        String modeStr = readMode().toString();
        return readParameters(modeStr);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This methods writes in the CPU of the EPOS device the values "
            + "of the parameters set for the mode.")
    public void writeParameters() throws ErrorInCommandExecutionException, FcsHardwareException {
        readMode();
        writeParameters(this.mode);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "This methods writes in the CPU of the EPOS device the values "
            + "of the parameters set for the mode given as argument")
    public void writeParameters(String modeInString) throws ErrorInCommandExecutionException,
            FcsHardwareException {
        EposMode aMode = EposMode.valueOf(modeInString);
        writeParameters(aMode);
    }

    /**
     * This methods writes in the CPU of the EPOS devis the values of the
     * parameters set for the mode. Usually the values of the parameters are
     * given by the configuration system.
     *
     * @param mode
     * @throws SDORequestException
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     */
    @Override
    public void writeParameters(EposMode mode) throws ErrorInCommandExecutionException, FcsHardwareException {
        String modeIS = mode.toString();
        Map<String, Integer> paramsMap = null;
        switch (mode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
            
            default:
                assert false : mode;
        }
        if (paramsMap == null) {
            throw new ErrorInCommandExecutionException(name
                    + "parameters for mode :" + modeIS + "are NULL");
        }
        if (paramsMap.isEmpty()) {
            throw new ErrorInCommandExecutionException(name
                    + "parameters for mode :" + modeIS + "are not defined.");
        }
        writeParameters(paramsMap);

    }

    /**
     * Write to the CPU of the EPOS a map of parameters.
     *
     * @param paramMap
     * @throws SDORequestException
     */
    @Override
    public void writeParameters(Map<String, Integer> paramMap) 
            throws FcsHardwareException {
        for (Map.Entry<String, Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue();
            writeParameter(paramName, value);
        }
    }

    /**
     * This method writes a parameter in the CPU of the EPOS.
     *
     * @param parameterName the name of the parameter
     * @param value in decimal
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set EPOS parameter with the value given as argument.")
    public String writeParameter(String parameterName, int value) 
            throws FcsHardwareException {
        Parameter param = Parameter.valueOf(parameterName);
        return this.writeParameter(param, value);
    }

    /**
     * Set the parameter given as argument with the decimal value given as
     * argument and write it in the controller CPU.
     *
     * @param param parameter to be set
     * @param value decimal value
     * @return
     * @throws SDORequestException
     */
    public String writeParameter(Parameter param, int value) 
            throws FcsHardwareException {
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), 
                Integer.toHexString(value));
        return String.format("%s has been written to the CanEpos CPU, value=%d", 
                param.toString(), value);
    }

    /**
     * This method writes a parameter in the CPU of the EPOS.
     *
     * @param parameterName the name of the parameter
     * @param hexaValue value of the parameter given in hexa
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set EPOS parameter with the hexadecimal value given as argument.")
    public String writeParameterInHexa(String parameterName, String hexaValue)
            throws FcsHardwareException {
        Parameter param = Parameter.valueOf(parameterName);
        return this.writeParameterInHexa(param, hexaValue);
    }

    /**
     * Set the parameter given as argument with the hexadecimal value given as
     * argument and write it in the controller CPU.
     *
     * @param param
     * @param hexaValue
     * @return
     * @throws SDORequestException
     */
    @Override
    public String writeParameterInHexa(Parameter param, String hexaValue) 
            throws FcsHardwareException {
        writeSDO(param.getIndex(), param.getSubindex(), param.getSize(), hexaValue);
        return String.format("%s has been written to the CanEpos CPU, value=%s", 
                param.toString(), hexaValue);
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter which parameter name is
     * given as argument.
     *
     * @param parameterName name of the parameter to read
     * @return value of the parameter in decimal format.
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads in the EPOS CPU the decimal value of the Parameter "
            + "which parameter name is given as argument.")
    public int readParameter(String parameterName) throws FcsHardwareException {
        Parameter param = Parameter.valueOf(parameterName);
        return readParameter(param);
    }
    

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in decimal format.
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public int readParameter(Parameter param) throws FcsHardwareException {
        String valueInHexa = readSDO(param.getIndex(), param.getSubindex());
        int valueInDecimal = new BigInteger(valueInHexa, 16).intValue();
        FCSLOG.debug(name + " readParameter:" + param.name() + "=" + valueInDecimal
                + " in decimal format  (valueInHexa=" + valueInHexa + ")");
        return valueInDecimal;
    }

    /**
     * Reads in the EPOS CPU the hexadecimal value of the Parameter.
     *
     * @param parameterName
     * @return value of the parameter in hexa format.
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Reads in the EPOS CPU the hexadecimal value of the Parameter "
            + "which parameter name is given as argument.")
    public String readParameterInHexa(String parameterName) throws FcsHardwareException {
        Parameter param = Parameter.valueOf(parameterName);
        return readParameterInHexa(param);
    }

    /**
     * Reads in the EPOS CPU the value of the Parameter.
     *
     * @param param
     * @return value of the parameter in hexa format.
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public String readParameterInHexa(Parameter param) throws FcsHardwareException {
        return readSDO(param.getIndex(), param.getSubindex());
    }

    @Override
    public boolean isTargetPositionReached(int position)
            throws FcsHardwareException {
        return this.readPosition() == position;
    }

    @Override
    public void initModule() {
        super.initModule();
        //listens to tcpProxy to detect the emergency messages.
        this.listens((Observable) tcpProxy);
    }

    /**
     * What to do when the Modules we observe send there new values.
     *
     * @param source
     * @param v
     */
    //TODO pull up to CanOpenDevice
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.debug(name + ":processUpdate from source=" + source.toString()
                + " ValueUpdate=" + v.getName());
        
        if (!(source instanceof CanOpenProxy)) return;
        /** observes only CanOpenProxy to get the emergency message.*/
        EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();

        
        if (!name.equals(emcyMsg.getDeviceName())) return;
        /** the emergency message is coming from this device.*/    

        FCSLOG.error(name + ":EmergencyMessage received from CanOpenProxy=" + emcyMsg.toString());
        //for the GUI
        this.inError = true;
        try {
            String errorInHexa = this.readErrorRegister();
            this.errorHistory = this.getErrorHistoryNames();


            String errorName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errorInHexa);
            if (errorName == null) {
                this.errorRegister = errorInHexa;
            } else {
                this.errorRegister = errorName;
            }
            this.errorHistory = this.getErrorHistoryNames();
            publishData();

        } catch (FcsHardwareException ex) {
            FCSLOG.error(name + " Error on FCS hardware.", ex);
            this.getSubsystem().raiseAlert(new Alert("FCS001","Couldn't read EPOS error register for " + name), 
                    AlertState.ALARM);
        }

        //Notify observers
        setChanged();
        this.notifyObservers(new ValueUpdate(name, emcyMsg));
    }

    /**
     * 
     * @return
     * @throws FcsHardwareException 
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read controller's parameter in the CPU for each mode and "
            + "warn if a parameter has not the same value as in the configuration."
            + "Then publish data.")
    @Override
    public String initializeAndCheckHardware() throws FcsHardwareException {
        if (!isBooted()) {
            throw new FcsHardwareException(name
                    + " is not booted - can't be initialized.");
        }
        this.mode = readMode();
        this.parametersOK = true;
        checkParameters(EposMode.CURRENT);
        checkParameters(EposMode.PROFILE_POSITION);
        checkParameters(EposMode.HOMING);
        checkFault();
        this.initialized = true;
        if (!parametersOK) {
            FCSLOG.warning(name + " Some parameter values are not"
                    + " the same in CPU and configuration system.");
        }
        //publishData for the GUI
        this.tcpProxy.publishHardwareData(this);
        this.publishData();
        FCSLOG.info(name + ": is INITIALIZED.");
        return name + ": is INITIALIZED.";
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the controller is enabled.")
    @Override
    public boolean isEnabled() throws FcsHardwareException {
        try {
            String statusWordInHexa = readStatusWord();
            int statusWord = Integer.parseInt(statusWordInHexa, 16);
            int[] statusWordInBinary = FcsUtils.toReverseBinary(statusWord);
            FCSLOG.debug(name + ":statusWordInBinary[0]=" + statusWordInBinary[0]);
            FCSLOG.debug(name + ":statusWordInBinary[1]=" + statusWordInBinary[1]);
            FCSLOG.debug(name + ":statusWordInBinary[2]=" + statusWordInBinary[2]);
            FCSLOG.debug(name + ":statusWordInBinary[3]=" + statusWordInBinary[3]);
            return (statusWordInBinary[0] == 1)
                    && (statusWordInBinary[1] == 1)
                    && (statusWordInBinary[2] == 1)
                    && (statusWordInBinary[3] == 0);

        } catch (ShortResponseToSDORequestException ex) {
            String msg = name + " couldn't read status word because ShortResponseToSDORequestException :" + ex.getMessage();
            FCSLOG.error(msg);
            FCSLOG.error(ex);
            throw new FcsHardwareException(msg,ex);

        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Clear faults on the controller.")
    public void faultReset() throws FcsHardwareException {
        writeControlWord("80");
        setChanged();
        this.notifyObservers(new ValueUpdate(name, "faultReset"));
        this.inError = false;
        this.errorRegister = "NO ERROR";
        this.errorHistory = new String[0];
        publishData();
    }

    /**
     * This methods enables the controller : i.e. this makes it able to receive
     * commands.
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "This methods enables the controller : i.e. this makes it able to receive commands.")
    @Override
    public String enable() throws FcsHardwareException {
        shutdown();
        switchOnEnableOperation();
        enabledToPublish = true;
        FCSLOG.debug(name + ": is ENABLED.");
        return name + " ENABLED";
    }

    /**
     * Shutdown the controller.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Shutdown the controller.")
    @Override
    public String shutdown() throws FcsHardwareException {
        writeControlWord("6");
        //TODO test the return code for the command writeSDO
        FCSLOG.debug(name + ": is SHUTDOWN.");
        return name + " DISABLED";
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "writeControlWord(\"F\")")
    @Override
    public void switchOnEnableOperation() throws FcsHardwareException {
        writeControlWord("F");
    }

    /**
     * Restore default parameters.
     *
     * @return
     * @throws SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Restore default parameters. AT YOUR OWN RISK.")
    public Object restoreParameters() throws FcsHardwareException {
        //TODO if is Power Disabled
        return this.writeSDO("1011", "1", "4", "64616F6C");
    }

    /**
     * Write a value in hexa in the control world (index=6040,
     * subindex=0,size=2)
     *
     * @param value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Write a value in hexa in the control world (index=6040, subindex=0,size=2)")
    @Override
    public void writeControlWord(String value) throws FcsHardwareException {
        this.writeSDO("6040", "0", "2", value);
    }

    /**
     * Read the control world (index=6040, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the control world (index=6040, subindex=0,size=2)")
    public String readControlWord() throws ShortResponseToSDORequestException,
            FcsHardwareException {
        return this.readSDO("6040", "0");
    }

    /**
     * Read the status world (index=6041, subindex=0,size=2)
     *
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the status world (index=6041, subindex=0,size=2)")
    public String readStatusWord() throws ShortResponseToSDORequestException, FcsHardwareException {
        return this.readSDO("6041", "0");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "QuickStop.")
    @Override
    public void quickStop() throws FcsHardwareException {

        FCSLOG.debug(name + " running QUICKSTOP command.");

        switch (this.mode) {
            case HOMING:
                writeControlWord("B");
                break;
            case PROFILE_POSITION:
                writeControlWord("B");
                break;
            case PROFILE_VELOCITY:
                break;
            case POSITION:
                break;
            case VELOCITY:
                writeControlWord("B");
                break;
            case CURRENT:
                writeControlWord("2");
                break;
            default:
                assert false;
        }
    }




    @Override
    public String off() throws BadCommandException, FcsHardwareException {
        switch (this.mode) {
            case HOMING:
                this.writeControlWord("11F");
                break;
            case PROFILE_POSITION:
                this.writeControlWord("10F");
                break;
            case CURRENT:
                stopCurrent();
                break;
            default:
                assert false: mode;
        }
        shutdown();
        this.turnedOff = true;
        FCSLOG.info(name + " is TURNED OFF");
        return name + " OFF";
    }


    /**
     * Read the values of the motor parameters and displays them for the end
     * user.
     *
     * @return
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING3,
            description = "Display motor parameters values stored in the EPOS CPU.")
    public String displayMotorParameters() throws FcsHardwareException {
        StringBuilder sb = new StringBuilder("Read decimal values for motor parameters are :");

        //motor type
        int readMotorType = readParameter(Parameter.MotorType);

        //position sensor type
        int readPositionSensorType = readParameter(Parameter.PositionSensorType);

        //motor data
        int readContinuousCurrentLimit = readParameter(Parameter.ContinuousCurrentLimit);
        int readMaxSpeedInCurrentMode = readParameter(Parameter.MaxSpeedInCurrentMode);
        int readThermalTimeConstantWinding = readParameter(Parameter.ThermalTimeConstantWinding);
        int readOutputCurrentLimit = readParameter(Parameter.OutputCurrentLimit);
        int readPolePairNumber = readParameter(Parameter.PolePairNumber);

        sb.append(" motor type =");
        sb.append(readMotorType);
        sb.append(" position sensor type =");
        sb.append(readPositionSensorType);
        sb.append(" continuousCurrentLimit =");
        sb.append(readContinuousCurrentLimit);
        sb.append(" maxSpeedInCurrentMode =");
        sb.append(readMaxSpeedInCurrentMode);
        sb.append(" thermalTimeConstantWinding =");
        sb.append(readThermalTimeConstantWinding);
        sb.append(" outputCurrentLimit =");
        sb.append(readOutputCurrentLimit);
        sb.append(" polePairNumber =");
        sb.append(readPolePairNumber);
        return sb.toString();
    }

    /*Methods available in CURRENT mode*/
    /**
     * In current mode this methods send a current to the motor. This make run
     * the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     *
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Set the current value in the EPOS CPU.")
    public void writeCurrent(int aValue) throws BadCommandException, FcsHardwareException {
        //TODO read controller CPU
        if (!mode.equals(EposMode.CURRENT)) {
            throw new BadCommandException(name + " is not in CURRENT mode.");
        }
        //TODO is it useful to test each time ? Response : YES because can be launched from the console.
        if (!isEnabled()) {
            throw new BadCommandException(name + " is not ENABLED.");
        }
        //check parameter aValue
        try {
            if (aValue > this.getMotorParameterValue("ContinuousCurrentLimit")) {
                throw new IllegalArgumentException(aValue + " is greater than Continuous Current Limit.");
            }
        } catch (EPOSConfigurationException ex) {
            FCSLOG.error(name + " ERROR in command writeCurrent:" + ex);
            throw new BadCommandException(name + ": ContinuousCurrentLimit is not defined." + ex);
        }
        String currentValueInHexa = Integer.toHexString(aValue);
        writeSDO("2030", "0", "2", currentValueInHexa);
        this.turnedOff = false;
        FCSLOG.debug(name + ": sent current to controller=" + aValue);
        //TODO check the return code of writeSDO
    }

    /**
     * In Current Mode this methods returns the current actualy received by the
     * motor.
     *
     * @return the current UNIT=mA / FORMAT=decimal
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentActualValue",
            description = "In Current Mode this methods returns the current actualy received by the motor.")
    public int readCurrent() throws BadCommandException, 
            ShortResponseToSDORequestException, FcsHardwareException {

        String currentInHexa = readSDO("6078", "0");
        int uns16 = Integer.parseInt(currentInHexa, 16);
        FCSLOG.debug(name + ":readCurrent="
                + CanOpenEPOS.convertToInteger16(uns16) + " Unsigned value read=" + uns16);
        actualCurrent = CanOpenEPOS.convertToInteger16(uns16);

        return actualCurrent;

    }

    /**
     * In Current Mode this methods returns the average of the current received
     * by the motor.
     *
     * @return the current UNIT=mA / FORMAT=decimal
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, alias = "readCurrentAverage",
            description = "In Current Mode this methods returns the average of the current  received by the motor.")
    public int readCurrentAverageValue() throws BadCommandException, FcsHardwareException {

        String currentInHexa = readSDO("2027", "0");
        int uns16 = Integer.parseInt(currentInHexa, 16);
        FCSLOG.debug(name + ":readAverageCurrent=" + CanOpenEPOS.convertToInteger16(uns16) + " Unsigned value read=" + uns16);
        return CanOpenEPOS.convertToInteger16(uns16);
    }

    /**
     * In current mode this methods set to zero the value of the current sent to
     * the motor. This stops the motor.
     *
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public void stopCurrent() throws BadCommandException, FcsHardwareException {
        if (mode.equals(EposMode.CURRENT)) {
            writeSDO("2030", "00", "2", "0");
        } else {
            throw new BadCommandException(getName() + "is not in CURRENT mode");
        }

    }




    /**
     * Reads in the CPU the value of the parameter PositionActualValue and
     * returns it in a decimal format.
     *
     * @return value in decimal format
     * @throws SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "readPositionActualValue",
            description = "In PROFILE_POSITION Mode this methods returns the actual position.")
    @Override
    public int readPosition() throws ShortResponseToSDORequestException,
            FcsHardwareException {
        int positionActualValue = this.readParameter(Parameter.PositionActualValue);
        FCSLOG.debug(name + ":readPosition=" + positionActualValue);

        return positionActualValue;
    }

    /**
     * Read the position returned by the absolute encoder (single serial data).
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read the position returned by the absolute encoder (single serial data).")
    @Override
    public int readSSIPosition() throws ShortResponseToSDORequestException,
            FcsHardwareException {
        return this.readParameter(Parameter.SSIEncoderActualPosition);
    }

    /**
     * In PROFILE_POSITION returns the value of the parameter ProfileVelocity.
     *
     * @return
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION returns the value of the parameter ProfileVelocity.")
    @Override
    public int readProfileVelocity() throws ShortResponseToSDORequestException, FcsHardwareException {
        profileVelocity = this.readParameter(Parameter.ProfileVelocity);
        FCSLOG.debug(name + ":readProfileVelocity=" + profileVelocity);
        return profileVelocity;
    }

    /**
     * Writes value 35 in hexa to set the Homing method as Actual (See EPOS
     * documentation)
     *
     * @return
     * @throws BadCommandException if not in HOMING mode.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public String setHomingMethodActual() throws BadCommandException, FcsHardwareException {
        if (!mode.equals(EposMode.HOMING)) {
            throw new BadCommandException(getName() + "is not in HOMING mode");
        }
        return this.writeParameter(Parameter.HomingMethod, 35);

    }

    /**
     * Starts homing : (See EPOS documentation) For engineering mode.
     *
     * @throws SDORequestException
     * @throws BadCommandException if the EPOS
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     */
    public void startHoming() throws BadCommandException,
            ErrorInCommandExecutionException, FcsHardwareException {
        if (!mode.equals(EposMode.HOMING)) {
            throw new BadCommandException(getName() + "is not in HOMING mode");
        }
        this.writeControlWord("1F");
    }

    /**
     * Set the Home Position with the value given as argument in decimal format.
     *
     * @param position in decimal format
     * @return
     * @throws SDORequestException
     * @throws BadCommandException
     */
    public String setHomePosition(int position) throws BadCommandException, FcsHardwareException {
        if (!mode.equals(EposMode.HOMING)) {
            throw new BadCommandException(getName() + "is not in HOMING mode");
        }

        return this.writeParameter(Parameter.HomePosition, position);

    }

    /**
     * Defines the actual position as the absolute position which value is given
     * as an argument.
     *
     * @throws SDORequestException
     * @throws ErrorInCommandExecutionException
     * @throws BadCommandException
     */
    @Override
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Define the actual position as position given as argument.")
    public void defineAbsolutePosition(int position) throws ErrorInCommandExecutionException, 
            BadCommandException, FcsHardwareException {
        FCSLOG.debug(name + " Defining Absolute Position:" + position);
        this.changeMode(EposMode.HOMING);
        this.writeParameters(EposMode.HOMING);
        if (!this.isEnabled()) {
            this.enable();
        }
        this.setHomePosition(position);
        this.setHomingMethodActual();
        this.writeControlWord("3F");
    }

    /**
     * In PROFILE_POSITION mode this methods set the target position. This make
     * run the motor.
     *
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be
     * sent.
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "In PROFILE_POSITION mode this methods set the target position."
            + " Target position has to be given in microns.")
    @Override
    public void writeTargetPosition(int aValue) throws BadCommandException,
            FcsHardwareException {
        if (!mode.equals(EposMode.PROFILE_POSITION)) {
            throw new BadCommandException(name + "is not in PROFILE_POSITION mode");
        }

        if (!isEnabled()) {
            throw new BadCommandException(name + "is not ENABLED");
        }

        String targetPositionInHexa = Integer.toHexString(aValue);
        writeSDO("607A", "0", "4", targetPositionInHexa);
        //TODO check the return code of writeSDO
    }

    /**
     * In HOMING mode and PROFILE_POSITION mode this indicates that the position
     * is reached.
     *
     * @return true if the position is reached.
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "In HOMING mode and PROFILE_POSITION mode this indicates that the position is reached.")
    @Override
    //TODO test this method on test benches
    public boolean checkTargetReached() throws FcsHardwareException {
        String statusWordInHexa = readStatusWord();
        int statusWord = Integer.parseInt(statusWordInHexa, 16);
        int[] statusWordInBinary = FcsUtils.toReverseBinary(statusWord);
        return (statusWordInBinary[10] == 1);
    }

    /**
     * Check if the Controller is in fault.
     *
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the Controller is in fault.")
    @Override
    public void checkFault() throws FcsHardwareException {
        if (!this.booted) {
            throw new FcsHardwareException(name + " is not booted - can't checkFault.");
        }
        int controllerNbOfErrors = readNumberOfErrors();
        if (controllerNbOfErrors > 0) {
            this.inError = true;
            String errorInHexa = this.readErrorRegister();
            String errorName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errorInHexa);
            if (errorName == null) {
                this.errorRegister = errorInHexa;
            } else {
                this.errorRegister = errorName;
            }
            this.errorHistory = this.getErrorHistoryNames();
            setChanged();
            this.notifyObservers(new ValueUpdate(name, "checkFault"));
            publishData();
            String errorHistoryString = displayErrorHistory();
            throw new FcsHardwareException(name
                    + " is in fault. " + errorHistoryString);
        } else {
            this.inError = false;
            this.errorRegister = "NO ERROR";
            this.errorHistory = new String[0];
            setChanged();
            this.notifyObservers(new ValueUpdate(name, "checkFault"));
            publishData();
        }

    }

    @Override
    public void checkParameters(EposMode aMode) throws FcsHardwareException  {
        Map<String, Integer> paramsMap = null;
        switch (aMode) {
            case HOMING:
                paramsMap = this.paramsForHoming;
                break;
                
            case PROFILE_POSITION:
                paramsMap = this.paramsForProfilePosition;
                break;

            case CURRENT:
                paramsMap = this.paramsForCurrent;
                break;
                
            default:
                assert false;
        }
        if (paramsMap == null) {
            throw new FcsHardwareException(name
                    + "parameters for mode :" + aMode.toString() + "are NULL in configuration system.");
        }
        if (paramsMap.isEmpty()) {
            throw new FcsHardwareException(name
                    + "parameters for mode :" + aMode.toString() + "are not defined in configuration system.");
        }
        checkParameters(paramsMap);
    }

    /**
     * For every parameter to be defined for this mode this method compares the
     * values in configuration with the value stored in controller CPU and
     * throws a HardwareException if these values are not equal. values stored
     * in the controller CPU.
     *
     * @param modeInString
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "For every parameter to be defined for this mode this method "
            + "compares the values in configuration with the value stored in controller CPU"
            + "and throws a HardwareException if these values are not equal.")
    @Override
    public void checkParameters(String modeInString) throws FcsHardwareException {
        EposMode aMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        checkParameters(aMode);
    }

    /**
     * Compare values of parameters in the map with the values of parameters in
     * the controller CPU.
     *
     * @param paramsMap
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     */
    public void checkParameters(Map<String, Integer> paramsMap) throws FcsHardwareException {
        boolean errorFound = false;
        StringBuilder sb = new StringBuilder(name);
        for (Map.Entry<String, Integer> entry : paramsMap.entrySet()) {
            String paramName = entry.getKey();
            int configValue = entry.getValue();
            int controllerValue = readParameter(paramName);

            if (configValue != controllerValue) {
                String msg = " ERROR in CPU controller for parameter="
                        + paramName + " value found in configuration=" + configValue
                        + " value read in CPU=" + controllerValue + "\n";
                FCSLOG.warning(name + msg);
                sb.append(msg);
                errorFound = true;
                //TODO test on testbench
                writeParameter(paramName,configValue);
            }

        }
        parametersOK = !errorFound;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to CURRENT mode and initialize the"
            + "CURRENT mode parameters from configuration values.")
    public void activateCurrentMode() throws FcsHardwareException {
        this.changeMode(EposMode.CURRENT);
        this.writeParameters(this.paramsForCurrent);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to HOMING mode and initialize the"
            + "HOMING mode parameters from configuration values.")
    public void activateHomingMode() throws FcsHardwareException {
        this.changeMode(EposMode.HOMING);
        this.writeParameters(this.paramsForHoming);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING3,
            description = "Change mode to PROFILE_POSITION mode and initialize the"
            + "PROFILE_POSITION mode parameters from configuration values.")
    public void activateProfilePositionMode() throws FcsHardwareException {
        this.changeMode(EposMode.PROFILE_POSITION);
        this.writeParameters(this.paramsForProfilePosition);
    }

    /**
     * A holding brake can be associated to a Controller. This brake is
     * activated when the controller is powered off. In CPPM test bench a Brake
     * is associated with the controllers of the trucks. This command activates
     * the brake to prevent the trucks motion.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Put the brake on to prevent the trucks motion.")
    @Override
    public void activateBrake() throws FcsHardwareException {
        String hexaValue = readParameterInHexa(Parameter.DigitalOutputFonctionnalityState);
        writeParameterInHexa(Parameter.DigitalOutputFonctionnalityState, FcsUtils.force2zero(hexaValue, 15));
        FCSLOG.debug(name + ": brake activated.");
    }

    /**
     * A holding brake can be associated to a Controller. In CPPM test bench a
     * Brake is associated with the controllers of the trucks. Release the brake
     * in order to be able to move the trucks.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Release the brake in order to be able to move the trucks.")
    @Override
    public void releaseBrake() throws FcsHardwareException {
        //If the controller is not enabled and we release brakes it could lead to a 
        //lost of a filter.
        if (!isEnabled()) {
            enable(); //added in sept 2015 and tested with trucks and onlineClamp
        }
        String hexaValue = readParameterInHexa(Parameter.DigitalOutputFonctionnalityState);
        writeParameterInHexa(Parameter.DigitalOutputFonctionnalityState, FcsUtils.force2one(hexaValue, 15));
        FCSLOG.debug(name + ": brake released.");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Activate the holding brake and shutdown the controller.")
    @Override
    public void disable() throws FcsHardwareException {
        activateBrake();
        shutdown();
        enabledToPublish = false;
    }

    public StatusDataPublishedByEPOSController getStatusData() {
        StatusDataPublishedByEPOSController status = FcsUtils.createStatusDataPublishedByEPOSController(this);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            alias = "refreshGUI",
            description = "Publish data for the controller on the status bus.")
    public void publishData() {
        StatusDataPublishedByEPOSController status = this.getStatusData();
        KeyValueData kvd = new KeyValueData(name, status);
        this.getSubsystem().publishSubsystemDataOnStatusBus(kvd);
    }

    /**
     * Change an int represented by an UNSIGNED16 to an INTEGER16.
     *
     * @param unsigned16
     * @return
     */
    public static int convertToInteger16(int unsigned16) {
        if (unsigned16 < 32768) {
            return unsigned16;
        } else {
            return unsigned16 - 65535;
        }
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        System.out.println(displayListParameters("HOMING"));
        System.out.println(displayListParameters("CURRENT"));
        System.out.println(displayListParameters("PROFILE_POSITION"));
        System.out.println(displayListParameters());

    }

}
