/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 *
 * This is the driver class for the ADC we use in the FCS test bench and in
 * SINGLE FILTER TEST CAN-CBX-AI814
 *
 * @author virieux
 */
public class CanOpenADC extends CanOpenDevice implements ADCInterface {

    /**
     * One bit read on the input of the ADC represents 0.3125 mV
     */
    public static final double adcVoltsPerBit = 0.3125;
    public static final double waterFreezeTemp = 273.15;

    /**
     * The transmission types for the analogue inputs are described in the
     * following: acyclic, synchronous: The transmission is initiated if a
     * SYNC-message has been received (PDO- transmission type 0) and data has
     * changed. cyclic, synchronous: The transmission is initiated if a defined
     * number of SYNC-messages have been received (PDO-transmission type
     * 1...240). synchronous, remote request: The state of the inputs is latched
     * with each SYNC-message and is transmitted after the reception of a
     * RTR-frame (PDO-transmission type 252). asynchronous, remote request:
     * After the reception of a RTR-frame the last latched state of the inputs
     * is transmitted (PDO-transmission type 253). event controlled,
     * asynchronous: The transmission is initiated if the state of selected
     * inputs has changed (PDO-transmission type 254, 255).
     */
    protected String transmissionType;

    public CanOpenADC(String aName, int aTickMillis, String nodeID, String serialNB, 
            String transmissionType) {
        super(aName, aTickMillis, nodeID, serialNB);
        this.transmissionType = transmissionType;
    }

    public String getTransmissionType() {
        return transmissionType;
    }

    public void setTransmissionType(String transmission_type) {
        this.transmissionType = transmission_type;
    }

    /**
     * Read the analog input whom number is given as a parameter
     *
     * @param inputNB
     * @return the value read in hexa (on 16bits)
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    public String readAnalogInput(int inputNB) throws SDORequestException,
            HardwareNotDetectedException, ShortResponseToSDORequestException, 
            FcsHardwareException {

        if (!bridge.isCWrapperConnected()) {
            throw new HardwareNotDetectedException(name + ": hardware not connected.");
        }

        if (isBooted()) {
            //FCSLOG.debug(name + " IS BOOTED");
            //int inputNB = Integer.parseInt(input, 16);
            if ((inputNB < 1) || (inputNB > 8)) {
                throw new IllegalArgumentException("CAN-CBX-AI814 : input has to be 1..8");
            }
            String input = Integer.toHexString(inputNB);
            return tcpProxy.readSDO(nodeID, "6401", input);

        } else {
            String message = String.format("Module %s / nodeID=%s is UNREACHABLE",
                    getName(), nodeID);
            FCSLOG.error(message);

            throw new HardwareNotDetectedException("CanOpenADC MISSING",
                    this.getName(), this.nodeID, this.serialNB);

        }
    }

    /**
     * Read the voltage at the ADC input inputNB on the ADC and returns the
     * value in Volt
     *
     * @param inputNB : integer from 1 to 8
     * @return the value read in hexa (on 16bits)
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException
     * @throws
     * org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    @Override
    public double readVoltage(int inputNB) throws SDORequestException,
            HardwareNotDetectedException, ShortResponseToSDORequestException, 
            FcsHardwareException {

        String hexa = readAnalogInput(inputNB);
        int valueInt = Integer.parseInt(hexa, 16); //voltage in V
        return valueInt * CanOpenADC.adcVoltsPerBit;
    }

    @Override
    public String initializeAndCheckHardware() throws FcsHardwareException {
        try {
            writeTransmissionTypeToDevice();
        } catch (SDORequestException | ErrorInCommandExecutionException | BadCommandException ex) {
        //changed for sonar    
            //throw new FcsHardwareException(name + ex.getMessage());
            throw new FcsHardwareException(name + ex);
        }
        this.initialized = true;
        return getName() + "is initialized with transmission type =" + this.transmissionType;
    }

    /**
     * This methods writes the transmission type given by the configuration to
     * the device.
     *
     * @return a message
     * @throws SDORequestError
     * @throws ErrorInCommandExecutionException
     */
    private String writeTransmissionTypeToDevice() throws SDORequestException, 
            ErrorInCommandExecutionException, BadCommandException, FcsHardwareException {
        if (this.transmissionType == null) {
            throw new BadCommandException(getName() 
                    + ": Can't write transmission type to device because transmaission type is null.");
        }
        String ack = this.tcpProxy.writeSDO(this.nodeID, "1801", "2", "1", this.transmissionType);
        String goodAck = "OK";
        if (!ack.equals(goodAck)) {
            throw new ErrorInCommandExecutionException(getName() + " :bad returned ack for wsdo, index 1801");
        }

        String ack2 = this.tcpProxy.writeSDO(this.nodeID, "1802", "2", "1", this.transmissionType);
        if (!ack2.equals(goodAck)) {
            throw new ErrorInCommandExecutionException(getName() + " :bad returned ack for wsdo, index 1802");
        }
        return String.format("Transmission type of device %s is now %s", this.name, this.transmissionType);
    }

}
