/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import java.nio.file.FileSystemNotFoundException;
import java.util.Map;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.*;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class gathers the description of the real hardware and the tools
 * (methods) to give access to this hardware. The concrete classes which extends
 * this abstract class will have to add the fields for the pieces of hardware
 * and to give the list of hardware in the method listHardware.
 * 
 *
 * @author virieux
 */
public abstract class BridgeToCanOpenHardware extends Module implements BridgeToHardware {



    protected CanOpenProxy tcpProxy;

    protected boolean configurationChecked;
    protected boolean hardwareInitialized;

    protected Map<String, PieceOfHardware> hardwareMap;

    public BridgeToCanOpenHardware(String aName, int aTickMillis, 
            CanOpenProxy tcpProxy) {
        super(aName, aTickMillis);
        this.tcpProxy = tcpProxy;
        this.configurationChecked = false;
        this.hardwareInitialized = false;
        
    }

    /**
     * Return a list of the hardware this bridge is connected to.
     *
     * @return
     */
    public PieceOfHardware[] listHardware() {
        Map<String, PieceOfHardware> children = this.getChildren(PieceOfHardware.class);

        PieceOfHardware[] list = new PieceOfHardware[children.size()];
        int i = 0;
        for (Map.Entry<String, PieceOfHardware> entry : children.entrySet()) {
            String key = entry.getKey();
            FCSLOG.debug("key=" + key);
            PieceOfHardware device = entry.getValue();
            FCSLOG.debug("hardware=" + device.getName() + "#" + device.toString());
            list[i] = device;
            i++;

        }
        return list;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "print my children; for tests")
    public void printChildren() {
        Map<String, PieceOfHardware> children = this.getChildren(PieceOfHardware.class);
        for (Map.Entry<String, PieceOfHardware> entry : children.entrySet()) {
            String key = entry.getKey();
            FCSLOG.debug("key=" + key);
            PieceOfHardware device = entry.getValue();
            FCSLOG.debug("hardware=" + device.getName() + "#" + device.toString());

        }
    }

    @Override
    public CanOpenProxy getTcpProxy() {
        return tcpProxy;
    }

    @Override
    public void startServer() {
        tcpProxy.startServer();
    }

    @Override
    public void startThreadReader() {
        tcpProxy.startThreadReader();
    }

    /**
     * This method returns true if the C-Wrapper is connected to the tcp proxy.
     * The C-Wrapper has to send the tcpProxy clientName as a keyword before any
     * communication can begin. So if the C-Wrapper doesn't send the good
     * keyword or if it doesn't connect to the tcp proxy, this method returns
     * false.
     *
     * @return
     */
    @Override
    public boolean isCWrapperConnected() {
        return tcpProxy.isReady(tcpProxy.getMyClientName());
    }

    /**
     * returns true if the hardware is booted, identified and initialized.
     *
     * @return
     */
    @Override
    public boolean isHardwareReady() {
        return tcpProxy.isHardwareReady();
    }

    /**
     * This methods has to be overridden in a simulation context.
     * @return 
     */
    @Override
    public boolean isRealHardware() {
        return true;
    }

    @Override
    public void initModule() {

        FCSLOG.info(name + ": init MODULE BRIDGE TO HARDWARE.");
        hardwareMap = this.getChildren(PieceOfHardware.class);
        PieceOfHardware[] hardwareList = listHardware();
        tcpProxy.setHardwareList(hardwareList);
        FCSLOG.info(name + ":NUMBER OF CAN OPEN DEVICES EXPECTED =" + hardwareList.length);
        FCSLOG.info(this.toString());

        //we have to read the errors table for the maxon motor and can open devices.
        try {
            CanOpenErrorsTable.loadDeviceErrorTable();
            CanOpenErrorsTable.loadErrorRegisterTable();
            CanOpenErrorsTable.loadCommunicationErrorTable();
        } catch (DeploymentException ex) {
            String msg = ex + ": Could not load CanOpen Error Tables";
            FCSLOG.error(msg);
            throw new FileSystemNotFoundException(msg);
        }
    }

    /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper.
     *
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * @throws java.lang.InterruptedException
     * @throws org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public Object sendCanOpen(String command) throws InterruptedException, 
            CanOpenCallTimeoutException, BadCommandException, FcsHardwareException {
        return tcpProxy.sendCanOpenCommand(command);
    }

    @Override
    public PDOStorage readPDOs() throws FcsHardwareException, BadCommandException {
        FCSLOG.debug(name + "/readPDOS");
        return this.tcpProxy.readPDOs();

    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append(": tickmillis=");
        sb.append(String.valueOf(tickMillis));
        sb.append(", tcpProxy=");
        sb.append(String.valueOf(tcpProxy.toString()));
        return sb.toString();
    }
}
