/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.Alert;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;
import org.lsst.ccs.subsystems.fcs.common.TcpProxyInterface;

/**
 * This is the Main Module for every software related to the Filters Exchanger :
 * - single-filter-test control-command - scale 1 prototype control-command -
 * real Filters Exchanger This class is designed to gather the commons methods
 * to all these softwares.
 *
 * @author virieux
 */
public abstract class MainModule extends Module {

    protected BridgeToHardware bridge;
    protected boolean hardwareReady;
    protected AtomicBoolean haltRequired;
    protected AtomicBoolean stopRequired;

    public MainModule(String aName, int aTickMillis, BridgeToHardware bridge) {
        super(aName, aTickMillis);
        this.hardwareReady = false;
        this.bridge = bridge;
    }

    public AtomicBoolean getHaltRequired() {
        return haltRequired;
    }

    public AtomicBoolean getStopRequired() {
        return stopRequired;
    }

    @Override
    public void initModule() {
        haltRequired = new AtomicBoolean(false);
        stopRequired = new AtomicBoolean(false);
        String msg = controlRealHardware() ? "Real Hardware" : "Simulation";
        FCSLOG.info(name + "==>" + msg);
    }

    /**
     * This happens when the hardware has been initialized. (after completeInitialization).
     * It starts the ticking in super.postStart().
     * @throws HardwareException 
     */
    @Override
    public void postStart() throws HardwareException {
        super.postStart();
        FCSLOG.fine(name + ":postStart begins.");
        try {
            this.updateStateWithSensors();
        } catch (FcsHardwareException | ErrorInCommandExecutionException | BadCommandException ex) {
            throw new HardwareException(true, ex);
        }
        FCSLOG.fine(name + ":postStart ends.");
    }

    /**
     * This methods updates the boolean hardwareReady from the hardware bridge.
     * But when the hardware is ready, it doesn't not ask hardware bridge again
 to avoid too much messages on the FCSLOG bus, could be replace by : return
 this.bridge.isHardwareReady();
     *
     * @return true if hardware is booted, identified and initialized
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Return true if hardware is booted, identified and initialized.")
    public boolean isHardwareReady() {
        if (hardwareReady) {
            return this.hardwareReady;
        } else {
            this.hardwareReady = this.bridge.isHardwareReady();
            return this.hardwareReady;
        }
    }

    @Command(type = Command.CommandType.QUERY, 
            description = "Return true if this main module controls real hardware.")
    public boolean controlRealHardware() {
        return this.bridge.isRealHardware();
    }

    /**
     * This method returns a String with the list of hardware that this bridge
     * manages.
     *
     * @return
     */
    @Command(level = Command.ENGINEERING1, description = "Return the list of CANopen hardware "
            + "that this subsystem manages.", type = Command.CommandType.QUERY)
    public String listHardware() {
        return this.bridge.getTcpProxy().listHardware();
    }
    
    @Command(level = Command.ENGINEERING1, description = "Return the list of CANopen hardware "
            + "that this subsystem manages.", type = Command.CommandType.QUERY)
    public List<String> listHardwareNames() {
        return this.bridge.getTcpProxy().listHardwareNames();
    }
    
    /**
     * Return a list of filter names that the subsystem manages.
     * For the GUI or the Console.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY,
            description = "Return the list of names of filters  "
            + "that this subsystem manages.",             
            level = Command.ENGINEERING1,
            alias="listFilterNames")
    public List<String> getFilterNames() {
        FilterManager filterManager = (FilterManager) getComponentByName("filterManager");
        if (filterManager == null) {
            return Collections.EMPTY_LIST;
        } else {
            return filterManager.getFilterNamesSortedByFilterID();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException,
            ErrorInCommandExecutionException, BadCommandException {
        getTcpProxy().publishData();
    }

    public TcpProxyInterface getTcpProxy() {
        return this.bridge.getTcpProxy();
    }

    @Override
    public TreeWalkerDiag signal(Signal signal) {
        switch (signal.getLevel()) {
            case HALT:
                FCSLOG.debug("HALT required");
                this.haltRequired.set(true);
                break;
            case STOP:
                FCSLOG.debug("STOP required");
                this.stopRequired.set(true);
                break;
            case INTERRUPT1:
                break;
            case INTERRUPT2:
                break;
            case RE_START:
                this.haltRequired.set(false);
                this.stopRequired.set(false);
                break;
            default:
                assert false;
        }
        //we want the signal to go to all my children.
        return TreeWalkerDiag.GO;
    }

    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "For tests purpose only - raise an ALERT.")
    public void simulateAlert(String errorId, String alarmMsg) {
        FCSLOG.error("A fake Alert has been raised.");
        String fcsErrorId;
        String message;
        if (alarmMsg == null) {
            message = "COUCOU c'est une blague !";
        } else {
            message = alarmMsg;
        }
        if (errorId == null) {
            fcsErrorId = "FCS002";
        } else {
            fcsErrorId = errorId;
        }
        Alert alert = new Alert(fcsErrorId,message);
        this.getSubsystem().raiseAlert(alert, ALARM);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "For tests purpose only - raise an ALERT.")
    public void simulateWarning(String errorId, String alarmMsg) {
        FCSLOG.error("A fake Alert has been raised.");
        String fcsErrorId;
        String message;
        if (alarmMsg == null) {
            message = "COUCOU c'est une blague !";
        } else {
            message = alarmMsg;
        }
        if (errorId == null) {
            fcsErrorId = "FCS002";
        } else {
            fcsErrorId = errorId;
        }
        Alert alert = new Alert(fcsErrorId,message);
        this.getSubsystem().raiseAlert(alert, WARNING);
    }

    @Override
    public String toString() {
        return super.toString();
    }

}
