/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This is the model for a single hook in the loader. Each one is checked in
 * locked and unlocked position by 2 sensors These sensors have to give a
 * consistent response otherwise the hook is in ERROR. There is 4 hooks in the
 * loader to hold a filter.
 *
 * @author virieux
 */
public class LoaderHookModule extends Module {

    private final NumericSensor lockSensor;
    private final NumericSensor unlockSensor;

    private FcsEnumerations.LockStatus lockStatus;

    //Used because we have to wait for the update from the sensors to know the state
    //of the hook
    protected final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private boolean locked;
    private boolean unlocked;
    private boolean inError;
    private boolean inTravel;

    /**
     * This is used when we update the latch state with the values returned by
     * the sensors.
     */
    protected volatile boolean updatingState = false;

    public LoaderHookModule(String name, NumericSensor lockSensor, NumericSensor unlockSensor) {
        super(name);
        this.lockSensor = lockSensor;
        this.unlockSensor = unlockSensor;
    }

    public NumericSensor getLockSensor() {
        return lockSensor;
    }

    public NumericSensor getUnlockSensor() {
        return unlockSensor;
    }

    public boolean isInError() {
        return inError;
    }

    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": getLockStatus was interrupted while waiting for update.");
                }

            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    @Override
    public void initModule() {
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.lockSensor.updateValue(hexaValues);
            this.unlockSensor.updateValue(hexaValues);

            locked = (this.lockSensor.getDigitalValue() == 1);
            unlocked = (this.unlockSensor.getDigitalValue() == 1);
            inTravel = (!locked && !unlocked);
            inError = (locked && unlocked);

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (locked) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = FcsEnumerations.LockStatus.UNLOCKED;
            } else if (inTravel) {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }

    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "display sensors info")
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("#locksensor=");
        sb.append(this.lockSensor.toString());
        sb.append("#unlocksensor=");
        sb.append(this.unlockSensor.toString());
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, description = "List and display hook sensors values.")
    public String listSensorsValues() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/locksensor value=");
        sb.append(this.lockSensor.getDigitalValue());
        sb.append("/unlocksensor value=");
        sb.append(this.unlockSensor.getDigitalValue());
        sb.append("/lockStatus=");
        sb.append(this.lockStatus.toString());
        sb.append("/");
        return sb.toString();
    }
    
    /**
     * To create an object StatusDataPublishedByLoaderHook for publication on the 
     * STATUS bus.
     * @return 
     */
    StatusDataPublishedByLoaderHook createStatusDataPublishedByLoaderHook() {
        StatusDataPublishedByLoaderHook status = new StatusDataPublishedByLoaderHook(name,
                lockSensor.getDigitalValue(),
                unlockSensor.getDigitalValue(),
                inError,
                lockStatus
        );
        return status;
    }

}
