/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.Arrays;

import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class's goal is to manage the filters.
 * @author virieux
 */
public class FilterManager extends Module {
    
    private final ArrayList<Filter> filters; 

    
    /**
     * A map to store the filters that FCS manages. 
     * The key of this map is the filter name.
     * 
     */
    protected TreeMap<String, Filter> filtersMap;

    /**
     * Creation of a Filter Manager from a list of Filters.
     * This is used by the framework to create a Filter Manager
     * from the list of filters given in description groovy file.
     * The number of filters is variable so to add a filter to this list,
     * the end user has just to add the filter description in the description
     * groovy file.
     * @param name
     * @param oneOrMoreFilters 
     */
    public FilterManager(String name, 
            Filter ...oneOrMoreFilters) {
        super(name);
        filters = new ArrayList<>();
        filters.addAll(Arrays.asList(oneOrMoreFilters));
    }
    
    @Override
    public void initModule() {
        /** 
        * Creation of the HashMap filtersMap.
        * To move this initialisation to the constructor is a bad idea. 
        * It has been tested and it doesn't work because in the construction phase
        * the environment of the objects Filter is not set, so the method getName() 
        * returns null.
        */
        filtersMap = new TreeMap<>();
        for (Filter f: filters) {
            FCSLOG.info("Add filter to filterManager:" + f.toString());
            filtersMap.put(f.getName(), f);
        }
        
    }
    
    public Filter getFilterByName(String filterName) {
        return filtersMap.get(filterName);
    }
    
    /**
     * Return a list of names of filters that this filter manager manages.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY,
        description = "Return the list of names of filters  "
        + "that this subsystem manages.",             
        level = Command.ENGINEERING1,
        alias="listFilterNames")
    public List<String> getFilterNames() {
        ArrayList<String> namesList = new ArrayList<>();
        namesList.addAll(filtersMap.keySet());
        return namesList;
    }
    
    /**
     * Returns a list of filter names. This list is sorted by filter ID.
     * @return 
     */
    public List<String> getFilterNamesSortedByFilterID() {
        ArrayList<String> namesList = new ArrayList<>();
        TreeMap<Integer,String> filtersByIDMap = new TreeMap<>();
        for (Map.Entry<String,Filter> entry: filtersMap.entrySet()) {
            Filter filter = entry.getValue();
            filtersByIDMap.put(filter.getFilterID(), filter.getName());
        }
        namesList.addAll(filtersByIDMap.values());
        return namesList;
    }
    
    
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "To test the filter manager - print the map filtersMap.")
    public String Test() {
        StringBuilder sb = new StringBuilder("TEST:\n");
        sb.append("FilterG=").append(filtersMap.get("filterG").toString()).append('\n'); 
        for (Map.Entry<String,Filter> entry: filtersMap.entrySet()) {
            sb.append("key=").append(entry.getKey());
            sb.append(",filter=").append(entry.getValue().toString()).append('\n');
        }
        return sb.toString();
    }

    @Command(alias = "printFilterList",
            type = Command.CommandType.QUERY, 
            level = Command.NORMAL,
            description = "Return a printed list of filters.")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("Filter List:\n");
        for (Filter f:filters ) {
            sb.append(f.toString());
            sb.append('\n');
        }
        return sb.toString();
    }


    

    
}
