package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.ConfigurableComponent;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterLocation;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This is a model for an optical filter for the camera.
 *
 */
public class Filter extends ConfigurableComponent {

    private Integer filterID;
    private String name;
    private double weight;
    private FilterLocation filterLocation = FilterLocation.UNKNOWN;
    private String socketName;

    public Filter(String name, int id, double weight) {
        this.name = name;  
        this.filterID = id;
        this.weight = weight;
        this.filterLocation = FilterLocation.UNKNOWN; 
    }

    public void setName(String name) {
        this.name = name;
    }
    
    

    /**
     *
     * @return true if the filter is held by the carousel.
     */
    public boolean isOnCarousel() {
        return this.filterLocation == FilterLocation.CAROUSEL;
    }

    /**
     *
     * @return true if the filter is held by the autochanger.
     */
    public boolean isOnAutoChanger() {
        return this.filterLocation == FilterLocation.AUTOCHANGER;
    }

    /**
     *
     * @return true if the filter is out the camera.
     */
    public boolean isOut() {
        return this.filterLocation == FilterLocation.OUT;
    }

    /**
     *
     * @return true if the location of the filter is unknown.
     */
    public boolean isUnknown() {
        return this.filterLocation == FilterLocation.UNKNOWN;
    }

    public FilterLocation getFilterLocation() {
        return filterLocation;
    }

    public void setFilterLocation(FilterLocation filterLocation) {
        this.filterLocation = filterLocation;
        this.publishData();
    }

    public Integer getFilterID() {
        return filterID;
    }

    public void setFilterID(Integer filterID) {
        this.filterID = filterID;
    }

    public double getWeight() {
        return weight;
    }

    public void setWeight(double weight) {
        this.weight = weight;
    }

    public String getSocketName() {
        return socketName;
    }

    public void setSocketName(String socketName) {
        this.socketName = socketName;
    }

    /**
     * This filter is now on carousel in socket given by its name.
     * It publishes data to update the GUI.
     * 
     * @param socketName 
     */
    public void putFilterOnSocket(String socketName) {
        synchronized (this) {
            this.filterLocation = FilterLocation.CAROUSEL;
            this.socketName = socketName;
        }
        publishData();
    }
    
    /**
     * This method is call by the CarouselSocket when the filter is not seen 
     * anymore by the filter presence sensor at STANDBY position.
     * When a filter is removed from CAROUSEL, the only place it can go it's the 
     * AUTOCHANGER.
     */
    public void removeFromCarousel() {
        synchronized (this) {
            this.filterLocation = FilterLocation.AUTOCHANGER;
            this.socketName = "NOT_ON_CAROUSEL";
        }
        FCSLOG.info(name + " removed from carousel");
        publishData();
    }
    
    /**
     * Change the location of the filter and publish on the bus.
     * @param filterLocation 
     */
    public void changeLocation(FilterLocation filterLocation) {
        this.filterLocation = filterLocation;
        publishData();
    }
    
    
    /**
     * Creates an object StatusDataPublishedByFilter and  publishes it on the 
     * Status Bus for trending data base and GUIs.
     *
     */
    public void publishData() {
        getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(name, 
                new StatusDataPublishedByFilter(filterLocation,socketName)));
    }

    @Command(alias = "printFilter",
            type = Command.CommandType.QUERY, 
            level = Command.NORMAL,
            description = "Return a printed filter.")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("=filterID:");
        sb.append(filterID);       
        sb.append(",Weight:");
        sb.append(weight);
        sb.append(",FilterLocation:");
        sb.append(filterLocation);
        if (filterLocation.equals(FilterLocation.CAROUSEL)) {
            sb.append(",socket:");
            sb.append(socketName);
        } 
        return sb.toString();
    }

}
