package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import static org.lsst.ccs.bus.states.AlertState.WARNING;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.ClearAlertHandler;
import static org.lsst.ccs.subsystems.fcs.CarouselModule.addAngle;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.AutochangerHandler;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.ClampsOrLatchesDisagreeException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This is a socket on the carousel : there is 5 sockets on a carousel. When a
 filter is on the carousel, it is attached at a socket. A socket has a clamp
 on each side : - clampX- (clampXminus) - clampX+ (clampXplus) clampX+ and
 clampX- are the same except for the method isLocked() See
 FilterClampXminusModule and FilterClampXplusModule

 The main job of this class CarouselSocket is to synchronize the actions on
 the 2 clamps. The state of the CarouselSocket is the state of the clamps when
 both clamps state are identical. To synchonize an action
 (updateSateWithSensor, unlock or release) on both clamps, we use a
 CyclicBarrier : we start 2 threads and they wait for each other action
 completion with a CyclicBarrier.

 During an action on both clamps, we want to lock the CarouselSocket so no
 other thread can try to access this object before the completion of action.
 We do that with a Lock and Condition. (see java.util.concurrent).

 CarouselSocket extends MobileItemModule because we want to lock or release
 the clamps when the carousel is at standby positionOnCarousel and we want to wait until
 the action is completed. We know if the action is completed in reading the
 clamps sensors. So we use the general mechanism provided by the
 MobileItemModule.
 *
 * this class raises ALERT FCS004 in method updateStateWithSensorsToCheckIfActionIsCompleted
 * 
 * @author virieux
 *
 */
public class CarouselSocket extends MobileItemModule implements ClearAlertHandler {

    private BridgeToHardware bridge;

    private CarouselClampModule clampXminus;

    private CarouselClampModule clampXplus;

    private EPOSController clampActuatorXminus;
    private EPOSController clampActuatorXplus;
    
    /*To be able to know if the autochanger holds a filter. */
    /*This can't be a AutochangerModule because when the carousel is in Standalone mode
    /*there is not autochanger.
    /*AutochangerHandler is an Interface. */
    private AutochangerHandler autochangerHandler;

    /**
     * socket positionOnCarousel (angle) in degrees on the carousel
     *
     */
    private final double positionOnCarousel;

    /**
     * carousel positionOnCarousel (angle) when this socket is at standby 
     * standbyPosition should be : 360 - positionOnCarousel, but could be different
     */
    private final double standbyPosition;

    /**
     * When the socket holds a filter, this field is initialized whith this
     * filter. When the socket is empty, this field is null.
     */
    private Filter filter;
    
    /**
     * The filter Manager knows the list of filters we can handle in the telescope.
     * 
     */
    private FilterManager filterManager;

    private volatile boolean updatingClamps = false;
    private volatile FilterClampState clampsState = FilterClampState.UNDEFINED;

    //When we update the 2 clamps, we want to take a lock on the socket until
    //the end of the update, so any other thread has to wait for this update.
    private final Condition stateUpdated = lock.newCondition();

    //TODO : there is also a timeout for each clamp, it's a lot of timeouts !
    //here we could take the maximum of the timeout of the clampXminus and clampXplus.
    //this is done in initModule 
    private long timeoutForUnlocking = 4000;
    private long timeoutForReleasing = 4000;

    public CarouselSocket(String moduleName, int aTickMillis,
            CarouselClampModule clampXminus, CarouselClampModule clampXplus,
            double position, double standbyPosition) {
        super(moduleName, aTickMillis);
        this.clampXminus = clampXminus;
        this.clampXplus = clampXplus;
        this.positionOnCarousel = position;
        this.standbyPosition = standbyPosition;

    }

    public CarouselClampModule getClampXminus() {
        return clampXminus;
    }

    public void setClampXminus(CarouselClampModule clampXminus) {
        this.clampXminus = clampXminus;
    }

    public CarouselClampModule getClampXplus() {
        return clampXplus;
    }

    public void setClampXplus(CarouselClampModule clampXplus) {
        this.clampXplus = clampXplus;
    }
    
    /**
     * Return the actual position of this socket.
     * @return 
     */
    public double getActualPosition() {
        return addAngle(positionOnCarousel,getCarouselPosition());
    }
    
    /**
     * Return the absolute position of the Carousel.
     * @return 
     */
    public double getCarouselPosition() {
        return ((CarouselModule)this.getComponentByName("carousel")).getPosition();
    }

    public double getPositionOnCarousel() {
        return positionOnCarousel;
    }

    public Filter getFilter() {
        return filter;
    }

    private void setFilter(Filter filter) {
        this.filter = filter;
    }

    public double getStandbyPosition() {
        return standbyPosition;
    }
    
    /**
     * Return true if this socket is at HALTED at STANDBY position.
     * @return 
     */
    public boolean isAtStandby() {
        return Double.doubleToRawLongBits(getCarouselPosition()) 
                == Double.doubleToRawLongBits(standbyPosition);
    }

    /**
     * ***********************************************************************************************
     */
    /**
     * ******************** END OF SETTERS AND GETTERS  **********************************************
     */
    /**
     * ***********************************************************************************************
     */
    @Override
    public void initModule() {
        super.initModule();
        this.bridge = (BridgeToHardware) getComponentByName("bridge");
        this.clampActuatorXminus = (EPOSController) getComponentByName("clampXminusController");
        this.clampActuatorXplus = (EPOSController) getComponentByName("clampXplusController");
        this.timeoutForReleasing = java.lang.Math.max(this.clampXminus.timeoutForReleasing, 
                this.clampXplus.timeoutForReleasing);
        this.timeoutForUnlocking = java.lang.Math.max(this.clampXminus.timeoutForUnlocking, 
                this.clampXplus.timeoutForUnlocking);
        
        if (getComponentByName("main") instanceof AutochangerHandler) {
            autochangerHandler = (AutochangerHandler)getComponentByName("main");
        } else {
            final String MSG = name + " ==>main doesn't implements AutochangerHandler -"
                    + " Please fix groovy description file or Main Module implementation.";
            FCSLOG.error(MSG);
            throw new IllegalArgumentException(MSG);
        }
        
        this.clampXminus.setController(clampActuatorXminus);
        this.clampXplus.setController(clampActuatorXplus);

    }
    
    /**
     * Checking if autochanger is holding the filter is delegated to autochangerHandler.
     * @return 
     */
    @Command(type = Command.CommandType.QUERY, 
            description = "Returns true if the autochanger holds the filter.")
    public boolean isAutochangerHoldingFilter() {
        return autochangerHandler.isAutochangerHoldingFilterAtSTANDBY();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1, 
            description = "Returns true if hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return ((MainModule) this.getComponentByName("main")).isHardwareReady();
    }

    /**
     * Returns the state of the clamps. If the state is being updated and
     * waiting for a response from a sensor, this methods waits until the state
     * is updated. If the state is not being updated, it returns immediatly the
     * state.
     *
     * @return state
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return clamps state - doesn't read again sensors.")
    public FilterClampState getClampsState() {

        lock.lock();
        try {
            while (updatingClamps) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(name + " was interrupted during getClampsState.");
                }

            }
            return clampsState;

        } finally {
            lock.unlock();
        }
    }

    /**
     * This method updates the state of the 2 clamps in reading the values sent
     * by the sensors. The values of the sensors are read clamp by clamp with
     * read SDO commands.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Deprecated
    public void updateClampsStateWithSensorsFromSDO() throws FcsHardwareException {

        lock.lock();
        this.updatingClamps = true;
        try {
            clampXminus.updateStateWithSensorsFromSDO();
            clampXplus.updateStateWithSensorsFromSDO();

            if (clampXminus.getClampState().equals(clampXplus.getClampState())) {
                this.clampsState = clampXminus.getClampState();
                //System.out.println("XXXXXXXXSocket at standby : clamps are updated");

            } else {
                throw new FcsHardwareException("Error in filter presence detection at standby position : "
                        + "the clamps don't agree.");
            }

        } finally {
            this.updatingClamps = false;
            this.stateUpdated.signal();
            lock.unlock();
        }
    }

    public void updateClampsStateWithSensors(PDOStorage pdoStorage) throws FcsHardwareException {
        lock.lock();
        this.updatingClamps = true;
        try {
            clampXminus.updateStateWithSensors(pdoStorage);
            clampXplus.updateStateWithSensors(pdoStorage);

            if (clampXminus.getClampState().equals(clampXplus.getClampState())) {
                this.clampsState = clampXminus.getClampState();
                FCSLOG.info(name + ":XXXXXXXXSocket at standby : clamps are updated");

            } else {
                String msg = name + ":Error in carousel at standby position : "
                        + "clampState at Xminus side is different from clampState at Xplus side.";
                msg = msg + " =>clampXminus state=" + clampXminus.getClampState();
                msg = msg + " =>clampXplus state=" + clampXplus.getClampState();
                FCSLOG.error(msg);
                Alert alert = new Alert("FCS004",msg);
                this.getSubsystem().raiseAlert(alert, ALARM);
                throw new ClampsOrLatchesDisagreeException(msg);
            }

        } finally {
            this.updatingClamps = false;
            this.stateUpdated.signal();
            lock.unlock();
        }
        publishData();
    }
    
    
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Update clamps state in reading sensors.")
    public void updateClampsStateWithSensors() throws FcsHardwareException, BadCommandException {

        FCSLOG.debug("updateClampsStateWithSensors/" + name + "/bridge=" + this.bridge.toString());
        updateClampsStateWithSensors(this.bridge.readPDOs());
    }

    /*
     * The carousel socket is empty if there is no filter in the socket.
     * That appends when clamps of both side of the socket are empty.
     * A clamp is empty when no filter is engaged in the clamp.
     * @throws HardwareError
     * If the 2 clamps are not in the same state this method throws
     * a HardwareError : this should not append except if 
     * the sensors are broken.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns true if there is no filter in the socket.")
    public boolean isEmpty() {
        return (clampsState.equals(FilterClampState.READYTOCLAMP))
                || (clampsState.equals(FilterClampState.UNCLAMPEDEMPTY));
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns true if there is a filter in the socket and the clamps"
                    + "are LOCKED.")
    public boolean isClampedOnFilter() {
        return clampsState.equals(FilterClampState.CLAMPEDONFILTER);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns true if there is a filter in the socket and the clamps"
                    + "are UNLOCKED.")
    public boolean isUnclampedOnFilter() {
        return clampsState.equals(FilterClampState.UNCLAMPEDONFILTER);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns true if there is NO filter in the socket and the clamps"
                    + "are UNLOCKED.")    
    public boolean isUnclampedEmpty() {
        return clampsState.equals(FilterClampState.UNCLAMPEDEMPTY);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Returns true if this socket is ready to clamp a filter.")
    public boolean isReadyToClamp() {
        return clampsState.equals(FilterClampState.READYTOCLAMP);
    }
    

    /**
     * Releases the 2 clamps of the socket.
     *
     * @return 
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Release the 2 clamps of this socket if the socket is at STANDBY position.")
    public String releaseClamps() throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        updateClampsStateWithSensors();

        FCSLOG.info("Checking conditions for release clamp " + name + " on socket "
                + "at standby position.");
        if (!this.isAtStandby()) throw new BadCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");

        if (!clampsState.equals(FilterClampState.UNCLAMPEDEMPTY)) {
            throw new BadCommandException("Can't release clamps if socket is "
                    + "not unclamped and empty.");
        }

        FCSLOG.info("Releasing clamp " + name + " on socket at standby position.");

        return this.executeAction(MobileItemAction.RELEASECLAMPS, timeoutForReleasing);
    }

    /**
     * This method unclamp the 2 clamps at standby positionOnCarousel. It turn ON the
     * actuators and wait for the end of the unlocking task. The unlock is
     * completed when the clamps sensors notice that the clamps are UNLOCKED. So
     * this method reads the sensors with the method
     * updateClampsStateWithSensors until the clamps are UNLOCKED or the
     * duration of the task is greater than the timeout (timeoutForUnlocking) To
     * read the sensors at a regular time schedule we use the
     * ScheduledThreadPoolExecutor (scheduler.scheduleAtFixedRate) from the
     * java.util.concurrent package.
     *
     * @return a message for the end user
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
        description = "Unlock the 2 clamps of this socket if the socket is at STANDBY position.")
    public String unlockClamps() throws FcsHardwareException, BadCommandException, 
            ErrorInCommandExecutionException {

        if (!this.isAtStandby()) throw new BadCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
        updateClampsStateWithSensors();

        if (!clampsState.equals(FilterClampState.CLAMPEDONFILTER)) {
            throw new BadCommandException(name + ":Can't unlock clamps if isn't clamped on filter.");
        }
        
        if (!this.isAutochangerHoldingFilter()) {
            throw new BadCommandException("CANNOT UNLOCK CLAMPS if FILTER "
                    + "is not HELD by autochanger.");
        }

        return this.executeAction(MobileItemAction.UNLOCKCLAMPS, timeoutForUnlocking);
    }

    /**
     * Software removing.
     * Removes a filter from carousel. This method has no action on hardware.
     * It updates the objects Socket and Filter.
     * A filter can be removed from Carousel only when the socketAtStandby 
     * is at standby filterPosition.
     *
     * @throws BadCommandException
     */
    public void removeFilter() throws BadCommandException {
        if (this.filter == null) {
            throw new BadCommandException("Carousel socket : there is no filter "
                    + "to remove at standby position");
        }
        //just for the GUI
        this.filter.removeFromCarousel();
        
        this.filter = null;
        this.publishData();
    }

    /**
     * This is only a software operation.
     * It updates the filter when a filter is seen by the sensors.
     * @param filter 
     */
    public synchronized void putFilterOnSocket(Filter filter) {
        setFilter(filter);
        this.publishData();
        filter.putFilterOnSocket(this.name);
    }
    
    @Override
    public void tick() {
        this.publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns a String representation of a CarouselSocket.")
    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append(",positionOnCarousel:");
        sb.append(positionOnCarousel);
        sb.append(",Filter in socket : ");
        if (filter == null) {
            sb.append(" NO FILTER").append("\n");
        } else {
            sb.append(filter.getName()).append("\n");
        }
        return sb.toString();
    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.UNLOCKCLAMPS)) {
            return this.clampsState.equals(FilterClampState.UNCLAMPEDONFILTER);
        } else if (action.equals(MobileItemAction.RELEASECLAMPS)) {
            return this.clampsState.equals(FilterClampState.READYTOCLAMP);
        } else {
            throw new IllegalArgumentException("Action on clamps on socket must "
                    + "be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        try {
            this.updateClampsStateWithSensors();
        } catch (ClampsOrLatchesDisagreeException ex) {
            FCSLOG.warning(ex);//for sonar
            String msg = name + " a little delai between the update of the 2 clamps : have to wait ...";
            FCSLOG.warning(msg);
            Alert alert = new Alert("FCS004",msg);
            this.getSubsystem().raiseAlert(alert, WARNING);
        }
    }

    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        if (action.equals(MobileItemAction.UNLOCKCLAMPS)) {
            if (!this.clampActuatorXminus.isEnabled()) {
                this.clampActuatorXminus.enable();
            }
            if (!this.clampActuatorXplus.isEnabled()) {
                this.clampActuatorXplus.enable();
            }
            if (!clampActuatorXminus.isEnabled()) {
                throw new BadCommandException(name
                        + " ControllerXminus has to be enabled first.");
            }
            if (!clampActuatorXplus.isEnabled()) {
                throw new BadCommandException(name
                        + " ControllerXplus has to be enabled first.");
            }
            this.clampActuatorXminus.writeCurrent(this.clampXminus.getCurrentToUnlock());
            this.clampActuatorXplus.writeCurrent(this.clampXplus.getCurrentToUnlock());
            
        } else if (action.equals(MobileItemAction.RELEASECLAMPS)) {
            this.clampActuatorXminus.off();
            this.clampActuatorXplus.off();
        } else {
            throw new IllegalArgumentException("Action on clamps on socket must be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void postAction(MobileItemAction action) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        this.publishData();
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) 
            throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        //TODO complete this
        FCSLOG.info(name + " stopAction : nothing to be done.");
    }

    @Override
    public void publishData() {
        getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(name, 
                createStatusDataPublishedByCarouselSocket()));
    }
    
    /**
     * Create an object to be published on the STATUS bus by a CarouselSocket.
     * @return 
     */
    public StatusDataPublishedByCarouselSocket createStatusDataPublishedByCarouselSocket() {
        StatusDataPublishedByCarouselSocket status = new StatusDataPublishedByCarouselSocket();
        status.setActualPosition(this.getActualPosition());
        status.setAtStandby(this.isAtStandby());
        status.setClampsState(clampsState);
        status.setEmpty(this.isEmpty());
        if (this.filter == null) {
            status.setFilterName(NO_FILTER);
        } else {
            status.setFilterName(this.filter.getName());
        }
        return status;
    }
    
    

    @Override
    public void quickStopAction(MobileItemAction action, long delay) 
            throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert) {
        switch (alert.getAlertId()) {
            
            case "FCS004":
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
                
            default:
                return ClearAlertHandler.ClearAlertCode.UNKWNOWN_ALERT;
        }
    }

}
