package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Map;
import java.util.Observable;
import java.util.TreeMap;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.RunningWay;
import org.lsst.ccs.subsystems.fcs.common.AutochangerHandler;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.common.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This is a representation of the hardware of the carousel. It receives
 * commands from the FCSMAinModule and send back an acknowledge. It publishes
 * data on the status bus. In engineering mode it can receive commands from the
 * engineering console. It has a latch, an engine which are connected to real or
 * simulated hardware.
 *
 * @author virieux
 *
 */
public class CarouselModule extends Module {

    private static final long serialVersionUID = -2376279469784152348L;

    private double position;
    private final double speed;
    private double current;

    /*This actuator opens the clamp Xminus when the carousel is halted at standby filterPosition.*/
    private EPOSController clampXminusController;

    /*This actuator opens the clamp Xplus when the carousel is halted at standby filterPosition.*/
    private EPOSController clampXplusController;

    /*Controls carousel rotation.*/
    private EPOSController carouselController;

    /*CANOpen devices to read the values of the clamps sensors.*/
    private PieceOfHardware clampsSensorsReaderDevice;

    /*CANOpen devices to read the values of the brakes sensors and temperatures.*/
    private PieceOfHardware cancbxAI420_1;
    private PieceOfHardware cancbxAI420_2;

    /*To be able to know if the autochanger holds a filter. */
    private AutochangerHandler autochangerHandler;

    private final ArrayList<CarouselSocket> sockets;
    /**
     * A map to store the sockets by their names. The key of this map is the
     * socket name.
     *
     */
    protected Map<String, CarouselSocket> socketsMap;

    private BridgeToHardware bridge;

    private volatile ModuleState state = ModuleState.HALTED;
    private boolean stopped = false;

    /**
     * A Constructor with a variable numbers of sockets.
     * In single-filter-test carousel has only 1 socket.
     * In Filter Exchanger final product and prototype, carousel has 5 sockets.
     * @param aName
     * @param aTickMillis
     * @param oneOrMoreSockets 
     */
    public CarouselModule(String aName, int aTickMillis,
            CarouselSocket... oneOrMoreSockets) {
        super(aName, aTickMillis);
        this.position = 0;
        this.current = 0;
        this.speed = 0;
        this.sockets = new ArrayList<>();
        (sockets).addAll(Arrays.asList(oneOrMoreSockets));
    }

    /**
     * Return a CarouselSocket which name is given as parameter.
     *
     * @param socketName
     * @return
     */
    public CarouselSocket getSocketByName(String socketName) {
        if (socketsMap.containsKey(socketName)) {
            return socketsMap.get(socketName);
        } else {
            throw new IllegalArgumentException(name + ": no such name for socket:" + socketName);
        }
    }

    /**
     * return carousel position.
     * @return 
     */
    public double getPosition() {
        return position;
    }

    public synchronized ModuleState getState() {
        return state;
    }

    public synchronized void setState(ModuleState state) {
        this.state = state;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXminusController() {
        return clampXminusController;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXplusController() {
        return clampXplusController;
    }

    public EPOSController getCarouselController() {
        return carouselController;
    }

    public PieceOfHardware getClampsSensorsReader() {
        return clampsSensorsReaderDevice;
    }

    public PieceOfHardware getCancbxAI420_1() {
        return cancbxAI420_1;
    }

    public PieceOfHardware getCancbxAI420_2() {
        return cancbxAI420_2;
    }

    /**
     * This method returns the clampX- which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClampModule getClampXminus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby filterPosition. It
     * can returns null if there is no socketAtStandby halted at standby
     * filterPosition.
     *
     * @return
     */
    public CarouselClampModule getClampXplus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXplus();
        }
    }

    /**
     * Used to publish on the STATUS bus for the GUI. Returns
     *
     * @return true if a socket is HALTED at STANDBY filterPosition, false
     * otherwise.
     */
    public synchronized boolean isAtStandby() {
        if (this.isRotating()) {
            return false;
        } else {
            for (CarouselSocket socket : sockets) {
                if (socket.isAtStandby()) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Return the socket HALTED at STANDBY filterPosition if there is one.
     * Otherwise return null.
     *
     * @return
     */
    public synchronized CarouselSocket getSocketAtStandby() {
        if (this.isRotating()) {
            return null;
        } else {
            for (CarouselSocket socket : sockets) {
                if (socket.isAtStandby()) {
                    return socket;
                }
            }
        }
        return null;
    }

    /**
     * This methods return the filter which is in the socketAtStandby at standby
     * filterPosition. When there is no filter at standby filterPosition it
     * returns null.
     *
     * @return filter at standby filterPosition or null if there is no filter at
     * standby filterPosition.
     */
    public synchronized Filter getFilterAtStandby() {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getFilter();
        }
    }

    /**
     * Return the filter which is in the socket at STANDBY filterPosition or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    public synchronized String getFilterAtStandbyName() {

        Filter filterAtStandby = getFilterAtStandby();
        if (filterAtStandby == null) {
            return NO_FILTER;

        } else {
            return filterAtStandby.getName();
        }

    }

    /**
     * This methods returns true if the autochanger is holding a filter at
     * STANDBY filterPosition.
     *
     * @return
     */
    public boolean isAutochangerHoldingFilter() {
        return autochangerHandler.isAutochangerHoldingFilterAtSTANDBY();
    }

    @Override
    public void initModule() {
        super.initModule();
        FCSLOG.info("[CarouselModule] Initializing the carousel module ");
        setState(ModuleState.HALTED);
        this.bridge = (BridgeToHardware) getComponentByName("bridge");
        this.clampXminusController = (EPOSController) getComponentByName("clampXminusController");
        this.clampXplusController = (EPOSController) getComponentByName("clampXplusController");
        this.carouselController = (EPOSController) getComponentByName("carouselController");
        this.clampsSensorsReaderDevice = (PieceOfHardware) getComponentByName("clampsSensorsReader");
        this.cancbxAI420_1 = (PieceOfHardware) getComponentByName("ai420_1");
        this.cancbxAI420_2 = (PieceOfHardware) getComponentByName("ai420_2");

        socketsMap = new TreeMap<>();
        for (CarouselSocket socket : sockets) {
            FCSLOG.info("Add socket to sockets Map:" + socket.toString());
            socketsMap.put(socket.getName(), socket);
        }

        if (getComponentByName("main") instanceof AutochangerHandler) {
            autochangerHandler = (AutochangerHandler) getComponentByName("main");
        } else {
            final String MSG = name + " ==>main doesn't implements AutochangerHandler -"
                    + " Please fix groovy description file or MainModule implementation.";
            FCSLOG.error(MSG);
            throw new IllegalArgumentException(MSG);
        }

    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    public void publishData() {
        getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("carousel",
                createStatusDataPublishedByCarousel()));
    }

    /**
     * Create an object StatusDataPublishedByCarousel to be published on the
     * STATUS bus.
     *
     * @return status
     */
    public StatusDataPublishedByCarousel createStatusDataPublishedByCarousel() {
        StatusDataPublishedByCarousel status = new StatusDataPublishedByCarousel();
        status.setName(name);
        status.setRotationInDegrees(position);
        status.setLocked(this.isLocked());
        status.setCurrent(current);
        status.setSpeed(speed);

        if (isAtStandby()) {
            status.setAtStandby(true);
            FCSLOG.debug(name + " Filter at STANDBY=" + this.getFilterAtStandbyName());
            status.setFilterAtStandbyName(this.getFilterAtStandbyName());
            status.setSocketAtStandbyName(this.getSocketAtStandby().getName());
        } else {
            status.setAtStandby(false);
        }
        /*List of filters on carousel*/
        Map<String, String> mapSocketFilter = new TreeMap<>();
        for (Map.Entry<String, CarouselSocket> entry : socketsMap.entrySet()) {
            CarouselSocket socket = entry.getValue();
            if (socket.getFilter() == null) {
                mapSocketFilter.put(socket.getName(), NO_FILTER);
            } else {
                mapSocketFilter.put(socket.getName(), socket.getFilter().getName());
            }
        }
        for (Map.Entry<String, String> entry : mapSocketFilter.entrySet()) {
            FCSLOG.debug(name + " socket name=" + entry.getKey() + " filter name=" + entry.getValue());
        }
        status.setFiltersOnCamera(mapSocketFilter);
        return status;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("\n Numbers of sockets: ");
        sb.append(sockets.size());
        for (CarouselSocket socket : sockets) {
            sb.append("\n Socket name ");
            sb.append(socket.getName());
            sb.append(" / ");
            sb.append(String.valueOf(socket));
        }
        return sb.toString();
    }


    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby filterPosition : - the carousel must not rotate - an empty
     * socketAtStandby is at standby filterPosition.
     *
     * @return true if the filterPosition of the carousel matches the
     * filterPosition when one of its sockets is at standby filterPosition and
     * this socketAtStandby is empty. false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Returns true if Carousel is stopped and no filter is at STANDBY position")
    public boolean isReadyToGrabAFilterAtStandby() throws FcsHardwareException {

        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }

    /**
     *
     * @return true if a filter is clamped at STANDBY filterPosition and
     * carousel is stopped
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Returns true if a filter is clamped at STANDBY position")
    public boolean isHoldingFilterAtStandby() throws FcsHardwareException {
        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        if (socketAtStandby.isEmpty()) {
            return false;
        }
        return socketAtStandby.isClampedOnFilter();

    }

    public boolean isRotating() {
        return false;
    }

    /**
     *
     * @return true if the brake is released
     */
    @Command(level = Command.ENGINEERING1, type = Command.CommandType.QUERY,
            description = "Returns true if the brake is released")
    public boolean isAbleToMove() {
        return true;
    }

    public boolean isLocked() {
        return true;
    }

    @Command(level = Command.ENGINEERING1, type = Command.CommandType.ACTION,
            description = "Engage the carousel brake to stop the rotation")
    public String engageBrake() {
        return name + " locked";

    }

    @Command(level = Command.ENGINEERING1, type = Command.CommandType.ACTION,
            description = "Release the carousel brake")
    public String releaseBrake() {
        return name + " unlocked";
    }

    /**
     * Updates the filterPosition of the carousel in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update carousel position in reading controller.")
    public void updatePosition() throws FcsHardwareException {
        try {
            this.position = carouselController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:" + ex);
        }
        this.publishData();
    }

    /**
     * Updates the field current in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update field current in reading controller.")
    public void updateCurrent() throws BadCommandException, FcsHardwareException {
        try {
            this.current = carouselController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:" + ex);
        }
        this.publishData();
    }

    /**
     * Read the clamp sensors one by one with RSDO command
     *
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromSDO() throws BadCommandException, FcsHardwareException {
        if (this.getSocketAtStandby() == null) {
            throw new BadCommandException(name
                    + " can't update Clamps State because there is no socket at standby.");
        }
        this.getSocketAtStandby().updateClampsStateWithSensors();
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time. Tested with success on single-filter-test in April 2013
     *
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO() throws BadCommandException,
            FcsHardwareException {
        if (this.getSocketAtStandby() == null) {
            throw new BadCommandException(name
                    + " can't update Clamps State because there is no socket at STANDBY.");
        }
        PDOStorage pdoStore = this.bridge.readPDOs();
        FCSLOG.finest(name + ":pdoStore=" + pdoStore.toString());
        this.getSocketAtStandby().updateClampsStateWithSensors(pdoStore);
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one
     * time. Tested with success on single-filter-test in April 2013
     *
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Read clamps sensors and update clamps state")
    public void updateClampsStateWithSensors() throws BadCommandException,
            FcsHardwareException {
        updateClampsStateWithSensorsFromPDO();
    }

    /**
     * A quick and dirty simulated rotation to test the carousel GUI.
     *
     * @param angle
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel to an absolute angle.")
    public String rotateToAbsoluteAngle(@Argument(name = "angle",
            description = "Angle to reach in degrees") double angle) {
        /*Quick and dirty simulation carousel rotation.*/
        this.position = angle; //just for tests
        this.publishData();
        return name + "rotation OK";
    }

    /**
     * A quick and dirty simulated rotation to test the carousel GUI.
     *
     * @param angle
     * @return
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel for an angle.")
    public String rotateToRelativeAngle(@Argument(name = "angle",
            description = "Angle to reach in degrees") double angle) {
        /*Quick and dirty simulation carousel rotation.*/
        this.position = addAngle(position, angle);
        this.publishData();
        return name + "rotation OK";
    }

    /**
     * Rotates carousel for an angle given as a parameter. This method can be
     * launched by the user from the TestConsole. It checks the value of the
     * rotation angle entered by the user, converts it in a double, and send the
     * result of the method rotate(double angle).
     *
     * @param angle : rotation angle The rotation angle can be positive or
     * negative. It must be a double and have a value between -360.0 and 360.0
     * @return the result of the command rotate(double angle)
     * @throws IllegalArgumentException, BadCommandException
     * @throws org.lsst.ccs.messaging.BadCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel for an angle")
    public String rotate(@Argument(name = "angle",
            description = "Angle to reach in degrees") double angle)
            throws BadCommandException {

        //availables values for an angle : [-360,+360]
        if (angle < -360 || angle > 360) {
            throw new IllegalArgumentException("Please enter an angle between -360 and +360");
        }

        RunningWay way;
        double absAngle;
        if (angle >= 0) {
            absAngle = angle;
            way = RunningWay.POSITIVE;
        } else {
            absAngle = -angle;
            way = RunningWay.NEGATIVE;
        }
        double requiredPosition = addAngle(getPosition(), angle);
        return rotate(absAngle, way, requiredPosition);
    }

    /**
     * Rotate the carousel to a final filterPosition.
     *
     * @param angle of the rotation : 0 <= angle <= 360
     * @param the final filterPosition has to be : 0 <= finalPosition <= 360
     * @param runningWay rotation way (enum)
     * @param finalPosition is the filterPosition the carousel must have when
     * the command is completed The method rotate is delegated to the engine,
     * which is an interface. An engine can be a simulated engine or a real
     * hardware driver.
     * @throws BadCommandException, IllegalArgumentException
     *
     */
    @Deprecated
    private String rotate(double angle, RunningWay runningWay, double finalPosition)
            throws BadCommandException {
        //assertion 0 <= angle <= 360
        //assertion 0 <= finalPosition <= 360
        if (angle < 0 || angle > 360) {
            throw new IllegalArgumentException(name
                    + " command rotate accepts angle only between 0 and 360)");
        }
        if (finalPosition < 0 || finalPosition > 360) {
            throw new IllegalArgumentException(name
                    + " command rotate accepts finalPosition only between 0 and 360)");
        }

        String message;

        if (isRotating()) {
            message = "Carousel is rotating, stop it before sending a new rotate command.";
            FCSLOG.error(message);
            throw new BadCommandException(message);
        }

        if (!this.isAbleToMove()) {
            message = "Carousel is unable to move. Check lock or flip rail.";
            FCSLOG.error(message);
            throw new BadCommandException(message);
        }

        stopped = false;
        setState(ModuleState.RUNNING);
        //Là, il faut faire tourner le moteur!
        message = "Rotating to required position: " + finalPosition;
        FCSLOG.info(message);

        while (isRotating()) {
            try {
                FCSLOG.info("...carousel rotating, please wait.....");
                FCSLOG.info(name + " WAITING TO BE AT REQUIRED POSITION=" + finalPosition);
                FCSLOG.info(state.toString());
                Thread.sleep(tickMillis);
            } catch (InterruptedException e) {
                return "Sleeping interrupted";
            }
        }
        if (!isRotating()) {
            FCSLOG.info("THREAD=" + Thread.currentThread().getName());
            setState(ModuleState.HALTED);
            FCSLOG.info("carousel is now stopped.");
            //FCSLOG.info("filter in standby filterPosition: " + getFilterAtStandbyName());
            if (Double.doubleToRawLongBits(getPosition())
                    == Double.doubleToRawLongBits(finalPosition)) {
                message = "Command completed";
                FCSLOG.info(message);
            } else {
                message = "Command non completed: carousel is not stopped at the required position";
                FCSLOG.info(message);
            }
        }

        if (stopped) {
            message = "Command rotate carousel interrupted by command stop";
            FCSLOG.info(message);
        }

        if (!stopped && (Double.doubleToRawLongBits(getPosition())
                != Double.doubleToRawLongBits(finalPosition))) {
            message = "Command non completed: carousel is not stopped at the required position";
            FCSLOG.error(message);
        }

        return message;

    }

    /**
     * What has to be done for each tick of the timer. We have to publish on the
     * status bus to refresh the GUI.
     */
    @Override
    public void tick() {
        this.publishData();
    }

    /**
     * This method has no action on hardware. This method updates the
     * socketAtStandby at standby filterPosition when a filter is put on the
     * carousel at standby filterPosition. A filter must be detected in the
     * socketAtStandby at standby filterPosition (socketAtStandby not empty).
     *
     * @param filter
     * @throws ErrorInCommandExecutionException
     * @throws BadCommandException
     */
    private synchronized void putFilterOnCarousel(Filter filter)
            throws BadCommandException {

        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(name + ": there is no socket at standby "
                    + "position to put a filter in.");
        }

        socketAtStandby.putFilterOnSocket(filter);
        this.publishData();
    }

    //TODO exception if filter is not on Carousel ?
    public double getFilterPosition(Filter filter) {
        double filterPosition = 0;
        for (CarouselSocket socket : sockets) {
            if (socket.getFilter().equals(filter)) {
                filterPosition = socket.getPositionOnCarousel();
            }
        }
        return filterPosition;
    }

    //TODO throw an exception if filter is not on carousel.
    public double getStandbyPositionForFilter(Filter filter) {
        double filterPosition = 0;
        for (CarouselSocket socket : sockets) {
            if (socket.getFilter().equals(filter)) {
                filterPosition = socket.getStandbyPosition();
                return filterPosition;
            }

        }
        return filterPosition;
    }

    /**
     * This method rotates the carousel to put the given filter in front of the
     * autochanger (standby filterPosition). This method can't be launched by
     * the user from the console. It computes the shortest way for the carousel.
     * So the running way can be positive or negative. This could be more
     * complicated if the carousel was not able to rotate more than TBD turns in
     * one way.
     *
     * @param filter
     * @return
     * @params filter : the filter can not be null and must be on the carousel
     * @throws BadCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Rotate carousel to put filter at standby position")
    protected String moveFilterToStandby(Filter filter) throws BadCommandException {

        //assert filter is on carousel
        if (!filter.isOnCarousel()) {
            throw new IllegalArgumentException("filter: " + filter.getName()
                    + " is not on carousel.");
        }
        String message = " [CarouselModule]";
        double requiredPosition = getStandbyPositionForFilter(filter);

        if (state == ModuleState.RUNNING) {
            throw new BadCommandException("Carousel is running, stop it before sending a new command.");
        }

        if (Double.doubleToRawLongBits(getPosition())
                == Double.doubleToRawLongBits(requiredPosition)) {
            FCSLOG.info(filter.getName() + " is at STANDBY position on carousel.");
            return filter.getName() + " is at STANDBY position on carousel.";
        }

        double angle = addAngle(getPosition(), getFilterPosition(filter));

        //
        RunningWay runningWay;
        if (angle < 180) {
            runningWay = RunningWay.NEGATIVE;
        } else {
            angle = 360 - angle;
            runningWay = RunningWay.POSITIVE;
        }
        //Release The clamps contact before the rotation
        releaseClampsContact();
        //rotate the carousel
        CarouselModule.this.rotate(angle, runningWay, requiredPosition);
        //engage the clamps contact when it's stopped at standby filterPosition
        engageClampsContact();
        //lock the carousel to hold it in filterPosition
        return message += engageBrake();
    }

    public String grabFilterAtStandby(Object filterName) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        FcsMainModule fcsMainModule = (FcsMainModule) this.getComponentByName("main");
        fcsMainModule.controlFilterName(filterName);

        Filter filterToGrabbe = fcsMainModule.getFilterByName((String) filterName);

        // the entered filter name is correct
        FCSLOG.info("Filter to move : " + filterName);
        FCSLOG.info("Filter location : " + filterToGrabbe.getFilterLocation());
        return name + grabFilterAtStandby(filterToGrabbe);
    }

    /**
     * This method has to be executed when a filter has just been moved on the
     * carousel at standby filterPosition by the autochanger. It updates the
     * carousel socketAtStandby at standby or throw an exception.
     *
     * This method tests if the clamps are Locked on a filter at standby
     * filterPosition The clamps Locks automaticaly, and the filter presence
     * sensor tells us if the filter is here or not. If the clamps are Locked on
     * a filter at standby filterPosition, this method updates the
     * socketAtStandby at standby filterPosition : it puts the filter on the
     * socketAtStandby.
     *
     * @param filter
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public synchronized String grabFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        if (filter == null) {
            throw new IllegalArgumentException(name + ": grabbeFilterAtStandby must receive a non null argument as a filter");
        }
        if (this.isRotating()) {
            throw new BadCommandException(name + ": grabbeFilterAtStandby can't grabbe a filter while rotating");
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(name + ": grabbeFilterAtStandby can't be executed because there is no socket at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();

        if ((socketAtStandby.isEmpty())) {
            throw new BadCommandException(name + ": grabbeFilterAtStandby can't be executed because no filter is detected in socket at standby position.");
        }

        if ((socketAtStandby.isClampedOnFilter())) {
            FCSLOG.info(name + ": grabbing " + filter.getName() + " at standby position.");
            putFilterOnCarousel(filter);
            FCSLOG.info(name + ": " + filter.getName() + " is grabbed on carousel.");
            String ack = name + ": " + filter.getName() + " is grabbed at standby position";
            return ack;
        } else {
            throw new BadCommandException(name + ": grabbeFilterAtStandby can't be executed because the clamps are not CLAMPED ON FILTER.");
        }

    }

    public void ungrabFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        if (filter == null) {
            throw new IllegalArgumentException("Filter to ungrabb should not be null");
        }
//                    if (!autochangerLatchesAreLocked())
//                        throw new BadCommandException("Carousel can't ungrabbe filter if it's not held by autochanger");
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(name + ":A socket has to be halted at standby position for this operation.");
        }

        if (!isOnStandby(filter)) {
            throw new BadCommandException(name + ": filter has to be at standby position to be ejected");
        }

        if (!socketAtStandby.isClampedOnFilter()) {
            throw new BadCommandException(name + ": can't ungrabb filter which is already unclamped.");
        }

        FCSLOG.info(name + ": ungrabbing " + filter.getName() + " at standby position.");

        //TODO test if the filter is held by the autochanger trucks.
        // The clamps can be opened.
        this.unlockClamps(filter);
        FCSLOG.info(name + ": socket at standby state : " + socketAtStandby.toString());

        FCSLOG.info(name + ": " + filter.getName() + " is ungrabbed from carousel.");
    }

    @Command(level = Command.ENGINEERING1, description = "Release clamps at standby position to get ready to clamp again", type = Command.CommandType.ACTION)
    public String releaseClamps() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            throw new BadCommandException(name
                    + ":Can't release clamps while a socket is not halted at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();
        if (!socketAtStandby.isEmpty()) {
            throw new BadCommandException(name
                    + ":Can't release clamps while "
                    + "a filter is in the socket at standby.");
        }

        if (socketAtStandby.isUnclampedEmpty()) {
            socketAtStandby.releaseClamps();
        } else {
            throw new BadCommandException(name + ":Can't release clamps when both clamps are not in state UNCLAMPEDEMPTY.");
        }

        socketAtStandby.updateClampsStateWithSensors();

        if (socketAtStandby.isReadyToClamp()) {
            return "Clamps are released and ready to clamp again.";
        } else {
            throw new ErrorInCommandExecutionException("Could not release clamps.");
        }
    }

    /**
     * This methods ReleaseBrakes the 2 clamps at STANDBY filterPosition.
     *
     * @param filter
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public String unlockClamps(Filter filter)
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            throw new BadCommandException("Can't unlock clamps while a socket is not halted at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();

        if (!this.isAutochangerHoldingFilter()) {
            throw new BadCommandException("Can't unlock clamps if the filter is not held by the autochanger.");
        }

        if (socketAtStandby.isClampedOnFilter()) {
            FCSLOG.info("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();

        } else {
            throw new BadCommandException("Can't unlock clamps when a filter is not clamped at standby.");
        }

        socketAtStandby.updateClampsStateWithSensors();

        //After the execution of the unlockClampsWithThreadSleep method in case of success.
        if (socketAtStandby.isUnclampedOnFilter()) {
            FCSLOG.info("Just about to remove filter from carousel");
            socketAtStandby.removeFilter();
            this.publishData();
            FCSLOG.info("Command unlockClamps completed");
            return "Clamps unlocked";

        } else {
            FCSLOG.info(state);
            throw new ErrorInCommandExecutionException(name + ":Could not unlock filter at standby");

        }
    }

    //for ENGINEERING MODE
    @Command(level = Command.ENGINEERING3, description = "Unlock the clamps",
            type = Command.CommandType.ACTION)
    public String unlockClamps() throws BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        return unlockClamps(this.getFilterAtStandby());
    }

    public boolean isOnStandby(Filter filter) {
        return Double.doubleToRawLongBits(this.getPosition())
                == Double.doubleToRawLongBits(this.getStandbyPositionForFilter(filter));
    }

    /**
     * Add 2 angles : an angle is a double which value is between -360 and +360
     *
     * @param angle1
     * @param angle2
     * @return result of the angles addition. Always > 0
     */
    public static double addAngle(double angle1, double angle2) {
        //if ( abs(angle1) > 360 || abs(angle2) > 360) throw new AssertionError("Adding ");

        double angle = angle1 + angle2;
        if (angle >= 360) {
            angle -= 360;
        }
        if (angle < 0) {
            angle += 360;
        }
        return angle;
    }

    public void releaseClampsContact() {
        //TODO;
    }

    public void engageClampsContact() {
        //TODO ;
    }
}
