/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This class represents a model for the 2 Latches of the autochanger. The
 * Latches hold a filter on the trucks. There is one latch on each side of the
 * filter. Because the ACTION open (or close) for 2 latches is different from
 * opening (or closing) one Latch, then the other one, it's more convenient to
 * have a class to represent 2 Latches at a time.
 *
 * @author virieux
 */
public class AutochangerTwoLatches extends MobileItemModule {

    private AutoChangerModule autochanger;
    private final AutochangerLatchModule latchXminus;
    private final AutochangerLatchModule latchXplus;

    private volatile LockStatus lockStatus;
    private volatile FilterPresenceStatus filterPresenceStatus;
    private long timeoutForClosing;
    private long timeoutForOpening;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latches state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    public AutochangerTwoLatches(String moduleName, int aTickMillis,
            AutochangerLatchModule latchXminus,
            AutochangerLatchModule latchXplus) {
        super(moduleName, aTickMillis);
        this.latchXminus = latchXminus;
        this.latchXplus = latchXplus;
        this.timeoutForClosing = Math.max(latchXminus.getTimeoutForClosing(),
                latchXplus.getTimeoutForClosing());
        this.timeoutForOpening = Math.max(latchXminus.getTimeoutForOpening(),
                latchXplus.getTimeoutForOpening());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if the 2 latches are initialized.")
    boolean isInitialized() {
        return latchXminus.isInitialized() && latchXplus.isInitialized();
    }

    boolean isInError() {
        return this.lockStatus == LockStatus.ERROR;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns lockStatus.")
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns filterPresenceStatus.")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public EPOSController getLatchXminusController() {
        return latchXminus.getLatchController();
    }

    public EPOSController getLatchXplusController() {
        return latchXplus.getLatchController();
    }

    //for GUI
    public AutochangerLatchModule getLatchXminus() {
        return latchXminus;
    }

    //for GUI
    public AutochangerLatchModule getLatchXplus() {
        return latchXplus;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if lockStatus=LOCKED.")
    public boolean isLocked() {
        return lockStatus.equals(LockStatus.LOCKED);
    }

    @Override
    public void initModule() {
        super.initModule();
        this.autochanger = (AutoChangerModule) this.getComponentByName("autochanger");

    }

    @Override
    public boolean isHardwareReady() {
        return autochanger.isHardwareReady() && isInitialized();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        latchXminus.checkHardware();
        latchXplus.checkHardware();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPENLATCHES:
                return latchXminus.isUnlocked() && latchXplus.isUnlocked();

            case CLOSELATCHES:
                return latchXminus.isLocked() && latchXplus.isLocked();
                
            default:
                assert false: action;
        }
        return false;
    }

    /**
     * Update latches lockStatus from an array of hexa values read from the
     * sensors.
     *
     * @param readHexaValues
     */
    void updateStateWithSensors(String[] readHexaValues) {
        lock.lock();

        try {
            updatingState = true;
            this.latchXminus.updateStateWithSensors(readHexaValues);
            this.latchXplus.updateStateWithSensors(readHexaValues);

            boolean inError = (this.latchXminus.getLockStatus().equals(LockStatus.ERROR))
                    || (this.latchXplus.getLockStatus().equals(LockStatus.ERROR));

            boolean locked = (this.latchXminus.getLockStatus().equals(LockStatus.LOCKED))
                    && (this.latchXplus.getLockStatus().equals(LockStatus.LOCKED));

            boolean unlocked = (this.latchXminus.getLockStatus().equals(LockStatus.UNLOCKED))
                    && (this.latchXplus.getLockStatus().equals(LockStatus.UNLOCKED));

            boolean inTravel = (this.latchXminus.getLockStatus().equals(LockStatus.INTRAVEL))
                    && (this.latchXplus.getLockStatus().equals(LockStatus.INTRAVEL));

            if (inError) {
                this.lockStatus = LockStatus.ERROR;
            } else if (locked) {
                this.lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                this.lockStatus = LockStatus.UNLOCKED;
            } else if (inTravel) {
                this.lockStatus = LockStatus.INTRAVEL;
            } else {
                this.lockStatus = LockStatus.UNKNOWN;
            }

            //filter presence
            if (this.latchXminus.isEmpty() && this.latchXplus.isEmpty()) {
                filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            } else if (!this.latchXminus.isEmpty() && !this.latchXplus.isEmpty()) {
                filterPresenceStatus = FilterPresenceStatus.ENGAGED;
            } else {
                filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
            }

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }

    }

    /**
     * This methods updates the whole autochanger state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update autochanger state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        autochanger.updateStateWithSensors();
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        updateStateWithSensors();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close latches.")
    public String close() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        if (lockStatus.equals(LockStatus.LOCKED)) {
            throw new BadCommandException(name + " are already LOCKED.");
        }
        autochanger.checkPreConditionsForClosingLatches();
        return this.executeAction(FcsEnumerations.MobileItemAction.CLOSELATCHES,
                timeoutForClosing);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open latches.")
    public String open() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        if (latchXminus.isUnlocked() && latchXplus.isUnlocked()) {
            throw new BadCommandException(name + " are already UNLOCKED.");
        }
        autochanger.checkPreConditionsForOpeningLatches();
        return this.executeAction(FcsEnumerations.MobileItemAction.OPENLATCHES,
                timeoutForOpening);
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) 
            throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        switch (action) {
            case OPENLATCHES:
                latchXminus.getLatchController().enable();
                latchXplus.getLatchController().enable();
                latchXminus.getLatchController().writeCurrent(latchXminus.getCurrentToOpen());
                latchXplus.getLatchController().writeCurrent(latchXplus.getCurrentToOpen());
                break;

            case CLOSELATCHES:
                latchXminus.getLatchController().enable();
                latchXplus.getLatchController().enable();
                latchXminus.getLatchController().writeCurrent(-latchXminus.getCurrentToOpen());
                latchXplus.getLatchController().writeCurrent(-latchXplus.getCurrentToOpen());
                break;
                
            default:
                assert false: action;
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        FCSLOG.debug(name + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.latchXminus.getLatchController().off();
        this.latchXplus.getLatchController().off();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    public StatusDataPublishedByAutochangerTwoLatches getStatusData() {
        return createStatusDataPublishedByTwoLatches();
    }
    
    public StatusDataPublishedByAutochangerTwoLatches createStatusDataPublishedByTwoLatches() {
        StatusDataPublishedByAutochangerTwoLatches status = new StatusDataPublishedByAutochangerTwoLatches();
        status.setName(name);
        status.setLockStatus(lockStatus);
        status.setFilterPresenceStatus(filterPresenceStatus);
        return status;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("autochangerLatches", getStatusData()));
    }

}
