/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
//import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 * This class is a model for the latch which lock a filter on the autochanger
 * trucks. The ACTION methods : open and close of this class is used in
 * ENGINEERING mode only when an operator may need to open or close only one
 * Latch. In NORMAL mode, when we have to open or close both Latches in the same
 * time, the class AutochangerTwoLatches is used. On the autochanger there are 2
 * latches : one on each side of the filter : at X+ and X-.
 *
 * @author virieux
 */
public class AutochangerLatchModule extends MobileItemModule {

    private AutoChangerModule autochanger;
    private final EPOSController latchController;
    private final NumericSensor filterPresenceSensor;
    private final NumericSensor lockSensor;
    private final NumericSensor unlockSensor;

    private volatile boolean filterEngaged;
    private boolean locked;
    private boolean unlocked;
    private boolean inError;
    private boolean inTravel;

    private int minCurrent;// for GUI
    private int maxCurrent;// for GUI
    private int currentToOpen;
    private int readCurrent;

    private LockStatus lockStatus;

    private volatile boolean initialized;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    private final long timeoutForOpening;
    private final long timeoutForClosing;

    /**
     * FOR THE GUI*
     */
    /**
     * true if the controller is in fault - false if the user does a faultReset*
     */
    private boolean controllerInFault;

    public AutochangerLatchModule(String moduleName,
            int aTickMillis,
            EPOSController latchController,
            long timeoutForClosing,
            long timeoutForOpening,
            NumericSensor lockSensor,
            NumericSensor unlockSensor,
            NumericSensor filterPresenceSensor,
            int minCurrent,
            int maxCurrent,
            int currentToOpen) {
        super(moduleName, aTickMillis);
        this.lockStatus = LockStatus.UNKNOWN;
        this.filterEngaged = false;
        this.latchController = latchController;
        this.timeoutForClosing = timeoutForClosing;
        this.timeoutForOpening = timeoutForOpening;
        this.lockSensor = lockSensor;
        this.unlockSensor = unlockSensor;
        this.filterPresenceSensor = filterPresenceSensor;
        this.minCurrent = minCurrent;
        this.maxCurrent = maxCurrent;
        this.currentToOpen = currentToOpen;
    }


    public NumericSensor getFilterPresenceSensor() {
        return filterPresenceSensor;
    }


    public NumericSensor getLockSensor() {
        return lockSensor;
    }


    public NumericSensor getUnlockSensor() {
        return unlockSensor;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns this latch controller name.")
    public String getControllerName() {
        return latchController.getName();
    }

    //for the GUI
    public EPOSController getLatchController() {
        return latchController;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if latchController is in Fault.")
    public boolean isControllerInFault() {
        return controllerInFault;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if latch is initialized.")
    public boolean isInitialized() {
        return initialized;
    }

    @ConfigChanger
    public void setMinCurrent(int minCurrent) {
        this.minCurrent = minCurrent;
    }

    @ConfigChanger
    public void setMaxCurrent(int maxCurrent) {
        this.maxCurrent = maxCurrent;
    }

    @ConfigChanger
    public void setCurrentToOpen(int currentToOpen) {
        this.currentToOpen = currentToOpen;
    }

    public int getMinCurrent() {
        return minCurrent;
    }

    public int getMaxCurrent() {
        return maxCurrent;
    }

    public int getCurrentToOpen() {
        return currentToOpen;
    }

    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }

    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    public FcsEnumerations.LockStatus getLockStatus() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.warning(name + ": interrupted in getLockStatus.");
                }
            }
            return lockStatus;

        } finally {
            lock.unlock();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=LOCKED")
    public boolean isLocked() {
        return this.getLockStatus().equals(LockStatus.LOCKED);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=UNLOCKED")
    public boolean isUnlocked() {
        return this.getLockStatus().equals(LockStatus.UNLOCKED);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if LockStatus=ERROR, this means that "
            + "unlockSensor and lockSensor return non consistant values.")
    public boolean isInError() {
        return this.getLockStatus().equals(LockStatus.ERROR);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger is empty.")
    public boolean isEmpty() {
        return !filterEngaged;
    }

    @Override
    public void initModule() {
        super.initModule();
        this.autochanger = (AutoChangerModule) this.getComponentByName("autochanger");

        if (latchController == null) {
            FCSLOG.error(name + "==>>> latchController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(name + "==>>> null latchController - fix groovy description file.");
        } else {
            //listens to latchController to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            if (latchController instanceof Observable) {
                this.listens((Observable) latchController);
            }
        }
    }

    @Override
    public void tick() {
        this.publishData();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        try {
            latchController.initializeAndCheckHardware();
        } catch (ShortResponseToSDORequestException ex) {
            //changed for sonar in sept 2015
            FCSLOG.warning(name + ":" + ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        //TODO check if there is a homing to be done.
        this.initialized = true;
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * This methods updates the whole autochanger state in reading all the
     * sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update latch state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        autochanger.updateStateWithSensors();
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the
     * method.
     *
     * @param hexaValues
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;
            this.filterPresenceSensor.updateValue(hexaValues);
            this.lockSensor.updateValue(hexaValues);
            this.unlockSensor.updateValue(hexaValues);

            filterEngaged = this.filterPresenceSensor.getDigitalValue() == 1;

            locked = this.lockSensor.getDigitalValue() == 1;
            unlocked = this.unlockSensor.getDigitalValue() == 1;
            inTravel = !locked && !unlocked;
            inError = locked && unlocked;

            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (locked) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = FcsEnumerations.LockStatus.UNLOCKED;
            } else if (inTravel) {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;
            }

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }

    }

    /**
     * Updates the field readCurrent of the latch in reading the CPU of the
     * controller. This method is used in the simulator mainly.
     *
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update latch current in reading controller.")
    public void updateCurrent() throws BadCommandException, SDORequestException, 
            FcsHardwareException {
        try {
            this.readCurrent = this.latchController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            //changed for sonar in sept 2015
            //FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:" + ex.getMessage());
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:" + ex);
        }
        this.publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Returns true if autochanger hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return autochanger.isHardwareReady();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Close latch.")
    public String close() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        if (locked) {
            throw new BadCommandException(name + " is already LOCKED.");
        }
        autochanger.checkPreConditionsForClosingLatches();
        return this.executeAction(FcsEnumerations.MobileItemAction.CLOSE,
                timeoutForClosing);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Open latch.")
    public String open() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        if (unlocked) {
            throw new BadCommandException(name + " is already UNLOCKED.");
        }
        autochanger.checkPreConditionsForOpeningLatches();
        return this.executeAction(FcsEnumerations.MobileItemAction.OPEN,
                timeoutForOpening);
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case OPEN:
                return lockStatus.equals(LockStatus.UNLOCKED);

            case CLOSE:
                return lockStatus.equals(LockStatus.LOCKED);
             
            default:
                assert false : action;
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        updateStateWithSensors();
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        switch (action) {
            case OPEN:
                latchController.enable();
                latchController.writeCurrent(this.currentToOpen);
                break;

            case CLOSE:
                latchController.enable();
                latchController.writeCurrent(-this.currentToOpen);
                break;
                
            default:
                assert false : action;
        }
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        FCSLOG.debug(name + " is ABORTING action " + action.toString()
                + " within delay " + delay);
        this.latchController.off();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        //TODO : is there something different to do ? TBD
        abortAction(action, delay);
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        //because we don't want to let the controller on power
        this.latchController.off();
        this.publishData();
        FCSLOG.info(name + ":" + action.toString() + " completed - doing postAction.");
    }

    /**
     * What to do when the Modules we observe send there new values. This class
     * observes its controller to publish data when its controller is in fault
     * or after a faultReset. Needed to update the GUI.
     *
     * @param source
     * @param v
     */
    //TODO this method has almost the same body for all the hardware which are moved by an EPOSController
    //so it should be refactorized to be written only one time for example in a default method of the interface EPOSController.
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.debug(name + ":processUpdate from source=" + source.toString()
                + " ValueUpdate=" + v.getName());
        //TODO to be tested on CPPM testbench in January 2016
        //replaced CanOpenEPOS by EPOSController in January 2016 to breal dependies between packages.
        if (!(source instanceof EPOSController)) {
            return;
        }
        if (v.getValue() instanceof EmergencyMessage) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            FCSLOG.debug(name + ":EmergencyMessage received from CanOpenProxy="
                    + emcyMsg.toString());

            if (latchController.getName().equals(emcyMsg.getDeviceName())) {
                String errCode = emcyMsg.getDeviceErrorCode();
                switch (errCode) {
                    case ("00"):
                        FCSLOG.debug(name + ":faultReset ?=" + emcyMsg.toString());
                        controllerInFault = false;
                        //controllerErrorMessage = emcyMsg.getDeviceErrorName();                       
                        break;

                    default:
                        FCSLOG.debug(name + ":EmergencyMessage received for "
                                + "my controller from CanOpenProxy=" + emcyMsg.toString());
                        controllerInFault = true;
                    //controllerErrorMessage = emcyMsg.getDeviceErrorName();
                    }
                this.publishData();
            }

        } else if (v.getValue() instanceof String) {
        //TODO to be tested on CPPM testbench in January 2016
        //replaced CanOpenEPOS by EPOSController in January 2016 to breal dependies between packages.
            EPOSController ctrl = (EPOSController) source;
            if (ctrl.getName().equals(latchController.getName())) {
                String msgFromController = (String) v.getValue();
                if (msgFromController.equals("faultReset")) {
                    this.controllerInFault = false;
                    //this.controllerErrorMessage = null;
                    this.publishData();
                }
            }
        }
    }

    public StatusDataPublishedByAutochangerLatch getStatusData() {
        StatusDataPublishedByAutochangerLatch status = createStatusDataPublishedByLatch();
        return status;
    }
    
    /**
     * Creates an object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutochangerLatch createStatusDataPublishedByLatch() {
        StatusDataPublishedByAutochangerLatch status = new StatusDataPublishedByAutochangerLatch();
        status.setName(name);
        status.setLockSensorValue(lockSensor.getDigitalValue());
        status.setUnlockSensorValue(unlockSensor.getDigitalValue());
        status.setFilterPresenceSensorValue(filterPresenceSensor.getDigitalValue());
        status.setLockStatus(lockStatus);
        status.setControllerInFault(controllerInFault);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Publish Data on the Status Bus.")
    @Override
    public void publishData() {
        StatusDataPublishedByAutochangerLatch status = this.getStatusData();
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(name, status));
    }

}
