/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.common.EmergencyMessage;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.*;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
//import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;

/**
 *
 * @author virieux
 */
public class AutoChangerTrucksModule extends MobileItemModule {

    protected NumericSensor handoffPositionSensor0;
    protected NumericSensor handoffPositionSensor1;
    protected NumericSensor onlinePositionSensor0;
    protected NumericSensor onlinePositionSensor1;
    protected NumericSensor standbyPositionSensor0;
    protected NumericSensor standbyPositionSensor1;

    private final int encoderRibbonMinValue;
    private final int encoderRibbonMaxValue;
    private final int minActualPositionValue;
    private final int maxActualPositionValue;

    private int position;
    private final int standbyPosition;
    private final int handoffPosition;
    private final int onlinePosition;

    private int current;
    private int speed;

    private int absoluteTargetPosition;
    private int relativeTargetPosition;
    private final long timeoutForTrucksMotion;

    private EPOSController linearRailMasterController;
    private EPOSController linearRailSlaveController;
    private AutoChangerModule autochanger;

    private final Condition stateUpdated = lock.newCondition();

    /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    /**
     * true if the master controller is in fault - false if the user does a faultReset*
     */
    private boolean masterControllerInFault;

    /**
     * true if the slave controller is in fault - false if the user does a faultReset*
     */
    private boolean slaveControllerInFault;

    /**
     * true if the 2 position sensors value at Handoff is 1 *
     */
    private boolean atHandoff;

    /**
     * true if the 2 position sensors value at Online is 1 *
     */
    private boolean atOnline;

    /**
     * true if the 2 position sensors value at Standby is 1 *
     */
    private boolean atStandby;

    /**
     * true if the 2 position sensors value at Standby differ *
     */
    private boolean standbySensorsInError;

    /**
     * true if the 2 position sensors value at Handoff differ *
     */
    private boolean handoffSensorsInError;

    /**
     * true if the 2 position sensors value at Handoff differ *
     */
    private boolean onlineSensorsInError;

    /**
     * true if the homing of the trucks have been done *
     */
    private boolean homingDone;

    public AutoChangerTrucksModule(String moduleName, int aTickMillis,
            int encoderRibbonMinValue, int encoderRibbonMaxValue,
            int minActualValue, int maxActualValue,
            int standbyPosition,
            int handoffPosition,
            int onlinePosition,
            long timeoutForTrucksMotion,
            NumericSensor handoffPositionSensor0,
            NumericSensor handoffPositionSensor1,
            NumericSensor onlinePositionSensor0,
            NumericSensor onlinePositionSensor1,
            NumericSensor standbyPositionSensor0,
            NumericSensor standbyPositionSensor1
    ) {
        super(moduleName, aTickMillis);
        this.encoderRibbonMinValue = encoderRibbonMinValue;
        this.encoderRibbonMaxValue = encoderRibbonMaxValue;
        this.minActualPositionValue = minActualValue;
        this.maxActualPositionValue = maxActualValue;
        this.standbyPosition = standbyPosition;
        this.handoffPosition = handoffPosition;
        this.onlinePosition = onlinePosition;
        this.timeoutForTrucksMotion = timeoutForTrucksMotion;
        this.standbyPositionSensor0 = standbyPositionSensor0;
        this.handoffPositionSensor0 = handoffPositionSensor0;
        this.onlinePositionSensor0 = onlinePositionSensor0;
        this.standbyPositionSensor1 = standbyPositionSensor1;
        this.handoffPositionSensor1 = handoffPositionSensor1;
        this.onlinePositionSensor1 = onlinePositionSensor1;
        this.absoluteTargetPosition = 0;
        this.relativeTargetPosition = 0;
        this.masterControllerInFault = false;
        this.slaveControllerInFault = false;
        this.standbySensorsInError = false;
        this.onlineSensorsInError = false;
        this.handoffSensorsInError = false;
        this.atHandoff = false;
        this.atOnline = false;
        this.atStandby = false;
        this.homingDone = false;
        this.current = 0;
        this.speed = 0;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return actual trucks position. Doesn't read again the position on the controller.")
    public int getPosition() {
        return position;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return HANDOFF position.")
    public int getHandoffPosition() {
        return handoffPosition;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return STANDBY position.")
    public int getStandbyPosition() {
        return standbyPosition;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return ONLINE position.")
    public int getOnlinePosition() {
        return onlinePosition;
    }

    //for the simulator
    public int getEncoderRibbonMinValue() {
        return encoderRibbonMinValue;
    }

    //for the simulator
    public int getEncoderRibbonMaxValue() {
        return encoderRibbonMaxValue;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if the master controller is in fault.")
    public boolean isMasterControllerInFault() {
        return masterControllerInFault;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return true if the slave controller is in fault.")
    public boolean isSlaveControllerInFault() {
        return slaveControllerInFault;
    }

    public EPOSController getLinearRailMasterController() {
        return linearRailMasterController;
    }

    public EPOSController getLinearRailSlaveController() {
        return linearRailSlaveController;
    }

    public NumericSensor getStandbyPositionSensor0() {
        return standbyPositionSensor0;
    }

    public NumericSensor getHandoffPositionSensor0() {
        return handoffPositionSensor0;
    }

    public NumericSensor getOnlinePositionSensor0() {
        return onlinePositionSensor0;
    }

    public NumericSensor getStandbyPositionSensor1() {
        return standbyPositionSensor1;
    }

    public NumericSensor getOnlinePositionSensor1() {
        return onlinePositionSensor1;
    }

    public NumericSensor getHandoffPositionSensor1() {
        return handoffPositionSensor1;
    }

    public int getCurrent() {
        return current;
    }

    public int getSpeed() {
        return speed;
    }



    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Standby are equal.")
    public boolean isStandbySensorsInError() {
        return standbySensorsInError;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Handoff are equal.")
    public boolean isHandoffSensorsInError() {
        return handoffSensorsInError;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant position sensors at Online are equal.")
    public boolean isOnlineSensorsInError() {
        return onlineSensorsInError;
    }

    public long getTimeoutForTrucksMotion() {
        return timeoutForTrucksMotion;
    }

    /**
     * Returns the boolean field atHandoff. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atHandoff is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at handoff position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtHandoffPosition() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return atHandoff;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the boolean field atOnline. If the atOnline boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atOnline is not being updated, it
     * returns its immediatly.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at ONLINE position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtOnlinePosition() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return atOnline;

        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the boolean field atStandby. If the atHandoff boolean is being
     * updated and waits for a response from a sensor, this methods waits until
     * update is completed. If the field atStandby is not being updated, it
     * returns its immediatly..
     *
     * @return atStandby
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the carrier is at STANDBY position. "
            + "This command doesn't read again the sensors.")
    public boolean isAtStandbyPosition() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return atStandby;

        } finally {
            lock.unlock();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the homing of the trucks has been done.")
    public boolean isHomingDone() {
        return homingDone;
    }

    @Override
    public void initModule() {
        super.initModule();
        autochanger = (AutoChangerModule) this.getComponentByName("autochanger");
        linearRailMasterController = (EPOSController) this.getComponentByName("linearRailMasterController");
        linearRailSlaveController = (EPOSController) this.getComponentByName("linearRailSlaveController");
        //listens to its controllers to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        if (linearRailMasterController instanceof Observable) {
            this.listens((Observable) linearRailMasterController);
        }
        if (linearRailSlaveController instanceof Observable) {
            this.listens((Observable) linearRailSlaveController);
        }
    }

    @Override
    public void tick() {
        this.publishData();
    }

    /**
     * This method is called by method checkHardware in AutoChangerModule during
     * INITIALIZATION phase.
     *
     * @return
     * @throws HardwareException
     */
    @Override
    //tested on CPPM testbench in september 2015
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();

        FCSLOG.debug(name + " checking hardware.");
        try {
            linearRailSlaveController.initializeAndCheckHardware();
            linearRailMasterController.initializeAndCheckHardware();
            configureControllers();
            homing();
            linearRailSlaveController.enable();
            linearRailMasterController.enable();

        } catch (SDORequestException ex) {
            //changed to satisfy sonar in sept2015
            FCSLOG.error(name + ":" + ex);
        } catch (ErrorInCommandExecutionException | BadCommandException ex) {
            throw new HardwareException(true, ex);
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + ":" + ex);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false, ex);
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Configure Autochanger trucks master and slave controllers.",
            alias = "initControllers")
    //tested on CPPM testbench in september 2015
    public void configureControllers() throws SDORequestException, FcsHardwareException {
        FCSLOG.info(name + " Begin configuration of the controllers");
        this.linearRailMasterController.shutdown();
        this.linearRailSlaveController.shutdown();
        configureDigitalInputOfLinearRails(linearRailMasterController);
        configureDigitalInputOfLinearRails(linearRailSlaveController);
        configureDigitalOutputOfLinearRails(linearRailMasterController);
        configureDigitalOutputOfLinearRails(linearRailSlaveController);
        FCSLOG.info(name + " End configuration of the controllers");
        //TODO tester si la config est bonne.

    }

    public boolean isInitialized() {
        return linearRailMasterController.isInitialized()
                && linearRailSlaveController.isInitialized()
                && homingDone;
    }

    /**
     * Move Autochanger trucks to the Handoff position.
     *
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Handoff position.",
            alias = "goToHandoff")
    //tested on CPPM testbench in september 2015
    public void goToHandOff() throws BadCommandException,
            ErrorInCommandExecutionException,
            FcsHardwareException {
        if (this.atHandoff) {
            throw new BadCommandException(name + " already at Handoff Position.");
        }
        moveToAbsoluteTargetPosition(this.handoffPosition);
    }

    /**
     * Move Autochanger trucks to the Online position.
     *
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Online position.")
    //tested on CPPM testbench in september 2015
    public void goToOnline() throws BadCommandException,
            ErrorInCommandExecutionException,
            FcsHardwareException {
        if (this.atOnline) {
            throw new BadCommandException(name + " is already at Online Position.");
        }
        moveToAbsoluteTargetPosition(this.onlinePosition);
    }

    /**
     * Move Autochanger trucks to the Standby position.
     *
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the Standby position.")
    //tested on CPPM testbench in september 2015
    public void goToStandby() throws BadCommandException,
            ErrorInCommandExecutionException,
            FcsHardwareException {
        if (this.atStandby) {
            throw new BadCommandException(name + " is already at  Standby Position.");
        }
        moveToAbsoluteTargetPosition(this.standbyPosition);
    }

    /**
     * Move the trucks to an absolute position given as argument.
     *
     * @param targetPosition
     * @throws BadCommandException
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws ErrorInCommandExecutionException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the absolute position given as argument.",
            alias = "mobeABSPos")
    //tested on CPPM testbench in september 2015
    public void moveToAbsoluteTargetPosition(int targetPosition) throws BadCommandException,
            SDORequestException, ShortResponseToSDORequestException,
            ErrorInCommandExecutionException, FcsHardwareException {

        this.updatePosition();
        if (position == targetPosition) {
            throw new BadCommandException(name
                    + " is already at target position=" + targetPosition);
        }

        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            String msg = name + " cant' do homing because controllers are not both initialized:\n";
            msg = msg + name + ((PieceOfHardware) linearRailSlaveController).toString();
            msg = msg + name + ((PieceOfHardware) linearRailMasterController).toString();
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (!homingDone) {
            throw new FcsHardwareException(name
                    + " can't be moved because the homing has not been done yet. ");
        }

        if (targetPosition < minActualPositionValue || targetPosition > maxActualPositionValue) {
            throw new IllegalArgumentException(name + "argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue);
        }

        this.absoluteTargetPosition = targetPosition;

        FCSLOG.info(name + " going to absolute position: " + absoluteTargetPosition);

        this.executeAction(MOVE_TO_ABSOLUTE_POSITION, timeoutForTrucksMotion);
    }

    /**
     * Move the trucks to a relative position given as argument.
     *
     * @param relativePosition
     * @throws BadCommandException
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException
     * @throws ErrorInCommandExecutionException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Move Autochanger trucks to the relative position given as argument.",
            alias = "moveRELPos")
    //tested on CPPM testbench in september 2015
    //ONLY for small values for test purpose to realign the trucks
    public void moveToRelativeTargetPosition(int relativePosition) throws BadCommandException,
            SDORequestException, ShortResponseToSDORequestException,
            ErrorInCommandExecutionException,
            FcsHardwareException {

        if (relativePosition < -2000 || relativePosition > 2000) {
            throw new IllegalArgumentException("Relative position value has to be between -2000 and 2000");
        }

        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            String msg = name + " cant' do homing because controllers are not both initialized:\n";
            msg = msg + name + ((PieceOfHardware) linearRailSlaveController).toString();
            msg = msg + name + ((PieceOfHardware) linearRailMasterController).toString();
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (!homingDone) {
            throw new BadCommandException(name
                    + " can't be moved because the homing has not been done. ");
        }

        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        updatePosition();

        if (position + relativePosition < minActualPositionValue || position + relativePosition > maxActualPositionValue) {
            throw new IllegalArgumentException(name + "actual position + argument has to be between "
                    + minActualPositionValue + " and " + maxActualPositionValue);
        }

        this.relativeTargetPosition = position + relativePosition;

        FCSLOG.info(name + " going to relative position: " + relativeTargetPosition);

        this.executeAction(MOVE_TO_RELATIVE_POSITION, timeoutForTrucksMotion);
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void startAction(FcsEnumerations.MobileItemAction action)
            throws BadCommandException, ErrorInCommandExecutionException,
            FcsHardwareException {
        //TODO quand on aura les 2 encodeurs, il faudra vérifier que les 
        //paramètres de PROFILE_POSITION sont identiques sur les 2 contrôleurs.
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(this.absoluteTargetPosition);
                linearRailMasterController.writeControlWord("3F");
                break;

            case MOVE_TO_RELATIVE_POSITION:
                this.enableControllersAndReleaseBrake();
                linearRailMasterController.writeTargetPosition(this.relativeTargetPosition);
                linearRailMasterController.writeControlWord("7F");
                break;
                
            default:
                assert false : action;
        }
    }
    
    /**
     * This method is used in the startAction method.
     * It is was has to be done before moving the trucks.
     * @throws SDORequestException
     * @throws ShortResponseToSDORequestException 
     */
    private void enableControllersAndReleaseBrake() throws SDORequestException, 
            ShortResponseToSDORequestException,
            FcsHardwareException {
        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        linearRailSlaveController.changeMode(EposMode.MASTER_ENCODER);
        if (!linearRailMasterController.isEnabled()) {
            linearRailMasterController.enable();
        }
        if (!linearRailSlaveController.isEnabled()) {
            linearRailSlaveController.enable();
        }
        linearRailMasterController.releaseBrake();
        linearRailSlaveController.releaseBrake();
    }

    @Override
    //TODO test on CPPM test bench
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay)
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        FCSLOG.info("ABORTING action: "+ action.toString());

        this.linearRailMasterController.off();
        this.linearRailSlaveController.off();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay)
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        this.linearRailMasterController.quickStop();
        this.linearRailSlaveController.quickStop();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void postAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        //because we don't want to let the controller on power
        this.linearRailSlaveController.disable();
        this.linearRailMasterController.disable();

        FCSLOG.info(name + ":" + action.toString() + " completed - doing postAction.");
    }

    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Read actual position and do the homing of the master controller"
            + "with this actual position.")
    public void homing() throws SDORequestException, ErrorInCommandExecutionException,
            ShortResponseToSDORequestException, BadCommandException, FcsHardwareException {

        if (!linearRailSlaveController.isInitialized() || !linearRailMasterController.isInitialized()) {
            FCSLOG.error(name + " cant' do homing because controllers are not both initialized:");
            FCSLOG.info(name + ((PieceOfHardware) linearRailSlaveController).toString());
            FCSLOG.info(name + ((PieceOfHardware) linearRailMasterController).toString());
        }

        //TODO check if parameters are OK in checkHardware method
        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        linearRailMasterController.writeParameters(EposMode.PROFILE_POSITION);
        linearRailMasterController.enable();

        linearRailSlaveController.changeMode(EposMode.MASTER_ENCODER);
        linearRailSlaveController.writeParameters(EposMode.PROFILE_POSITION);
        linearRailSlaveController.enable();

        FCSLOG.debug(name + "==> BEGIN homing of the trucks");
        int ssiPosition = linearRailMasterController.readSSIPosition();
        int offset = ssiPosition - encoderRibbonMinValue;
        FCSLOG.debug(name + " position=" + ssiPosition + ",offset=" + offset);

        //defineAbsolutePosition change EPOS mode to HOMING
        linearRailMasterController.defineAbsolutePosition(offset);
        linearRailMasterController.off();
        linearRailMasterController.changeMode(EposMode.PROFILE_POSITION);
        homingDone = true;
        FCSLOG.debug(name + " ==> END homing of the trucks");
    }

    /**
     * Updates the field position of the carrier in reading the CPU of the
     * controller.
     *
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller.")
    public void updatePosition() throws BadCommandException, SDORequestException, 
            FcsHardwareException {
        try {
            this.position = linearRailMasterController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            FCSLOG.warning(name + "=> ERROR IN READING CONTROLLER:" + ex);

        }
        this.publishData();
    }

    /**
     * For the GUI and end users for test purpose. Update positon in reading the
     * master controller and returns the value read.
     *
     * @return trucks position
     * @throws BadCommandException
     * @throws SDORequestException
     */
    //tested on CPPM testbench in september 2015
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Update trucks position in reading controller and return position.")
    public int readPosition() throws BadCommandException, SDORequestException, FcsHardwareException {
        updatePosition();
        return position;
    }

    @Override
    //tested on CPPM testbench in september 2015
    public boolean isHardwareReady() {
        //commented out for CPPM testbench
        //return autochanger.isHardwareReady();
        return this.isInitialized();
    }

    @Override
    //tested on CPPM testbench in september 2015
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION:
                return this.position == this.absoluteTargetPosition;

            case MOVE_TO_RELATIVE_POSITION:
                return this.position == this.relativeTargetPosition;
                
            default :
                assert false: action;
        }
        return false;
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        try {
            linearRailMasterController.checkFault();
            linearRailSlaveController.checkFault();
            this.position = this.linearRailMasterController.readPosition();
            FCSLOG.debug(name + " position=" + this.position);
        } catch (ShortResponseToSDORequestException ex) {
            //changed for sonar in sept2015
            //fcslog.warning(name + "=> SDO ERROR IN READING CONTROLLER:" + ex.toString());
            FCSLOG.warning(name + "=> SDO ERROR IN READING CONTROLLER:" + ex);
        }
        //for the GUI : is it the right place ?
        //Commented out for CPPM testbench because autochanger doesn't exist.
//        try {
//            this.updateStateWithSensors();
//        } catch (ShortResponseToSDORequestException | SDORequestException ex) {
//            FCSLOG.warning(name + "=> SDO ERROR IN READING CONTROLLER:" + ex.getMessage());
//        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "Update Trucks state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        autochanger.updateStateWithSensors();
    }

    /**
     * This methods updates the lockStatus from an array of hexaValues.
     *
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    //TODO test with real hardware
    protected void updateStateWithSensors(String[] readHexaValues)
            throws FcsHardwareException {
        lock.lock();

        try {
            updatingState = true;

            this.handoffPositionSensor0.updateValue(readHexaValues);
            this.handoffPositionSensor1.updateValue(readHexaValues);
            this.onlinePositionSensor0.updateValue(readHexaValues);
            this.onlinePositionSensor1.updateValue(readHexaValues);
            this.standbyPositionSensor0.updateValue(readHexaValues);
            this.standbyPositionSensor1.updateValue(readHexaValues);

            //TODO compute a state 
            this.atHandoff = handoffPositionSensor0.getDigitalValue() == 1
                    && handoffPositionSensor1.getDigitalValue() == 1;
            this.atOnline = onlinePositionSensor0.getDigitalValue() == 1
                    && onlinePositionSensor1.getDigitalValue() == 1;
            this.atStandby = standbyPositionSensor0.getDigitalValue() == 1
                    && standbyPositionSensor1.getDigitalValue() == 1;

            this.handoffSensorsInError = handoffPositionSensor0.getDigitalValue() 
                    != handoffPositionSensor1.getDigitalValue();
            this.onlineSensorsInError = onlinePositionSensor0.getDigitalValue() 
                    != onlinePositionSensor1.getDigitalValue();
            this.standbySensorsInError = standbyPositionSensor0.getDigitalValue() 
                    != standbyPositionSensor1.getDigitalValue();

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();

        }

    }

    /**
     * This configures the digital inputs of the EPOS controller to work with a
     * holding brake. This configuration is required for the controllers of the
     * Linear Rails of the autochanger. Digital Inputs activated for Linear
     * Rails : Drive Enable Input=01 Mask=Enabled Polarity=HighActive QuickStop
     * Input=02 Mask=Enabled Polarity=HighActive
     *
     *          *********************************************
     * 1 - Configure the Digital Input Functionnality index 0x2070 subindex 01
     * to 04 Functionnality tab Value = 15 General Purpose A Value = 14 General
     * Purpose B Value = 13 General Purpose C Value = 12 General Purpose D Value
     * = 11 General Purpose E Value = 10 General Purpose F Value = 9 General
     * Purpose G Value = 8 General Purpose H Value = 7 General Purpose I Value =
     * 6 General Purpose J Value = 5 QuickStop 1 Value = 4 Drive enable 1 Value
     * = 3 Position marker 0 Value = 2 Home switch 0 Value = 1 Positive Limit
     * switch 0 Value = 0 Negative Limit switch 0
     * ********************************************* 2 - Configure the Digital
     * Input Polarity index 0x2071 subindex 03 Value=0 HighActive Value=1 Low
     * Active ********************************************* 3 - Configure the
     * Digital Input Mask index 0x2071 subindex 02 Value=0 HighActive Value=1
     * Low Active ******************************************** 4 - Configure the
     * Digital Input Execution Mask index 0x2071 subindex 04 Used to suppress
     * execution of the digital input functionalities. The function (Negative or
     * Positive Limit Switch Error Routine) will be executed when the associated
     * bit in functionalities state register goes high and the bit in this
     * execution mask is set. Functionnality tab Value = 15 General Purpose A 0
     * Value = 14 General Purpose B 0 Value = 13 General Purpose C 0 Value = 12
     * General Purpose D 0 Value = 11 General Purpose E 0 Value = 10 General
     * Purpose F 0 Value = 9 General Purpose G 0 Value = 8 General Purpose H 0
     * Value = 7 General Purpose I 0 Value = 6 General Purpose J 0 Value = 5
     * QuickStop 1 Value = 4 Drive enable 1 Value = 3 Position marker 0 Value =
     * 2 Home switch 0 Value = 1 Positive Limit switch 0 Value = 0 Negative
     * Limit switch 0
     *
     * To configure the autochanger linear rails controllers. This method is to
     * be executed during initialization phase.
     *
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the autochanger linear rails controllers.")
    //tested on CPPM testbench in september 2015
    public static void configureDigitalInputOfLinearRails(EPOSController ctl) throws SDORequestException, FcsHardwareException {
        //Drive Enable
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1, "0004");

        //QuickStop
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput2, "0005");

        //DigitalInputPolarity 
        //Value=0 HighActive
        //Value=1 Low Active
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity, "0");

        //DigitalInputFonctionnalitiesMask
        //Used to display the state of the digital input functionalities. 
        //If a bit is set to “1”, the functionality state will be displayed.
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask, "30");

        //DigitalInputFonctionnalitiesExecutionMask
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask, "30");

    }

    /**
     * Configuration of an EPOS controller to work with a holding brake. This is
     * used for controllers of the Autochanger Linear Rails.
     *
     *          * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ Digital Outputs
     * activated for Linear Rails are:
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ General_A Output=01
     * Mask=Enabled Polarity=HighActive Ready/Fault Output=02 Mask=Enabled
     * Polarity=HighActive Holding Brake Output=03 Mask=Enabled
     * Polarity=LowActive ******************************************** 1- First
     * of all, mask all outputs index 0x2079 subindex 01
     * ********************************************** 2- Configure the Digital
     * Output Fonctionnalities index 0x2079 subindex 01 to 03 Functionnality tab
     * Value = 15 General Purpose Out_A Value = 14 General Purpose Out_B Value =
     * 13 General Purpose Out_C Value = 12 General Purpose Out_D Value = 11
     * General Purpose Out_E Value = 10..8 not used Value = 7..3 reserved Value
     * = 2 Holding Brake Value = 1 Position compare Value = 0 Ready/Fault
     * ************************************************************************************
     * 3 - Configure the Digital Output Functionnality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value
     * = 1 associated output inverted => LowActive Tab Digital Output
     * Functionnality Mask Value Bit_15 General Purpose Out_A 0 Bit_14 General
     * Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12 General Purpose
     * Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General Purpose Out_F 0
     * Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H 0 Bit_7
     * reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0 Bit_3
     * reserved 0 Bit_2 Holding Brake 1 (read only) Bit_1 Position Compare 0
     * (read only) Bit_0 Ready/Fault 0 (read only)
     * ********************************************************************************
     * 4 - Configure the Digital Output Functionnality Mask index 0x2078
     * subindex 0x02 Value = 0 functionnality not activatd Value = 1
     * functionnality activated
     * ********************************************************************************
     * Digital Output Functionnality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12
     * General Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General
     * Purpose Out_F 0 Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H
     * 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0
     * Bit_3 reserved 0 Bit_2 Holding Brake 1 Bit_1 Position Compare 0 Bit_0
     * Ready/Fault 1
     * ********************************************************************************
     * To configure the autochanger linear rails controllers. This method is to
     * be executed during initialization phase.
     *
     * @param ctl controller
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1,
            description = "To configure the autochanger linear rails controllers.")
    //tested on CPPM testbench in september 2015
    public static void configureDigitalOutputOfLinearRails(EPOSController ctl) throws SDORequestException, FcsHardwareException {
        //1-Mask all outputs
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "0");

        //2-Configuration of Digital Output Fonctionnalities
        //General_A
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1, "000F");
        //Ready/Fault
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput2, "0");
        //Holding Brake
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput3, "0002");

        //3-Configuration of Digital Output Fonctionnality Polarity
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity, "0004");

        //4-Configuration of Digital Output Fonctionnality Mask
        ctl.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "8005");
    }

    /**
     * Build the class StatusDataPublishedByAutoChangerTrucks
     *
     * @return status data
     */
    public StatusDataPublishedByAutoChangerTrucks getStatusData() {
        return createStatusDataPublishedByAutoChangerTrucks();
    }
    
    /**
     * Creates an object to be published on the status bus.
     * @return 
     */
    public StatusDataPublishedByAutoChangerTrucks createStatusDataPublishedByAutoChangerTrucks() {
        StatusDataPublishedByAutoChangerTrucks s = new StatusDataPublishedByAutoChangerTrucks();
        s.setName(name);
        s.setPosition(position);
        s.setMasterControllerInFault(isMasterControllerInFault());
        s.setSlaveControllerInFault(isSlaveControllerInFault());
        s.setCurrent(current);
        s.setSpeed(speed);
        s.setHandoffInError(this.handoffSensorsInError);
        s.setOnlineInError(this.onlineSensorsInError);
        s.setStandbyInError(this.standbySensorsInError);
        s.setHandoffSensorValue0(handoffPositionSensor0.getDigitalValue());
        s.setHandoffSensorValue1(handoffPositionSensor1.getDigitalValue());
        s.setOnlineSensorValue0(onlinePositionSensor0.getDigitalValue());
        s.setOnlineSensorValue1(onlinePositionSensor1.getDigitalValue());
        s.setStandbySensorValue0(standbyPositionSensor0.getDigitalValue());
        s.setStandbySensorValue1(standbyPositionSensor1.getDigitalValue());
        return s;
    }

    @Override
    public void publishData() {
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData(name, getStatusData()));
    }

    /**
     * What to do when the Modules we observe (the controllers) send new
     * values.
     *
     * @param source
     * @param v
     */
    @Override
    //tested on CPPM testbench in september 2015
    public void processUpdate(Observable source, ValueUpdate v) {
        FCSLOG.debug(name + ":processUpdate from source=" + source.toString()
                + " ValueUpdate=" + v.getName());
        //TODO to be tested on CPPM testbench in January 2016
        //replaced CanOpenEPOS by EPOSController in January 2016 to breal dependies between packages.
        if (!(source instanceof EPOSController)) { 
            return;
        }

        if (v.getValue() instanceof EmergencyMessage) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            FCSLOG.debug(name + ":EmergencyMessage received from CanOpenProxy="
                    + emcyMsg.toString());
            //TODO check that an ALERT has been raised from the controller.
            this.handleEmergencyMessageFromControllers(emcyMsg);

        //a faultReset or a checkFault has been addressed to a controller
        } else if (v.getValue() instanceof String) {
        //TODO to be tested on CPPM testbench in January 2016
        //replaced CanOpenEPOS by EPOSController in January 2016 to breal dependies between packages.
            EPOSController ctrl = (EPOSController) source;
            String msgFromController = (String) v.getValue();
            //TODO check that msgController is not null 
            
            this.handleStringMessageFromControllers(ctrl.getName(),msgFromController);

        }
    }
    
    /**
     * When an Emergency message comes from one of the 2 controllers :
     * - update the boolean masterControllerInFault or slaveControllerInFault.
     * - publish data
     * An ALERT has already been raised by the device itself.
     * (see CanOpenEPOS processUpdate method)
     * @param emcyMsg 
     */
    private void handleEmergencyMessageFromControllers(EmergencyMessage emcyMsg) {
        //an emergency message is coming from the master controller
        if (this.linearRailMasterController.getName().equals(emcyMsg.getDeviceName())) {
            String errCode = emcyMsg.getDeviceErrorCode();
            switch (errCode) {
                case "00":
                    FCSLOG.debug(name + ":faultReset ?=" + emcyMsg.toString());
                    masterControllerInFault = false;
                    break;

                default:
                    FCSLOG.debug(name + ":EmergencyMessage received for "
                            + "master controller from CanOpenProxy=" + emcyMsg.toString());
                    masterControllerInFault = true;

            }
            this.publishData();

            //an emergency message is coming from the slave controller    
        } else if (this.linearRailSlaveController.getName().equals(emcyMsg.getDeviceName())) {
            String errCode = emcyMsg.getDeviceErrorCode();
            switch (errCode) {
                case "00":
                    FCSLOG.debug(name + ":faultReset ?=" + emcyMsg.toString());
                    slaveControllerInFault = false;
                    break;

                default:
                    FCSLOG.debug(name + ":EmergencyMessage received for "
                            + "slave controller from CanOpenProxy=" + emcyMsg.toString());
                    slaveControllerInFault = true;

            }
            this.publishData();
        }
    }
    
    private void handleStringMessageFromControllers(String controllerName, 
            String msgFromController) {
        if (controllerName.equals(linearRailMasterController.getName())) {
             
            switch (msgFromController) {
                case "faultReset":
                    this.masterControllerInFault = false;
                    this.publishData();
                    break;

                case "checkFault":
                    this.masterControllerInFault = linearRailMasterController.isInError();
                    break;
                    
                default:
                    assert false: msgFromController;
                }
            
            //slave controller
            } else if (controllerName.equals(linearRailSlaveController.getName())) {
            switch (msgFromController) {
                    case "faultReset":
                        this.slaveControllerInFault = false;
                        this.publishData();
                        break;
                    case "checkFault":
                        this.slaveControllerInFault = linearRailSlaveController.isInError();
                        break;
                    default:
                        assert false: msgFromController;
                }
            }
    }

}
