
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import static org.lsst.ccs.subsystems.fcs.FCSCst.FCSLOG;

/**
 * This class is used with the final hardware in which we have a clamp online to
 * hold the filter at online position. It's not used for Single Filter Test but
 * is used for testbenchCPPM.
 *
 * @author virieux
 */
public class AutoChangerModule extends Module implements HardwareController {

    

    private BridgeToHardware bridge;
    private final String plutoGatewayName;
    private PlutoGatewayModule plutoGateway;

    private final NumericSensor loaderConnectedSensor0;
    private final NumericSensor loaderConnectedSensor1;

    private final AutoChangerTrucksModule autochangerTrucks;
    private final AutochangerTwoLatches latches;
    private final AutochangerThreeOnlineClamps onlineClamps;
    private CarouselModule carousel;
    private LoaderModule loader;

    private boolean loaderConnectedSensorsInError;
    private boolean empty;
    private boolean loaderConnected;

    //TODO initialization has to go in the constructor and the value provided 
    //in the description file.
    private final boolean standalone = true;

    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

    public AutoChangerModule(String name, int tickMillis,
            String plutoGatewayName,
            NumericSensor loaderConnectedSensor0,
            NumericSensor loaderConnectedSensor1,
            AutoChangerTrucksModule trucks,
            AutochangerTwoLatches latches,
            AutochangerThreeOnlineClamps onlineClamps) {
        super(name, tickMillis);
        this.plutoGatewayName = plutoGatewayName;
        this.loaderConnectedSensor0 = loaderConnectedSensor0;
        this.loaderConnectedSensor1 = loaderConnectedSensor1;
        this.autochangerTrucks = trucks;
        this.latches = latches;
        this.onlineClamps = onlineClamps;
        this.loaderConnected = false;
        this.loaderConnectedSensorsInError = false;
    }

    public NumericSensor getLoaderConnectedSensor0() {
        return loaderConnectedSensor0;
    }

    public NumericSensor getLoaderConnectedSensor1() {
        return loaderConnectedSensor1;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Return false if the 2 redondant loader sensors are equal.")
    public boolean isLoaderConnectedSensorsInError() {
        return loaderConnectedSensorsInError;
    }

    //for the simulator
    public int getHandoffPosition() {
        return this.autochangerTrucks.getHandoffPosition();
    }

    //for the simulator

    public int getOnlinePosition() {
        return this.autochangerTrucks.getOnlinePosition();
    }

    //for the simulator

    public int getStandbyPosition() {
        return this.autochangerTrucks.getStandbyPosition();
    }

    public AutochangerTwoLatches getLatches() {
        return latches;
    }

    public AutoChangerTrucksModule getTrucks() {
        return autochangerTrucks;
    }

    public AutochangerThreeOnlineClamps getOnlineClamps() {
        return onlineClamps;
    }

    public PlutoGatewayModule getPlutoGateway() {
        return plutoGateway;
    }

    /**
     * Returns the boolean field loaderConnected. If the loaderConnected boolean
     * is being updated and waits for a response from a sensor, this methods
     * waits until loaderConnected is updated. If the field loaderConnected is
     * not being updated, it returns immediatly the field loaderConnected.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if the loader is connected to the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return loaderConnected && !loaderConnectedSensorsInError;

        } finally {
            lock.unlock();
        }
    }

    public boolean isCarouselHoldingFilterAtStandby() throws FcsHardwareException {
        if (standalone) {
            return true;
        } else {
            return this.carousel.isHoldingFilterAtStandby();
        }
    }

    @Override
    public void initModule() {
        FCSLOG.info(name + ": init MODULE");
        this.plutoGateway = (PlutoGatewayModule) this.getComponentByName(plutoGatewayName);
        this.bridge = (BridgeToHardware) this.getComponentByName("bridge");
        this.carousel = (CarouselModule) this.getComponentByName("carousel");
        this.loader = (LoaderModule) this.getComponentByName("loader");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger hardware is initialized and "
            + "bridge is hardwareReady.")
    public boolean isHardwareReady() {
        return bridge.isHardwareReady() && isInitialized();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized()
                && autochangerTrucks.isInitialized();
//                autochangerTrucks.isInitialized() &&
//                //latches.isInitialized();
    }

    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        FCSLOG.debug(name + " checking hardware.");
        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        //TODO this has to be done in each Module
        autochangerTrucks.checkHardware();
        latches.checkHardware();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    @Override
    public void checkStarted() throws HardwareException {
        FCSLOG.info(name + " BEGIN checkStarted");

        //check that all hardware is booted and identified.
        bridge.getTcpProxy().checkNewHardware();

        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true, ex);
        }
        autochangerTrucks.checkHardware();

    }

    /**
     * This method is executed during a shutdown of the subystem to prevent a
     * shutdown when some hardware is moving. Nothing to do here because all the
     * moving hardware of the autochanger is represented by the class
     * MobilItemModule where the method checkStopped is implemented.
     *
     * @throws HardwareException
     */
    @Override
    public void checkStopped() throws HardwareException {
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the latches can be opened.")
    public boolean checkPreConditionsForOpeningLatches() throws SDORequestException,
            BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        FCSLOG.info(name + " checking pre-conditions for opening latches");

        updateStateWithSensors();

//        if (!latches.isInitialized()) {
//            String msg = name  + ": latches are not intialized. PLease initialize hardware first.";
//            FCSLOG.error(msg);
//            throw new BadCommandException(msg);
//        }
//                
//        if (latches.isInError()) {
//            String msg = name  + ": latches are in ERROR state - can't unlockClamps latches.";
//            FCSLOG.error(msg);
//            throw new FcsHardwareException(msg);
//        }
        if (isEmpty()) {
            String msg = name + ": no filter in autochanger - can't open latches.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (!autochangerTrucks.isAtStandbyPosition() && !autochangerTrucks.isAtHandoffPosition()) {
            String msg = name + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open clamp.";
            FCSLOG.error(msg);
            throw new BadCommandException(msg);
        }

        if (autochangerTrucks.isAtStandbyPosition() && !isCarouselHoldingFilterAtStandby()) {
            String msg = name + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter "
                    + "- can't open latches.";
            FCSLOG.error(msg);
            throw new BadCommandException(msg);
        }

        if (autochangerTrucks.isAtHandoffPosition() && !isLoaderHoldingFilterAtHandoff()) {
            String msg = name + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter "
                    + "- can't open latches.";
            FCSLOG.error(msg);
            throw new BadCommandException(msg);
        }

        if (autochangerTrucks.isAtStandbyPosition() && isCarouselHoldingFilterAtStandby()) {
            return true;
        } else {
            return autochangerTrucks.isAtHandoffPosition() && isLoaderHoldingFilterAtHandoff();
        }

    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1,
            description = "Check if the latches can be closed.")
    public boolean checkPreConditionsForClosingLatches() throws FcsHardwareException, BadCommandException {

        updateStateWithSensors();

//        if (!latches.isInitialized()) {
//            String msg = name  + ": latches are not intialized. PLease initialize hardware first.";
//            FCSLOG.error(msg);
//            throw new BadCommandException(msg);
//        }
//        
        if (latches.isInError()) {
            String msg = name + ": latches are in ERROR state - can't close latches.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (isEmpty()) {
            String msg = name + ": no filter in autochanger - can't close latches.";
            FCSLOG.error(msg);
            throw new FcsHardwareException(msg);
        }

        if (!autochangerTrucks.isAtStandbyPosition() && !autochangerTrucks.isAtHandoffPosition()) {
            String msg = name + ": autochanger is not "
                    + "at handoff position neither at standby - can't close latches.";
            FCSLOG.error(msg);
            throw new BadCommandException(msg);
        } else {
            return true;
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if loader is connected and holding a filter. "
            + "This command doesn't read again the sensors.")
    boolean isLoaderHoldingFilterAtHandoff() throws BadCommandException,
            FcsHardwareException {
        if (loaderConnected) {
            return this.loader.isHoldingAFilter();
        }
        return false;
    }

    /**
     * This methods updates the autochangerTrucks, latches and online clamps state in
 reading all the sensors.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING1, description = "Update clamp state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {

        if (!plutoGateway.isBooted()) {
            throw new FcsHardwareException(name + ": plutoGateway not booted - can't read sensors");
        }

        if (!plutoGateway.isInitialized()) {
            throw new FcsHardwareException(name + ": plutoGateway not initialized - can't read sensors");
        }

        lock.lock();
        try {
            updatingState = true;
            this.plutoGateway.updateValues();
            String[] readHexaValues = this.plutoGateway.getHexaValues();
            FCSLOG.finest("readHexaValues[0]=" + readHexaValues[0]);
            FCSLOG.finest("readHexaValues[1]=" + readHexaValues[1]);
            FCSLOG.finest("readHexaValues[2]=" + readHexaValues[2]);
            FCSLOG.finest("readHexaValues[3]=" + readHexaValues[3]);

            //update of loaderConnectedSensors
            this.loaderConnectedSensor0.updateValue(readHexaValues);
            this.loaderConnectedSensor1.updateValue(readHexaValues);

            this.loaderConnected = loaderConnectedSensor0.getDigitalValue() == 1
                    && loaderConnectedSensor1.getDigitalValue() == 1;

            this.loaderConnectedSensorsInError = loaderConnectedSensor0.getDigitalValue()
                    != loaderConnectedSensor1.getDigitalValue();

            this.autochangerTrucks.updateStateWithSensors(readHexaValues);
            this.latches.updateStateWithSensors(readHexaValues);
            this.onlineClamps.updateStateWithSensors(readHexaValues);

            this.empty = this.latches.getLatchXminus().isEmpty()
                    && this.latches.getLatchXplus().isEmpty();

        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
            this.autochangerTrucks.publishData();
            this.latches.publishData();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if autochanger trucks are at HANDOFF. "
            + "This command doesn't read again the sensors.")
    boolean isAtHandoff() {
        return autochangerTrucks.isAtHandoffPosition();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
            description = "Return true if there is no filter in the autochanger. "
            + "This command doesn't read again the sensors.")
    boolean isEmpty() {
        lock.lock();
        try {
            while (updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    FCSLOG.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return empty;

        } finally {
            lock.unlock();
        }
    }

    void goToHandOff() throws BadCommandException, ErrorInCommandExecutionException,
            FcsHardwareException {
        autochangerTrucks.goToHandOff();
    }

    void grabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void ungrabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void moveFilterToHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void grabFilterAtStandby(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void tick() {
        //Don't execute here methods which take a lock.
        this.publishData();
    }

    public StatusDataPublishedByAutoChanger getStatusData() {
        StatusDataPublishedByAutoChanger status = createStatusDataPublishedByAutoChanger();
        return status;
    }
    
    /**
     * Creates an Object to be published on the STATUS bus.
     * @return 
     */
    public StatusDataPublishedByAutoChanger createStatusDataPublishedByAutoChanger() {
        StatusDataPublishedByAutoChanger status = new StatusDataPublishedByAutoChanger();
//        status.setTrucksEmpty(autochanger.isTrucksEmpty());
//        status.setTrucksLocation(autochanger.getTrucksLocation());
        status.setName(name);
        status.setLoaderConnectedSensorValue0(this.loaderConnectedSensor0.getDigitalValue());
        status.setLoaderConnectedSensorValue1(this.loaderConnectedSensor1.getDigitalValue());
        status.setLoaderConnectedSensorsInError(this.isLoaderConnectedSensorsInError());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        StatusDataPublishedByAutoChanger status = this.getStatusData();
        this.getSubsystem().publishSubsystemDataOnStatusBus(new KeyValueData("autochangerGeneral", status));
    }

}
