/*
 * To change this template, choose Tools | Templates
 * and on the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceInLatchStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.NumericSensor;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerLatch;
import org.lsst.ccs.subsystems.fcs.common.FilterLatch;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for a latch. 
 * A latch is what holds a filter in the trucks of the autochanger.
 There is 2 latches : one on each side of the filter.
 The state of the  latches are monitoring by digital sensors :
 one sensor to know if a filter is in the latch or not and 
 2 status sensors to know if the latch is EngageBrakeed or ReleaseBrakeed.
 If the 2 status sensors are inconsistent, the latch is in error.
 * @author virieux
 */
public class FilterLatchModule extends MobileItemModule implements FilterLatch {










    private static class LatchError extends Exception {
        FilterLatchModule latch;
               
        public LatchError(String string, FilterLatchModule aThis) {
            super(string);
            latch = aThis;
        }
    }


    private boolean filterEngaged;
    private boolean locked;
    private boolean unlocked;
    private boolean inError;
    
    private FilterPresenceInLatchStatus presenceStatus;
    private LockStatus lockStatus;
    
    LatchActuatorModule latchActuator;
    protected NumericSensor filterPresenceSensor;
    protected NumericSensor lockSensor;
    protected NumericSensor unlockSensor;
    CompactIOModule filterSensorsDIO;
     /**
     * This is the name of the DIO on which are plugged the filter sensors.
     * Used in the groovy file and in initModule.
     * 
     */
    String filterSensorsDIOName;
    
    
    //Used because we have to wait for the update from the sensors to know the state
    //of the latch
    final Condition stateUpdated = lock.newCondition();
    
     /**
     * This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    //private long timeToWaitForCompleteOpen;
    //private long timeToWaitForCompleteClose;
    private long timeoutForOpening;       
    private long timeoutForClosing;

    public FilterLatchModule(String moduleName, int aTickMillis, 
            long timeoutForClosing,
            long timeoutForOpening,
            String filterSensorsDIOName, 
            NumericSensor lockSensor, 
            NumericSensor unlockSensor, 
            NumericSensor filterPresenceSensor,
            LatchActuatorModule latchActuator) {
        super(moduleName, aTickMillis);
        this.timeoutForClosing = timeoutForClosing;
        this.timeoutForOpening = timeoutForOpening;
        this.filterSensorsDIOName = filterSensorsDIOName;
        this.lockSensor = lockSensor;
        this.unlockSensor = unlockSensor;
        this.filterPresenceSensor = filterPresenceSensor;       
        this.latchActuator = latchActuator;

    }

    
    @ConfigChanger
    public void setTimeoutForClosing(long timeoutForClosing) {
        this.timeoutForClosing = timeoutForClosing;
    }

    
    @ConfigChanger
    public void setTimeoutForOpening(long timeoutForOpening) {
        this.timeoutForOpening = timeoutForOpening;
    }
    
    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }
    
    //for FCSUtils
    public NumericSensor getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    //for FCSUtils
    public LatchActuatorModule getLatchActuator() {
        return latchActuator;
    }

    //for FCSUtils
    @Override
    public NumericSensor getLockSensor() {
        return lockSensor;
    }

    //for FCSUtils
    @Override
    public NumericSensor getUnlockSensor() {
        return unlockSensor;
    }
    
    //for FCSUtils
    @Override
    public boolean isControllerInFault() {
        return false;
    }

    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    @Override
    public void initModule() {
        this.filterSensorsDIO = (CompactIOModule) this.getModule(this.filterSensorsDIOName);
        this.lockStatus = LockStatus.UNKNOWN;
        this.presenceStatus = FilterPresenceInLatchStatus.UNKNOWN;
        this.filterEngaged = false;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return ((MainModule) this.getModule("main")).isHardwareReady();
    }
    
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.OPEN)) {
            return this.getLockStatus().equals(LockStatus.UNLOCKED);
        } else if (action.equals(MobileItemAction.CLOSE)) {
            return this.getLockStatus().equals(LockStatus.LOCKED);
        } else {
            throw new IllegalArgumentException("Action on latch must be OPEN or CLOSE");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        this.updateStateWithSensors();
    }
    
    @Command ( level=Command.ENGINEERING1, description="Read latch sensors and update latch state", type=Command.CommandType.QUERY)
    public void updateStateWithSensors() throws Exception {
        filterSensorsDIO.updateValue();
        
        String filterSensorsHexaValue = filterSensorsDIO.getHexaValue();

        this.updateState(filterSensorsHexaValue);
    }

    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        if (action.equals(MobileItemAction.OPEN)) {
            this.latchActuator.open();
        } else if (action.equals(MobileItemAction.CLOSE)) {
            this.latchActuator.close();        
        } else throw new IllegalArgumentException(getName() + "Action on latch must be OPEN or CLOSE");
    }
    
        @Override
    public void abortAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void postAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
    }

    public LockStatus getLockStatus() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": getLockStatus was interrupted while waiting for update.");
                }

            }
            return lockStatus;
            
        } finally {
            lock.unlock();
        }
        
    }

    public void setLockStatus(LockStatus lockStatus) {
        this.lockStatus = lockStatus;
    }
    
    public FilterPresenceInLatchStatus getPresenceStatus() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": getPresenceStatus was interrupted while waiting for update.");
                }

            }
            return presenceStatus;
            
        } finally {
            lock.unlock();
        }
        
    }


    public StatusDataPublishedByAutochangerLatch getStatusData() {
        StatusDataPublishedByAutochangerLatch status = FcsUtils.createStatusDataPublishedByLatch(this);
        return status;
    }

    /**
     * Open the latch if the carousel is holding the filter.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws Exception 
     */
    @Command ( level=Command.ENGINEERING3, description="Open latch", type=Command.CommandType.ACTION)
    public  String open()  throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, Exception {
        
        this.updateStateWithSensors();
        
        if (!this.isLocked()) return getName() + "is already OPEN";

        CarouselModule carousel = (CarouselModule) this.environment.getComponentByName("carousel");
        
        if (!carousel.isHoldingFilterAtStandby()) throw new BadCommandException(getName() + "CANNOT OPEN LATCH at STANDBY : CAROUSEL is NOT HOLDING THE FILTER");
               
        this.executeAction(MobileItemAction.OPEN, this.timeoutForOpening);
        
        this.latchActuator.powerOff();
    
        return getName() + " is OPEN";
        
    }

    /**
     * Close the latch : turn on the actuator, read the sensors and wait until the latch is closed.
     * If the latch is not closed during the delai : timeoutForClosing throws an Exception.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws Exception 
     */
    @Command ( level=Command.ENGINEERING3, description="Close latch", type=Command.CommandType.ACTION)
    public String close() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, Exception {
        
        this.updateStateWithSensors();
        
        if (this.isLocked()) return getName() + "is already CLOSED";
               
        this.executeAction(MobileItemAction.CLOSE, this.timeoutForClosing);
        
        this.latchActuator.powerOff();
    
        return getName() + " is CLOSED";
        
    }

    /**
     * 
     * @return true if the latch is closed.
     */
    @Command ( level=Command.ENGINEERING1, description="Return true if the latch is closed", type=Command.CommandType.QUERY)
    public boolean isLocked() {       
        return (this.getLockStatus().equals(LockStatus.LOCKED));
    }

    
    
    public void updateState(String hexaValue) {
        lock.lock();
        try {
            updatingState = true; 
            this.filterPresenceSensor.updateValue(hexaValue);
            this.lockSensor.updateValue(hexaValue);
            this.unlockSensor.updateValue(hexaValue);
            
            if (this.filterPresenceSensor.getDigitalValue() == 1) {
                filterEngaged = true;
                this.presenceStatus = FilterPresenceInLatchStatus.ENGAGED;
            } else {
                filterEngaged = false;
                this.presenceStatus = FilterPresenceInLatchStatus.NOFILTER;
            }
            locked = (this.lockSensor.getDigitalValue() == 1);
            unlocked = (this.unlockSensor.getDigitalValue() == 1);
            inError = (locked && unlocked);
            
            if (inError) {
                lockStatus = LockStatus.ERROR;
            } else if (locked) {
                lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = LockStatus.UNLOCKED;
            } else {
                lockStatus = LockStatus.UNKNOWN;
            }

            

            
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }
        
        this.publishData();

    }
    
     /**
     * Publish Data on status bus for trending data base and GUIs.
     * 
     */
    @Override
     public void publishData() {
        StatusDataPublishedByAutochangerLatch status = this.getStatusData(); 
        this.publish("autochangerLatch", status);
     }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/ lock sensor=");sb.append(this.lockSensor.getName());
        sb.append("/ unlock sensor=");sb.append(this.unlockSensor.getName());
        sb.append("/ filter presence sensor=");sb.append(this.filterPresenceSensor.getName());
        return sb.toString();
    }
    
    @Override
    public void quickStopAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }


    
}
