/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.EmergencyMessage;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByHardware;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.*;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This Module starts a tcpip server, waits for the connection of a client whose 
 * name is the value of the field myClientName.
 * This client is supposed to be a CanOpen C-wrapper.
 * When the client starts it scans the can open network and send to the server
 * some information on the can open nodes living on the network.
 * 
 * @author virieux
 */
public class CanOpenProxy extends FcsTcpProxy implements HardwareController {

    
    
    /**
     * To store the values returned by the PDO.
     */
    public static class PDOStorage implements Serializable {
        
        String sep = "_";
        String varSuffixe = "ADC";
        /**
         * a hash map for the storage of the values sent from the PDO.
         * key = varSuffixe + nodeID + sep + input numero  Example : ADC23_5 
         * value = value in hexa
         */
        //private HashMap<String, String> mapSensorsValues = new HashMap<String, String>();    
        private final HashMap<String, String> mapSensorsValues = new HashMap<>(); 
        
        /**
         * update the PDO storage in reading a PDO line.
         * 
         * @param pdoLine something like : "ADC1_1=10,ADC1_2=F53B, ...."
         */
        void updatePDOs(String pdoLine) throws PDOBadResponseException {
            fcslog.finest(": pdoLine=" + pdoLine);
            String[] words = pdoLine.split(",");
            fcslog.finest(": words.length=" + words.length);
            for (int ix=1; ix<words.length; ix++) {
                //example toStore= ADC1_1=10
                String[] toStore = words[ix].split("=");
                //example varName= ADC1_1
                String varName = toStore[0];
                String varValue = toStore[1];
                String[] mots = varName.split(sep);
                String nodeID = mots[0].substring(3);
                fcslog.finest(String.format("%nwords(%s):%s nodeID=%s", ix, words[ix], nodeID));
                if (mots.length > 1) {
                    String key = nodeID + sep + mots[1];
                    mapSensorsValues.put(varName,varValue);
                } else {
                    fcslog.error("Received a bad response from PDO request: " + pdoLine);
                    throw new PDOBadResponseException("Received a bad response from PDO request.", pdoLine);
                }                
            }

        }

        public String getVarValue(String nodeID, String inputNB) {
            String token = varSuffixe + nodeID + sep + inputNB;
            return mapSensorsValues.get(token);
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder("PDO STORAGE:");
            sb.append("(SIZE=").append(mapSensorsValues.size()).append(")");
            sb.append(mapSensorsValues.toString());
            return sb.toString();
        }  
    }
    
    //private ApcTcpProxy tcpProxy;
    /*This is the key word that has to be sent by the client when it connects to the tcp proxy. */
    private final String myClientName;
    
    /*number of can open nodes expected on the network */
    private int expectedNodesNB = 0;
    
    /*number of booted nodes*/
    private int bootedNodesNB = 0;
    
   
    
    /*An array of booted hardware*/
    private ArrayList<CanOpenNode> nodes;
    
    
    private boolean hardwareBootProcessEnded = false;
    private boolean hardwareIdentified = false;
    
    /**
     * A timeout for the booting process : 
     * we don't want to wait longer that this elapse of time.
     * This has to be tuned during the test bench.
     * UNIT = milliseconds
     */
    private long hardwareBootTimeout;
    
    private long hardwareBootProcessBeginTime;
    
    /*An array of hardware controled by this main Module*/
    //PieceOfHardware[] hardwareList = {actuatorA, actuatorB, actuatorC};
    public PieceOfHardware[] hardwareList;
    
    private boolean canOpenNodeNumbersOK = false;
    StringBuilder errorMessageSB;
    boolean hardwareIDError;
    
    
    private PDOStorage pdoStorage;
    
    public CanOpenProxy(String aName, int aTickMillis, 
            int portNumber, 
            String aClientName, 
            long hardwareBootTimeout) {
        super(aName, aTickMillis, portNumber);
        this.myClientName = aClientName;
        this.hardwareBootTimeout = hardwareBootTimeout;
        nodes = new ArrayList<>(20);
        
    }
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    

    /**
     * @return the expectedNodesNB
     */
    public int getExpectedNodesNB() {
        return expectedNodesNB;
    }

    /**
     * @param nodeNB
     */
    public void setExpectedNodesNB(int nodeNB) {
        this.expectedNodesNB = nodeNB;
    }

    public long getHardwareBootTimeout() {
        return hardwareBootTimeout;
    }

    @ConfigChanger
    public void setHardwareBootTimeout(long hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }
    
    @ConfigChanger
    public void setHardwareBootTimeout(int hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }

    public PieceOfHardware[] getHardwareList() {
        return hardwareList;
    }

    
    
    
     /**
     * @return the bootedNodesNB
     */
    public int getBootedNodesNB() {
        return bootedNodesNB;
    }


   

    /**
     * @return the myClientName
     */
    public String getMyClientName() {
        return myClientName;
    }
    
    public ArrayList<CanOpenNode> getNodes() {
        return nodes;
    }
    
    public Map<String, CanOpenNode> bootedNodesMap;

    public Map<String, CanOpenNode> getBootedNodesMap() {
        return bootedNodesMap;
    }
    
    

    public PDOStorage getPdoStorage() {
        return pdoStorage;
    }
    
    public boolean isHardwareIdentified() {
        return hardwareIdentified;
    }
    
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    /**
     * This methods stops the CWrapper client, and the TCP server.
     */
    public void disconnectHardware() {
        //TODO we have to test if all the hardware is stopped before doing stopServer
        this.stopServer();
        nodes = new ArrayList<>(20);
        hardwareBootProcessBeginTime = 0;
        bootedNodesNB = 0;
        hardwareBootProcessEnded = false;
        hardwareIdentified = false;
        canOpenNodeNumbersOK = false;
        this.hardwareIDError = false;
        this.pdoStorage = new PDOStorage();
    }

    public void connectHardware() throws HardwareException { 
            this.startServer();
            this.startThreadReader();
    }
    
    @Override
    public void initModule() {
       super.initModule();
       hardwareBootProcessBeginTime = 0;
       bootedNodesNB = 0;
       hardwareBootProcessEnded = false;
       hardwareIdentified = false;
       canOpenNodeNumbersOK = false;
       this.hardwareIDError = false;
       this.pdoStorage = new PDOStorage();
    }
    
    @Override
    public void tick() {   
        //publishData();
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        long hardwareBootDuration = 0;
        
        fcslog.debug(name + ": BEGIN CHECKHARDWARE");
        
        // we can start the hardware booting process when the tcp client is ready. 
        if ((hardwareBootProcessBeginTime == 0) && isReady(myClientName)) {
            hardwareBootProcessBeginTime = System.currentTimeMillis();
            fcslog.debug(name + ": A CLIENT IS CONNECTED - BOOT PROCESS BEGIN");
        }
        
        // wait until the number of booted devices is the expected number (or greater)
        // or the timeout for the hardware boot process is exceeded.
        while ((hardwareBootDuration < hardwareBootTimeout) 
                && (this.bootedNodesNB < this.expectedNodesNB)) {
            hardwareBootDuration = System.currentTimeMillis() - hardwareBootProcessBeginTime;
            fcslog.debug(name + ":hardwareBootDuration=" + hardwareBootDuration);
        }
        hardwareBootProcessEnded = true;
        // and retrieve some information from the booted CanOpen nodes
        //(info,node_id)
        identifyHardware();
        this.publishData();
        fcslog.debug(name + ": END OF BOOT PROCESS");
        fcslog.info(name + "=>" + listNodes());        
        
        //check if the serial number of the hardware match with the node ID.
        checkCanOpenNodes();
    
        fcslog.debug(name + ": END CHECKHARDWARE");
        //for the GUI
        this.publishData();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    /**
     * checkNewHardware is called by command checkStarted() in higher level Modules and
     * is executed when a command completeInitialization is sent to the subsystem.
     * The command completeInitialization is sent by the end-user to complete the 
     * initialization phase when the subsystem was stopped during this phase because
     * of an error (HardwareException).
     * @throws org.lsst.ccs.HardwareException 
     */

    public void checkNewHardware() throws HardwareException {
        fcslog.info(name + " BEGIN checkStartedToCompleteInitialization");
        
        if (!this.isTcpServerStarted()) 
            throw new HardwareException(true, name 
                    + ": could not start tcp server.");
        identifyHardware();     

        checkCanOpenNodes();

    }
    
    
    @Override
    public void checkStarted() throws HardwareException {
        fcslog.info(name + " BEGIN checkStarted");
    }


    @Override
    public void checkStopped() throws HardwareException {       
    }
    
    
    
    
    

    
    /**
     * 
     * @param aNodeID
     * @return 
     */
    public String getNodeName(String aNodeID) {
        String nodeName = "Unbooted nodeID:" + aNodeID;
        for (PieceOfHardware hardware : this.hardwareList)  {
            if (hardware.getNodeID().equals(aNodeID)) {
                nodeName = hardware.getName();
                return nodeName;
            }
        }
        return nodeName;
    }
    
     /*Check if the String nodeID already exists in the nodes table.
     * If it is booted it is on the nodes table.    
     * It means that this node ID is on the can open bus.
     */
    public boolean isBooted(String aNodeID) {
        
        if (bootedNodesNB == 0) return false;
        boolean nodeFound = false;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
              if (!(nodes.get(i) == null) &&  (nodes.get(i).getNodeID().equals(aNodeID))) {
                nodeFound = true;
            }
        }
        return nodeFound;
    }

    /**
     * List the can open nodes which are in the nodes table.
     * @return the list of can open nodes and the information stored in this.nodes.
     */
    public String listNodes() {
        
        StringBuilder sb = new StringBuilder("Nodes LIST = (values are in HEXA)");
        if (this.bootedNodesNB == 0) {
            sb.append(name);
            sb.append(": No booted CANopen devices.");
        }
        for (int i=0; i < this.bootedNodesNB; i++) {
            sb.append("\n");
            if (nodes.get(i) == null) {
                sb.append("null node");
            } else sb.append(nodes.get(i).toString()) ;
        }
        return sb.toString();

    }
    
    
    /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper. 
     * Test if the command is valid before sending it to the can open stack.
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * @throws org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException
     * @throws org.lsst.ccs.bus.BadCommandException
     * 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Send a CanOpen command to the Can Bus.")
    public String sendCanOpen(String command) throws CanOpenCallTimeoutException, BadCommandException  {
        if (!this.tcpServerStarted) throw new BadCommandException(name + ": is not started, can't send CanOpen commands.");
        try {
          String[] words = command.split(",");
          String keyWord = words[0]; 
            switch (keyWord) {
                case "rsdo":
                    if (words.length != 4) throw new BadCommandException("rsdo command must have 3 parameters: nodeID, index, subindex");
                    break;
                case "wsdo":
                    if (words.length != 6) throw new BadCommandException("wsdo command must have 5 parameters: nodeID, index, subindex, size, data");
                    break;
                case "info":
                    if (words.length != 2) throw new BadCommandException("info command must have 1 parameter: nodeID");
                    break;
                case "sync":
                    if (words.length != 1) throw new BadCommandException("sync command takes no parameter");
                    break;
                case "srtr":
                    if (words.length < 2) throw new BadCommandException("srtr command takes at least one parameter (cobib)");
                    break;
                case "quit":
                    if (words.length != 1) throw new BadCommandException("quit command takes no parameter");
                    break;
                default:
                    throw new IllegalArgumentException(command);
            }
          return (String) call(getMyClientName(), command);
          
        }   catch (CWrapperNotConnected ex) {
            fcslog.error(name + ":CWrapper not connected-" + ex.toString());
            this.getSubsystem().raiseAlarm(name + " can't communicate with harwdare because CWrapper is not connected." + ex.toString());
            return ex.toString();
        }  
    }
      
    
    /**
     * build a Can Open wsdo Command that can understand the C wrapper.
     * exemple : wsdo,2,6411,01,2,3000
     */
    private static String buildWsdoCommand(String nodeID, String index, String subindex, String size, String data) {
        char sep = ',';
        String command = String.format("wsdo%1$s%2$s%1$s%3$s%1$s%4$s%1$s%5$s%1$s%6$s",sep,nodeID,index,subindex,size,data);
        return command;
    }
    
    /**
     * build a Can Open rsdo Command that can understand the C wrapper.
     * exemple : rsdo,1,1018,0
     */
    private static String buildRsdoCommand(String nodeID, String index, String subindex) {
        char sep = ',';
        String command = String.format("rsdo%1$s%2$s%1$s%3$s%1$s%4$s",sep,nodeID,index,subindex);
        return command;
    }

    /**
     * Write a SDO request and send it to the can open stack, then analyses the response
     * or throws an exception if the request failed.
     * The parameters are given in HEXA.
     * @param nodeID FORMAT=HEXA
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=HEXA
     * @param value FORMAT=HEXA
     * @return "OK" if OK
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public String writeSDO(String nodeID, String index, String subindex, String size, String value) throws SDORequestException  {
        try {
            String sdoResponseLine = sendCanOpen(buildWsdoCommand(nodeID,index,subindex,size,value));
            String[] words = sdoResponseLine.split(",");
            String command = words[0];
            String receivedNodeID = words[1];
            String errorCode = words[2];
            switch (errorCode) {
                case "0":
                    return "OK";
                case "-1":
                    fcslog.error("Wrong writeSDO command : " + sdoResponseLine);
                    throw new SDORequestException(name + ":Wrong writeSDO command : " + sdoResponseLine);
                default:
                    fcslog.error("writeSDO request was going wrong");
                    String error = String.format("%08d", Integer.parseInt(errorCode));
                    fcslog.info("ERROR=" + error);
                    String errorName = CanOpenErrorsTable.commErrorCodes.getProperty(error);
                    throw new SDORequestException(name + ":writeSDO request was going wrong", nodeID, errorCode, index, subindex, errorName);
            }
        } catch (BadCommandException ex) {
            SDORequestException err = new SDORequestException(name + ":write SDO request bad command : ", nodeID,"bad command", index, subindex, ex.getMessage());
            fcslog.error(String.format("%s %s for %s this should not happen.", "malformed SDO request : ",ex.getMessage(),nodeID));
            throw err;
        } catch (CanOpenCallTimeoutException ex) {
            SDORequestException err = new SDORequestException(name + ":write SDO request timeout expired : ", nodeID,"timeout expired", index, subindex, ex.getMessage());
            fcslog.error(String.format("%s %s is %s stopped ?", "SDO request timeout expired : ",ex.getMessage(),nodeID));
            throw err;
        }
       
    }
    
     /**
     * Command to be used by the end user at the console.
     * Write a SDO message and send it to the can open stack.
     * The parameters are given in the following format:
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=decimal
     * @param value FORMAT=decimal
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Send a CanOpen writeSDO command to the Can Bus.")
    public String writeSDO(int nodeID, String index, String subindex, int size, int value) throws SDORequestException, BadCommandException {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        if (size <0 || size > 4) 
            throw new IllegalArgumentException("size must be > 0 and < 4");
        return writeSDO(Integer.toHexString(nodeID),index,subindex,Integer.toHexString(size),Integer.toHexString(value));
    }
    
    
     /**
     * Read a SDO with the given index and subindex
     * RETURNS the value read in hexa (String) if no error occured
     * or returns the error code if a error was detected.
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public String readSDO(String nodeID, String index, String subindex) throws SDORequestException, ShortResponseToSDORequestException {
        try {
            String sdoLine = sendCanOpen(buildRsdoCommand(nodeID,index,subindex));
            String[] words = sdoLine.split(",");
            int responseLength = words.length;
            String msg;
            
            String command = words[0];
            String receivedNodeID = words[1];
            String errorCode = words[2];

           
            if (errorCode.equals("0") && (responseLength > 3)) {
                return words[3];
                
            } else if (errorCode.equals("0") && responseLength == 3) {
                msg=name + ":readSDO request received a too short response=" + sdoLine;
                fcslog.warning(msg);
                throw new ShortResponseToSDORequestException(msg);
                
            } else if  (errorCode.equals("-1")) {
                msg = name + ":wrong readSDO command : " + sdoLine;
                fcslog.error(msg);
                throw new SDORequestException(msg);

            } else {
                fcslog.error(name + ":readSDO request was going wrong, received response=" + sdoLine);
                String error = String.format("%08d", Integer.parseInt(errorCode));
                fcslog.info("ERROR=" + error);
                String errorName = CanOpenErrorsTable.commErrorCodes.getProperty(error);
                throw new SDORequestException("SDO request was going wrong", nodeID, errorCode, index, subindex, errorName);
            }
        } catch (BadCommandException ex) {
            SDORequestException err = new SDORequestException("read SDO request: ", nodeID, "bad command",index, subindex, ex.getMessage());
            fcslog.error(String.format("%s %s on node %s should not happen", "bad command : ",ex.getMessage(),nodeID));
            throw err;
        } catch (CanOpenCallTimeoutException ex) {
            SDORequestException err = new SDORequestException("read SDO request timeout expired : ", nodeID, "timeout expired",index, subindex, ex.getMessage());
            fcslog.error(String.format("%s %s is %s stopped ?", "SDO request timeout expired : ",ex.getMessage(),nodeID));
            throw err;
        }
       
    }
    
    /**
     * Command to be used by the end user at the console in engineering mode.
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=hexa
     * @param subindex FORMAT=hexa
     * @return the value sent by the can open stack if the read SDO request succeded.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Send a CanOpen readSDO command to the Can Bus.")
    public String readSDO(int nodeID, String index, String subindex) throws SDORequestException, BadCommandException, ShortResponseToSDORequestException  {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        return readSDO(Integer.toHexString(nodeID),index,subindex);
    }
    
    /**
     * This method sends a sync command to the can open stack and returns the reply.
     * example of sync reply we can receive :
     * sync,23_1=25a7,23_2=113,23_3=10,23_4=109,23_5=108,23_6=104,23_7=101,23_8=112
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    public PDOStorage readPDOs() throws FcsHardwareException, BadCommandException {
        try {
            String replyToSync = sendCanOpen("sync");           
            fcslog.finest(name + ": replyToSync=" + replyToSync);
            this.pdoStorage.updatePDOs(replyToSync);
            return this.pdoStorage;
            
        } catch (PDOBadResponseException ex) {
                        FcsHardwareException err = new FcsHardwareException(ex.toString());
            throw err;
            
        } catch (CanOpenCallTimeoutException ex) {
        
            FcsHardwareException err = new FcsHardwareException("Timeout expired when waiting for the reply to a sync command in readPDOs : "
                    + ex.toString());
            throw err;
        } 
    }
    
    /*for test*/
    public String displayReadPDOs() throws FcsHardwareException, BadCommandException {
        try {
            String replyToSync = sendCanOpen("sync");           
            fcslog.finest(name + ": replyToSync=" + replyToSync);
            this.pdoStorage.updatePDOs(replyToSync);
            return this.pdoStorage.toString();
            
        } catch (PDOBadResponseException ex) {
                        FcsHardwareException err = new FcsHardwareException(ex.toString());
            throw err;
            
        } catch (CanOpenCallTimeoutException ex) {
        
            FcsHardwareException err = new FcsHardwareException("Timeout expired when waiting for the reply to a sync command in readPDOs : "
                    + ex.toString());
            throw err;
        } 
    }

    /**
     * Identification of the hardware : we want to retrieve the information 
     * stored in the hardware of the booted nodes and update the array of nodes
     * with this information.
     * The Can Open node ID is a main information to retrieve.
     * @throws org.lsst.ccs.HardwareException
     */
    @Deprecated
    public void identifieHardware_old() throws HardwareException {
        fcslog.info(name + ":Identification of the hardware");
        for (int i=0; i < this.bootedNodesNB; i++ ) {

              if (!(nodes.get(i) == null)) {
                try {
                    fcslog.finest(name + ":Sending to can open command : info," + nodes.get(i).getNodeID());
                    Object result = sendCanOpen("info," + nodes.get(i).getNodeID());
                    fcslog.finest(name + ":Received on socket command = " + result);
                    String infoLine = (String) result;
                    String[] words = infoLine.split(",");
                    String command = words[0];
                    String nodeID = words[1];            
                    
                    if (command.equals("info") && nodeID.equals(nodes.get(i).getNodeID())) {
                        fcslog.debug(name + ":updating Node Info for node " + nodeID);
                        String type = words[2];
                        String vendor = words[3];
                        String productCode = words[4];
                        String revision = words[5];
                        String serialNB = words[6];
                        nodes.get(i).setNodeInfo(type, vendor, productCode, revision, serialNB);
                    } else {
                        fcslog.error(name + ":ERROR for command = " + command + "NodeID = " + nodeID);
                        //TODO throw an exception
                    }
                } catch (BadCommandException ex) {
                    String msg = name + " :Error in can open command : info," + nodes.get(i).getNodeID();
                    fcslog.error(msg);
                    throw new org.lsst.ccs.HardwareException(false,msg);
                } catch (CanOpenCallTimeoutException ex) {
                    fcslog.error(name + ":Timeout expired during hardware identifie process for node=" + nodes.get(i).getNodeID());
                    throw new org.lsst.ccs.HardwareException(false,ex);
                }
            }
        }
        fcslog.info(name + ":Hardware is identified");
        hardwareIdentified = true;
        fcslog.info(name + "=>" + listNodes());
    }
    
     /**
     * Identification of the hardware : we want to retrieve the information 
     * stored in the hardware of the booted nodes and update the array of nodes
     * with this information.
     * So for each booted node, we send a command : info,node_ID to the CWrapper
     * and then we update the nodes list with the information returned in the 
     * command result.
     * The serial number is a main information to retrieve.
     * @throws org.lsst.ccs.HardwareException
     */
    public void identifyHardware() throws HardwareException {
        fcslog.info(name + ":Identification of the hardware");
        for (int i=0; i < this.bootedNodesNB; i++ ) {

              if (!(nodes.get(i) == null) && (!nodes.get(i).isIdentified())) {
                try {
                    fcslog.debug(name + ":Sending to can open command : info," 
                            + nodes.get(i).getNodeID());
                    Object result = sendCanOpen("info," + nodes.get(i).getNodeID());
                    fcslog.debug(name + ":Received on socket command = " + result);
                    String infoLine = (String) result;
                    String[] words = infoLine.split(",");
                    String command = words[0];
                    String nodeID = words[1];            
                    
                    if (command.equals("info") && nodeID.equals(nodes.get(i).getNodeID())) {
                        fcslog.debug(name + ":updating Node Info for node " + nodeID);
                        String type = words[2];
                        String vendor = words[3];
                        String productCode = words[4];
                        String revision = words[5];
                        String serialNB = words[6];
                        nodes.get(i).setNodeInfo(type, vendor, productCode, revision, serialNB);
                    } else {
                        fcslog.error(name + ":ERROR for command = " + command 
                                + "NodeID = " + nodeID);
                        //TODO throw an exception
                    }
                    
                } catch (BadCommandException ex) {
                    String msg = name + " :Error in can open command : info," 
                            + nodes.get(i).getNodeID();
                    fcslog.error(msg);
                    throw new HardwareException(false,msg);
                    
                } catch (CanOpenCallTimeoutException ex) {
                    fcslog.error(name + ":Timeout expired during hardware identifie process for node=" + nodes.get(i).getNodeID());
                    throw new HardwareException(false,ex);
                }
            }
        }
        
        if (this.bootedNodesNB == this.expectedNodesNB) {
            fcslog.info(name + ":Hardware is identified");
            hardwareIdentified = true;
        }
        
        publishData();
    }

    /**
     * This method returns true if : all the hardware items are booted and identified 
     * and the hardware have the node ID expected within the configuration
     * and the hardware is initialized.
     * @return a boolean
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="Return true if all hardware is booted and identified.")
    public boolean isHardwareReady() {
        fcslog.info(name + ":hardwareBootProcessEnded=" + hardwareBootProcessEnded);
        fcslog.info(name + ":hardwareIdentified=" + hardwareIdentified);
        fcslog.info(name + ":canOpenNodeNumbersOK=" + this.canOpenNodeNumbersOK);
        fcslog.info(name + ":hardwareIDError=" + this.hardwareIDError);
        return this.hardwareBootProcessEnded && this.hardwareIdentified 
                && this.canOpenNodeNumbersOK && !this.hardwareIDError;
    }

    /**
     * This method returns the can open detected on the can open network 
     * for a piece of hardware identified by its serial number.
     * It can return NULL if the Serial Number is not on the can open network.
     * @param aSerialNB
     * @return 
     */    
    public CanOpenNode getDetectedNodeForSerialNumber(String aSerialNB) {
        CanOpenNode detectedNode = null;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            if (nodes.get(i).getSerialNB().equals(aSerialNB)) {
                detectedNode = nodes.get(i);
            }
        }
        return detectedNode;
    }

    @Override
    void processBootMessage(String nodeID)  {
        
        /*The node is already in the table of nodes*/
        if (isBooted(nodeID)) {
            if (hardwareBootProcessEnded) {
                    /*the boot process is ended: something weird has just happened*/ 
                    String msg = " An unsynchronous boot message is arrived for node ID = " + nodeID;
                    fcslog.error(msg);
                    this.getSubsystem().raiseAlarm(name + msg + " Possible power failure for node : " + nodeID);             
            }
        
        /*the node is not already in the table of nodes : we have to add it*/
        } else {

            //TODO : if the number of nodes we found is greater than the expectedNodesNB, throw an exception.                      
            nodes.add(new CanOpenNode(nodeID));
            bootedNodesNB = nodes.size();
            fcslog.info("Node " + nodeID + " added");
            fcslog.info("Number of booted devices=" + bootedNodesNB);
            fcslog.info("Number of expected devices="+getExpectedNodesNB());
            if (hardwareBootProcessEnded) {
                    /*the boot process is ended: the device has been powered on ?*/ 
                    String msg = name + ": boot message is arrived for node ID = " + nodeID;
                    fcslog.warning(msg);
                    this.getSubsystem().raiseAlarm(name + msg + " Possible power failure for node : " + nodeID);             
            }
        }
    }
    
    @Override
    void processEmcyMessage(String message) {
        String[] words = message.split(",");
        String nodeID = words[1];
        String deviceErrorCode = words[2];
        String errReg = words[3];
        String deviceErrorName = deviceErrorCode + "=device error code not found in deviceErrorCodes";
        String errorRegisterName = errReg + "=error register code not found in errorRegisterCodes";
        deviceErrorName = CanOpenErrorsTable.deviceErrorCodes.getProperty(deviceErrorCode);
        errorRegisterName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errReg);
        String deviceName = this.getNodeName(nodeID);
        setChanged();
        EmergencyMessage emcyMsg = new EmergencyMessage(name,nodeID,deviceName,deviceErrorCode,
                deviceErrorName,errReg,errorRegisterName);
        this.notifyObservers(new ValueUpdate(name,emcyMsg));
        fcslog.error(name + ":Received EMERGENCY message for nodeID=" + message);
        fcslog.error(name + ":EmergencyMessage=" + emcyMsg.toString());
        
        try {
            switch (errReg) {
                case "00":
                    log.warning(name + " received EMERGENCY message with no error. "
                            + "Is it after a faultReset ?");
                    break;

                case "01":
                    throw new CanOpenGenericException(deviceErrorName,nodeID, errReg);                       
                case "04":
                    throw new CanOpenVoltageException(deviceErrorName,nodeID, errReg);
                case "08":
                    throw new CanOpenTemperatureException(deviceErrorName,nodeID, errReg);
                case "10":
                    throw new CanOpenCommunicationException(deviceErrorName,nodeID, errReg);
                case "20":
                    throw new CanOpenDeviceException(deviceErrorName,nodeID, errReg);
                case "80":
                    throw new CanOpenMotionException(deviceErrorName,nodeID, errReg);
                default:
                    throw new CanOpenException(deviceErrorName,nodeID, errReg);
            }

        } catch (CanOpenException ex) {
            
            String nodeIDToString = String.format("NodeID %s (%02d in decimal)",nodeID,Integer.parseInt(nodeID, 16));
            this.getSubsystem().raiseAlarm(name + ":" + nodeIDToString + ":" + ex.toString());
            
        } finally {
            //publishData();
        }
    }
    

    
    @Override
    void processUnknownCommand(String message) {
        String[] words = message.split(",");
        String command = words[1];
        String errorMessage = "Unknown command:" + command;
        fcslog.error(errorMessage);
        
    }
    
    /**
     * Check if the list of the booted CANopen devices on the CANbus is identical to
     * the hardware list we have in the description.
     * @throws HardwareException 
     */
    public void checkCanOpenNodes() throws HardwareException {
//        if (!(isReady(this.myClientName) && isHardwareIdentified()))
//                    throw new BadCommandException(name + ":Hardware not booted or not identified");
        
        if (!(isReady(this.myClientName)))
                    throw new HardwareException(true,name + ": not yet connected with Hardware CWrapper.");
        
        fcslog.info(name + ":CHECKING HARDWARE CONFIGURATION ");
        
        
        if (this.bootedNodesNB == 0) {
            String msg = String.format(" (hardwareBootTimeout=%02d,bootedNodesNB=%02d,expectedNodesNB=%02d)",this.hardwareBootTimeout, this.bootedNodesNB, this.expectedNodesNB);
            fcslog.error(name + ":NO HARDWARE DETECTED - POWER FAILURE ?" + msg);
            throw new HardwareException(true,name + ":NO HARDWARE DETECTED - POWER FAILURE ?" + msg);      
        
        } else if (this.bootedNodesNB == this.expectedNodesNB) {
            this.canOpenNodeNumbersOK = true;
        }
        
        errorMessageSB = new StringBuilder(name);
        hardwareIDError = false;

        for (PieceOfHardware hardware : this.hardwareList)  {
            fcslog.info("ABOUT TO CHECK: " + hardware.getName() + " NODE_ID=" + hardware.getNodeID());
            checkHardwareID(hardware);
            fcslog.info(hardware.getName() + " BOOTED AND SERIAL NUMBER OK=" + hardware.isBooted());
        }

        //if a piece of hardware is missing, checkHardwareID has already detected it. 
        if (this.bootedNodesNB > this.expectedNodesNB) {
            String msg = String.format(" (hardwareBootTimeout=%02d,bootedNodesNB=%02d,expectedNodesNB=%02d)",this.hardwareBootTimeout, this.bootedNodesNB, this.expectedNodesNB);
            fcslog.error(name + ":MORE HARDWARE DETECTED THAN EXPECTED - CONFIGURATION ISSUE ?" + msg);
            errorMessageSB.append(msg);
            this.hardwareIDError = true;
        }
        
        if (hardwareIDError) throw 
                new HardwareException(true,errorMessageSB.toString());
        publishData();       
    }
    

    
        
     /**
     * When the hardware is booted and we have retrieve the information (serial number), 
     * we want to check for each piece of hardware that we have the good node ID
     * and the good serial number: so we compare the serial number found on the 
     * CANopen bus with the serial number in the configuration file.
     * @param pieceOfHardware 
     * @throws org.lsst.ccs.HardwareException 
     */
    public void checkHardwareID(PieceOfHardware pieceOfHardware) 
            throws HardwareException  {
        String msg;
//        if (!(this.hardwareBootProcessEnded && this.hardwareIdentified))
//                    throw new HardwareException(false,name + ":Hardware not booted or not identified");
        if (bootedNodesNB == 0) {
            throw new HardwareException(true,name + "NO HARDWARE BOOTED - nothing to check");
        }
               
        fcslog.info(name + " Checking nodeID and serial number for " + pieceOfHardware.getName());
              
        boolean serialNBFound = false;
        boolean nodeIDFound = false;
        
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            
            if (!(nodes.get(i) == null) && (nodes.get(i).getNodeID().equalsIgnoreCase(pieceOfHardware.getNodeID()))) {
                //the node id of this piece of hardware has been found in the nodes list.
                nodeIDFound = true;
 
                if (pieceOfHardware.getSerialNB().equalsIgnoreCase(nodes.get(i).getSerialNB())) {
                    //and it's the same serial ID : every thing is OK  
                    pieceOfHardware.setBooted(true);
                    this.publishHardwareData(pieceOfHardware);
                    fcslog.info(pieceOfHardware.getName() + " is booted  :" + pieceOfHardware.toString());
                } else {
                    //bad serial number for this pieceOfHardware
                    pieceOfHardware.setBooted(false);
                    msg = pieceOfHardware.getName() + " has a wrong serial number. Serial number found="
                            + nodes.get(i).getSerialNB() + ", should be=" + pieceOfHardware.getSerialNB();
                    fcslog.error(msg);
                    errorMessageSB.append(msg);
                    this.hardwareIDError = true;
                                   
                }
            }
            
        }
        
        if (!nodeIDFound) { //perhaps this hardware has a wrong node ID
            pieceOfHardware.setBooted(false);
            for (int i=0; i < this.bootedNodesNB; i++ ) {
                if (!(nodes.get(i) == null) 
                        &&  (nodes.get(i).getSerialNB().equalsIgnoreCase(pieceOfHardware.getSerialNB()))) {  
                    //the serial number of this piece of hardware has been found in the nodes list
                    //with another node ID.
                    serialNBFound = true;
                    msg = String.format(name + ":ERROR in CONFIGURATION for HARDWARE %s"
                            + "- %s serial number=%s - node ID found for this serial number=",
                        pieceOfHardware.getName(),pieceOfHardware.getSerialNB(),nodes.get(i).getNodeID());
                        fcslog.error(msg);
                        errorMessageSB.append(msg);
                        this.hardwareIDError = true;
                }
            }
            if (!serialNBFound) {
                msg = String.format(":HARDWARE NOT DETECTED - "
                    + "Possible power failure for node ID %s ? - "
                    + "(hardware:%s,serial number:%s)",
                    pieceOfHardware.getNodeID(),pieceOfHardware.getName(),pieceOfHardware.getSerialNB());
                fcslog.error(msg);
                errorMessageSB.append(msg);
                this.hardwareIDError = true;
            } 
            
        } 
    }
    
    /**
     * 
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return a String with the list of hardware.")
    public String listHardware() {
        StringBuilder sb = new StringBuilder("List of hardware:\n");
        for (PieceOfHardware pieceOfHardware : this.hardwareList)  {
            sb.append(pieceOfHardware.toString());
            sb.append("\n");
        }
        return sb.toString();      
    }
    
    /**
     * Configure a node as a hearbeat producer.
     * Every heartbeat_time milliseconds this node will produce a message like:
     * 0x700+nodeID: 05 - 1340268404.980000 
     * @param nodeID node ID in hexa
     * @param heartbeat_time FORMAT=hexa UNIT=milliseconds
     * @return error code returned by writeSDO
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws java.util.concurrent.TimeoutException
     * @throws org.lsst.ccs.bus.BadCommandException
     */
    public String configAsHeartbeatProducer(String nodeID, String heartbeat_time) throws SDORequestException, TimeoutException, BadCommandException {
        return writeSDO(nodeID,"1017","0","2",heartbeat_time);
    }
    
    /**
     * Command to be used by the end users.
     * Configure a node as a heartbeat producer.
     * Parameters given in decimal format.
     * @param nodeID FORMAT=decimal
     * @param heartbeat_time FORMAT=decimal UNIT=milliseconds
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws java.util.concurrent.TimeoutException
     * @throws org.lsst.ccs.bus.BadCommandException
     */
    public String configAsHeartbeatProducer(int nodeID, int heartbeat_time) throws SDORequestException, TimeoutException, BadCommandException {
        
        if (heartbeat_time < 0 || heartbeat_time > 65025)
            throw new IllegalArgumentException("heartbeat_time is coded on 2 bytes"
                    + " can't be > 65535");
        return configAsHeartbeatProducer(Integer.toHexString(nodeID), Integer.toHexString(heartbeat_time));
    }
    
     /**
     * Configure a node as a hearbeat consumer.
     * Every heartbeat_time milliseconds this node will produce a message like:
     * 0x700+nodeID: 05 - 1340268404.980000 
     * @param nodeID node ID in decimal
     * @param producerNodeID
     * @param heartbeat_time FORMAT=decimal UNIT=milliseconds
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws java.util.concurrent.TimeoutException
     * @throws org.lsst.ccs.bus.BadCommandException
     */
    public String configAsHeartbeatConsumer(int nodeID, int producerNodeID, int heartbeat_time) throws SDORequestException, TimeoutException, BadCommandException {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        if (producerNodeID <0 || producerNodeID > 127) 
            throw new IllegalArgumentException("producerNodeID must be > 0 and < 128");
        if (heartbeat_time < 0 || heartbeat_time > 65025)
            throw new IllegalArgumentException("heartbeat_time is coded on 2 bytes"
                    + " can't be > 65535");
        String value = String.format("%04x",producerNodeID) + String.format("%04x",heartbeat_time);
        return writeSDO(Integer.toHexString(nodeID),"1016","1","4",value);
    }   
    
     
     /**
     * Publish Data on status bus for trending data base and GUIs.
     */
     public void publishData() {
        for (PieceOfHardware device : this.getHardwareList()) {
            publishHardwareData(device);
        }       
     }
     
     /**
     * Publish Data on status bus for trending data base and GUIs.
     * @param device
     */
     public void publishHardwareData(PieceOfHardware device) {
        StatusDataPublishedByHardware status = 
                FcsUtils.createStatusDataPublishedByHardware(device);
        log.debug(name + ":publishHardwareData is publishing:" + status.toString());
        this.publish("tcpProxy", status);
     }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append("/myClientName=");sb.append(this.myClientName);
        return sb.toString();
    }
    
}
