/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.Map.Entry;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * @author virieux
 */
public class CanOpenDevice extends Module implements PieceOfHardware{
    
    protected static final Logger fcslog = FcsUtils.log;
    
    CanOpenProxy tcpProxy;
    BridgeToHardware bridge;
    
    String serialNB;
    String nodeID;
    boolean booted;
    boolean initialized;

    public CanOpenDevice(String aName, int aTickMillis,  String nodeID, String serialNB) {
        super(aName, aTickMillis);
        this.nodeID = nodeID;
        this.serialNB = serialNB;
        this.booted=false;
        this.initialized=false;
    }

    @Override
    public void setBooted(boolean booted) {
        this.booted = booted;
    }
    
    
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns this CANopen device's serial number.")
    @Override
    public String getSerialNB() {
        return this.serialNB;
    }

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns this CANopen device's node ID.")
    @Override
    public String getNodeID() {
        return this.nodeID;
    }

    @Override
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if this CANopen node is booted.")
    public boolean isBooted() {
        return booted;
    }
    
    

    @Override
    public boolean isInitialized() {
        return this.initialized;
    }
    
    @Override
    public void initModule() {
// // doesn't work because the devices for the loader has another tcpProxy
//        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy"); 
        
        Entry<String,Object> entry = this.getParent();
        bridge = (BridgeToHardware) entry.getValue();
        tcpProxy = (CanOpenProxy) bridge.getTcpProxy();
        this.initialized = false;
    }

    /**
     * This methods has to be overridden if there is some initialization to do for the device.
     * @return a message
     * @throws Exception 
     */
    @Override
    public String initializeHardware() throws FcsHardwareException {
        this.initialized = true;
        return getName() + ": no hardware initialization for this device to be done.";
    }
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Write on the Can Open Device with a command wsdo.")
    public String writeSDO(String index, String subindex,String length, String newValue) throws SDORequestException {
        return tcpProxy.writeSDO(this.nodeID,index,subindex,length,newValue);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING3, 
            description="Reads the Can Open Device with a command rsdo.")
    public String readSDO(String index, String subindex) throws SDORequestException, 
            ShortResponseToSDORequestException {
        return tcpProxy.readSDO(this.nodeID, index, subindex);
    }
    
        
    
    /**************************************************************************/
    /*Methods to read or display the errors that have occured on the device   */
    /*The errors are stored in 2 places :
    /* error register at index 1001 
    /* error history at index 1003               
    /* The error register and the error history 
    /**************************************************************************/
    
        
    /**
     * At index 1001 there deviceErrorFile an error register for the device. 
     * The device maps internal errors in this byte.
     * @return error code in Hexa
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING3, 
            description="Reads Error Register on the controller - in hexa (index 1001).")
    public String readErrorRegister() throws SDORequestException, ShortResponseToSDORequestException {
        String error = (String) readSDO("1001", "0");
        return String.format("%02x", Integer.parseInt(error));
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING3, 
            description="Display Error Register on the controller (index 1001).")
    public String displayErrorRegister() throws SDORequestException, ShortResponseToSDORequestException {
        String errorInHexa = readErrorRegister();      
        String errorName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errorInHexa);
        fcslog.debug("error register=" + errorInHexa + " error name=" + errorName);
        return errorName;
    }
    
    /**
     * The error history holds errors that have occurred on the device and 
     * have been signalled via the emergency object.
     * This methods reads the Maxon Motor Error History and returns an array of 
     * error code.
     * @return errors : a list of device error code
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException
     */
    public String[] readErrorHistory() throws SDORequestException, ShortResponseToSDORequestException {              
        int numberOfErrors = readNumberOfErrors();
        String[] errors = new String[numberOfErrors];
        for (int i=0; i < numberOfErrors;i++) {
            String subindex = Integer.toHexString(i+1);
            errors[i] = (String) readSDO("1003", subindex);
        }
        return errors;
    }
    

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Display Error Register on the controller (index 1001).")
    public String displayErrorHistory() throws SDORequestException, ShortResponseToSDORequestException {
        String[] errorHistory = this.readErrorHistory();
        StringBuilder sb = new StringBuilder("Error history contains " + errorHistory.length + " errors.");
        if (!(errorHistory.length == 0)) {
            sb.append("\nList of errors in history,"
                    + "the newest is the first, the oldest the last :\n");
            for (int ix=0; ix < errorHistory.length; ix++) {
                sb.append("Error code (in hexa)=");sb.append(errorHistory[ix]);
                sb.append("/ error name= ");
                sb.append(CanOpenErrorsTable.deviceErrorCodes.getProperty(errorHistory[ix].toUpperCase()));
                sb.append("\n");
            }
        }
        return sb.toString();
    }
    

    
    /**
     * Extract from the Maxon Motor firmware specification :
     * Contains the number of actual errors that are recorded in the array starting 
     * at subindex 1. Writing a “0” deletes the error history (empties the array). 
     * Values higher then “0” (zero) are not allowed to write."
     * 
     * Read and returns the number of errors registred in the Error History 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Read and returns the number of errors registred in the Error History.")
    public int readNumberOfErrors() throws SDORequestException, ShortResponseToSDORequestException {
        String errorNBinHEXA = (String) readSDO("1003", "0");
        return Integer.parseInt(errorNBinHEXA, 16);
    }
    
     /**
     * For the GUI or to display at the console : 
     * return a list of String with the error code and the error name.
     * 
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException 
     */
    public String[] getErrorHistoryNames() throws SDORequestException, 
            ShortResponseToSDORequestException {       
        
            String[] errorHistoryList = this.readErrorHistory();
            String[] errorHistoryNames = new String[errorHistoryList.length];
            if (!(errorHistoryList.length == 0)) {
                for (int i=0; i<errorHistoryList.length; i++) {
                    
                    String errorCode = errorHistoryList[i];
                    fcslog.debug("errorCode="+errorCode);
                    String errorName = CanOpenErrorsTable.deviceErrorCodes.getProperty(errorCode.toUpperCase());
                    fcslog.debug("errorName="+errorName);
                    errorHistoryNames[i] = errorCode + "=" + errorName;
                }
            }
            return errorHistoryNames;  
                    
    }
    
    /**
     * This method saves the parameters in the controller memory.
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
    //TODO move this method in CanOpenDevice
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, 
            description="This method saves the parameters in the controller memory.")    
    public Object saveParameters() throws SDORequestException {
        return this.writeSDO("1010", "1", "4", "65766173");
    }
    
    
    @Override
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Print CANopen device.")
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());       
        sb.append("/SerialNumber=");
        sb.append(this.serialNB);
        sb.append("/NodeID_in_hexa=");
        sb.append(this.nodeID);
        int x = Integer.parseInt(this.nodeID,16);
        sb.append("/NodeID_in_decimal=");sb.append(x);sb.append("/");
        // because the configuration system can call toString before the object is completely built
        // for tracing purpose.
        if (tcpProxy == null ) return sb.toString();
        if (isBooted())  {
            sb.append("BOOTED/");
            
        } else {
            sb.append("NOT YET BOOTED/");
        }       
        return sb.toString();
    }


    
    
}


        