/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.common;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This represents a Mechanical Item which can be moved by a motor.
 * Generaly we want to execute an action on this mechanical item :
 * - start the action,
 * - read sensors to know if the action is completed, 
 * - wait for the end of the action. 
 * This module provides a general way to start an action and wait until the action is completed.
 * It factorises some code that otherwise would have to be duplicated in each Module that needs such fonctionnalities.
 * @author virieux
 */
public abstract class MobileItemModule extends Module implements HardwareController {

    protected static final Logger fcslog = FcsUtils.log;
    protected volatile boolean hasToWaitForEndOfAction = false;
    protected volatile MobileItemAction currentAction;
    protected volatile boolean moving = false;
    //Used because we have to wait for the update from the sensors to know if the action is completed
    protected final Lock lock = new ReentrantLock();
    protected final Condition motionCompleted = lock.newCondition();

    ScheduledFuture<?> readSensorsHandle;
    final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    //prendre un Scheduler qui peut me permettre de donner un nom au Thread.
    
    
    
    
    
    
    protected AtomicBoolean haltRequired;
    protected AtomicBoolean stopRequired;

    public MobileItemModule(String moduleName, int aTickMillis) {
        super(moduleName, aTickMillis);
        haltRequired = new AtomicBoolean();
        haltRequired.set(false);
        stopRequired = new AtomicBoolean();
        stopRequired.set(false);
    }

    public boolean isMoving() {
        return moving;
    }

    //for the simulation
    public AtomicBoolean getHaltRequired() {
        return haltRequired;
    }
    
    

    /**
     * This stops the reading of the sensors.
     * 
     */
    private void cancelReadingSensors() {
        lock.lock();
        try {
            fcslog.debug(getName() + " => stop reading sensors");
            motionCompleted.signal();
        } finally {
            lock.unlock();
        }
        this.readSensorsHandle.cancel(true);
        fcslog.debug(getName() + " => readingSensors canceled");
    }
    
    //TODO why do we need that here ?
    public abstract boolean isHardwareReady();

    /*
     * This method must returns true if the action is completed, false otherwise.
     * It's abstract because it'up to the sub-classes to know when the action is completed.
     */
    public abstract boolean isActionCompleted(MobileItemAction action);
    
    public abstract void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception;
    
    //TODO
//    public abstract void updateStateWithSensors() throws HardwareException;
    
    public abstract void startAction(MobileItemAction action) 
            throws BadCommandException,ErrorInCommandExecutionException, FcsHardwareException;
    
    public abstract void abortAction(MobileItemAction action, long delay) 
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException;
 
    public abstract void quickStopAction(MobileItemAction action, long delay) 
                throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException;
    
    public abstract void postAction(MobileItemAction action) 
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException;
   
    public abstract void publishData();
    
    /**
     * This executes an action which moves the MobileItem and waits for the end od this action.
     * The MobileItemAction is an ENUM from the fcs.bus package.
     * @param action, timeout
     * @param timeoutForAction
     * @return 
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException if the action can't be completed 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException 
     */
    protected String executeAction(MobileItemAction action, long timeoutForAction) 
            throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        
        if (!isHardwareReady()) throw new FcsHardwareException(name 
                + ": hardware is not ready to execute ACTION commands." );
        
        lock.lock();
               
        try {
            this.hasToWaitForEndOfAction = true;
            
            this.currentAction = action;
            
            this.moving = true;
            
            fcslog.debug(name, "STARTING ACTION:" + action.toString());
            
            startAction(action);
            
            readSensorsUntilActionIsCompleted(action, System.currentTimeMillis(), timeoutForAction);
            
            waitForEndOfAction(action);
            
            
            //At this stage, the action can be completed with success or a 
            //HALT signal has been received.
            if (haltRequired.get()) {
                abortAction(action,0);
                moving = false;
                return name + "=> Received an ABORT command for action :" + action.toString();
                
            } else if (stopRequired.get()) {
                quickStopAction(action,0);
                moving = false;
                return name + "=> Received a STOP command for action :" + action.toString();
            
            } else if (isActionCompleted(action)) {
                moving = false;
                postAction(action);
                return name + " " + action.doneString();
                
            } else {
                throw new ErrorInCommandExecutionException(name + "=>" +  action.getFailureMsg());
            }
            
        } finally {

            fcslog.debug(name + ": finally in executeAction.");
            abortAction(action,0);
            this.moving = false;
            this.hasToWaitForEndOfAction = false;
            this.motionCompleted.signal();           
            haltRequired.set(false);
            stopRequired.set(false);
            lock.unlock();
            publishData();
        }
    }
    
    /**
     * Start reading the sensors at a fixed rate (scheduler.scheduleAtFixedRate) 
     * until the action is completed or we received a halt signal or the timeout 
     * for this action is past.
     * @param action
     * @param beginTime
     * 
     * @param timeout 
     */
    public void readSensorsUntilActionIsCompleted(final MobileItemAction action, 
            final long beginTime, final long timeout) {
        
        final Runnable readSensors;
        readSensors = new Runnable() {
            
            @Override
            public void run() {

                try {
                    fcslog.info(name + " : reading sensors");
                    long duration = System.currentTimeMillis() - beginTime;
                    
                    updateStateWithSensorsToCheckIfActionIsCompleted();
                    
                    boolean actionCompleted = isActionCompleted(action);
                    
                    //we got a HALT signal, we have to stop reading the sensors.
                    if (haltRequired.get()||stopRequired.get()) {
                        hasToWaitForEndOfAction = false;
                        fcslog.info(action.toString() + " ABORT or STOP REQUESTED FOR ACTION "
                                + "BY ABORT or STOP COMMAND");
                        cancelReadingSensors();
                        
                                      
                    //action is completed with success.
                    } else if (actionCompleted) {
                        hasToWaitForEndOfAction = false;
                        fcslog.info(action.toString() + " ACTION COMPLETED");
                        cancelReadingSensors();
                        
                    //action is not completed but timeout is over.
                    } else if (duration >= timeout) {
                        hasToWaitForEndOfAction = false;
                        fcslog.info(action.toString() + " ACTION NOT COMPLETED during allocated time");
                        cancelReadingSensors();
                        throw new FcsHardwareException(action.toString()
                                + " exceeded timeout for this task: duration=" + duration + ",timeout=" + timeout);
                    
                    //action is not completed, we have to continue reading sensors.
                    } else {                       
                        fcslog.info(action.toString() + " not completed.....");
                        fcslog.info(action.name() + "/duration=" + duration);                       
                                                
                    } 
                    
                } catch (Exception ex) {
                   
                    hasToWaitForEndOfAction = false;
                    fcslog.error("ERROR during action:" + ex.getMessage());
                    cancelReadingSensors();
                    getSubsystem().raiseAlarm(ex.toString());                    
                }
            }
        };
        this.readSensorsHandle = scheduler.scheduleAtFixedRate(readSensors, 500, 500, TimeUnit.MILLISECONDS);
    }

    /**
     * This method waits until the action is completed.
     * This methods is called by executeAction and has already acquired the lock.
     */
    private void waitForEndOfAction(MobileItemAction action) {
            while (hasToWaitForEndOfAction) {
                try {
                    fcslog.info(name +  " waiting for end of " + action.toString());
                    motionCompleted.await();
                } catch (InterruptedException ex) {
                    fcslog.info(name + ": InterruptedException received=" + ex.toString());
                    break;
                }
            }
            fcslog.info(name + " STOP WAITING FOR END OF ACTION");
    }
    
    private void halt(MobileItemAction action, long delay) {
        if (action == null) {
            fcslog.warning(name + ": no current action running => nothing to abort.");
            return;
        }
        try {
            fcslog.debug(name + ": ABORTING ACTION " + action.toString() + "within delay=" + delay);
            abortAction(action,delay);
            moving = false;
            
        } catch (BadCommandException | ErrorInCommandExecutionException | FcsHardwareException ex) {
            fcslog.error(name + " couldn't abort action " + action.toString());
            getSubsystem().raiseAlarm(ex.toString());
        }        
    }
    
    private void quickStop(MobileItemAction action, long delay) {
        if (action == null) {
            fcslog.warning(name + ": no current action running => nothing to stop.");
            return;
        }
        try {
            fcslog.debug(name + ": QUICKSTOP for" + action.toString() + "within delay=" + delay);
            quickStopAction(action,delay);
            moving = false;
            
        } catch (BadCommandException | ErrorInCommandExecutionException | FcsHardwareException ex) {
            fcslog.error(name + " couldn't do stop action " + action.toString());
            getSubsystem().raiseAlarm(ex.toString());
        }        
    }
    
    /**
     * just to avoid taping "sendSignal HALT"
     * @param delay 
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, 
            description="Abort the running action within a delay.")
    public void abort(long delay) throws BadCommandException {
            sendSignalWithTimeLimit("HALT",delay);
    }
    
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, 
            description="Abort the running action.")
    public void abort() throws BadCommandException {
        abort(0);
    }
    
    /**
     * just to avoid taping "sendSignal STOP"
     * @param delay
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, 
            description="Stop the running action with a quickstop.")
    public void stop(long delay) throws BadCommandException {
            sendSignalWithTimeLimit("STOP",delay);
    }
    
    /**
     * just to avoid taping "sendSignal STOP"
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, 
            description="Stop the running action with a quickstop.")
    public void stop() throws BadCommandException {
        stop(0);
    }
  
     /**
     * just to avoid taping "sendSignal STOP"
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, 
            description="Stop the running action with a quickstop.")
    public void quickstop() throws BadCommandException {
        stop(0);
    }
   
    

    

    
    /**
     * This shutdowns the scheduler.
     * This is executed during CLOSING phase when the MobileItemMobile is stopped.
     * cf checkStopped
     */
    @Override
    public void shutdownNow() {
        super.shutdownNow();
        scheduler.shutdown();
    }
    
    @Override
    public TreeWalkerDiag signal(Signal signal) {
        SignalLevel sl = signal.getLevel();
        fcslog.info(sl.toString());

        if (!this.moving)  {
            fcslog.warning(name + " is not moving; nothing to stop.");
            
        } else {

            switch(signal.getLevel()) {
                case HALT :
                    fcslog.info(getName() + " HALT required");
                    this.haltRequired.set(true);
                    halt(currentAction,signal.getExpectedMaxDelay());
                    break;
                case STOP :
                    fcslog.info(getName() + " STOP required");
                    this.stopRequired.set(true);
                    quickStop(currentAction,signal.getExpectedMaxDelay());
                    break;
                case INTERRUPT1 :
                    break;
                case INTERRUPT2 :
                    break;
                case RE_START :
                    this.haltRequired.set(false);
                    break;
                    
            }
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        //TODO
        //updateStateWithSensors();
        return TreeWalkerDiag.GO;
    }

    @Override
    public void checkStarted() throws HardwareException {
    }

    @Override
    public void checkStopped() throws HardwareException {
        if (this.isMoving()) throw new HardwareException(false,name 
                + " is moving, has to be stopped before a shutdown.");
    }

    
 
    





    
}
