/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.Sensor;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This class represents a 14bits sensor : 
 * the value returned by the sensor is 
 * coded with 14 bits, so its value is between 0 and 32767.
 * @author virieux
 */
public abstract class Sensor14bits extends Module implements Sensor {
    
   protected static final Logger fcslog = FcsUtils.log;
   protected int value;   
   protected final static int minValue = 0;
   protected final static int maxValue = 32767; //because it's on 16 bits
   
   //updatingValue is true while we are reading the value sent by the Sensor
   //this could take some time
   volatile boolean updatingValue = false;
   final Lock lock = new ReentrantLock();
   final Condition valueUpdated = lock.newCondition();

    public Sensor14bits(String moduleName, int aTickMillis) {
        super(moduleName, aTickMillis);
    }
   

   
    @Override
   public void initModule() {
       this.updatingValue = false;
       this.value=minValue;
   }
   
  /**
   * This method returns value if the sensor is not updating otherwise it waits
   * until the update is completed and returns the new read value from the hardware.
   * @return value  
   */
    @Override
   public int getValue() {
       lock.lock();
       try {
           while (updatingValue) {
                try {
                    //TODO put a timeout
                    valueUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(getName() + ex.getMessage());
      
                }
           }
           return value;
       } finally {
           lock.unlock();
       }
   }
   
   //for Spring
   public void setValue(int aValue) {
       this.value = aValue;
   }


    
    
       
   /**
   * This method updates the sensor value : first it reads a new value from the 
     * hardware or from the simulated sensor,
     * then it checks if the value is in the range between minimal value and 
     * maximal value. If it's ok it updates the sensor value with the new value, 
     * otherwise it throws an exception.
   * 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException
   */
    @Override
   public void updateValue() throws SensorValueOutOfRangeException, FcsHardwareException {

       lock.lock();
            
       try {
           updatingValue = true;
           int newValue = readNewValue();
           if ((newValue < minValue) || (newValue > maxValue)) {
                throw new SensorValueOutOfRangeException("ERROR in Sensor14bits during updateValue for ", this.getName(), Sensor14bits.minValue, Sensor14bits.maxValue, newValue);
           }
           this.value = newValue;

       } finally {

           updatingValue = false;
           valueUpdated.signal();
           lock.unlock();
       }
   }
    
    
   /**
     * This methods update the field value in extracting it from the parameter sensorsValues
     * which represents all the values read for an ADC.
     * @param sensorsValues
     * @return the value for this sensor.
     */
   //abstract public int updateValue(HashMap<String, String> sensorsValues);
    abstract public int updateValue(PDOStorage pdoStorage);
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns the value of the sensor stored in FCS memory.")
    public String printValue() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/value=");sb.append(this.value);
        return sb.toString();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/returned value=");
        sb.append(Integer.toString(value));
        return sb.toString();
    }

    
                 
}
