/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.CurrentCommandContext;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is model for the clamps mechanism in the loader.
 * 4 hooks are used for the clamping of the filter in the loader.
 * This hooks are moved all together by the hooksController.
 * 
 * @author virieux
 */
public class LoaderClampModule extends MobileItemModule {
    

    private final LoaderHookModule hook1;
    private final LoaderHookModule hook2;
    private final LoaderHookModule hook3;
    private final LoaderHookModule hook4;
    private EPOSController hooksController;
    private LoaderModule loader;
    
    
    FcsEnumerations.LockStatus lockStatus;
   
    
    long timeoutForClosingHooks;
    long timeoutForClampingHooks;
    long timeoutForOpeningHooks;
    long timeoutForGoingToHomePosition;
    
    private volatile boolean initialized;
    
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    private int position;
    int minPosition;// for GUI
    int maxPosition;// for GUI
    int minCurrent;// for GUI
    int maxCurrent;// for GUI
    int targetPositionToOpen;
    int targetPositionToClose;
    int targetPositionToClamp;
    int currentToClamp;
    int currentToGoHome;
    int readCurrent;
    
    private boolean controllerInError;
    private String  errorMessage;
    
    Map<String,Integer> paramsForCurrentToGoHome;
    Map<String,Integer> paramsForCurrentToClamp;

    public LoaderClampModule(String moduleName, int aTickMillis, 
            LoaderHookModule hook1, 
            LoaderHookModule hook2, 
            LoaderHookModule hook3, 
            LoaderHookModule hook4,
            long timeoutForClosingHooks, 
            long timeoutForClosingHooksStrongly, 
            long timeoutForOpeningHooks,
            long timeoutForGoingToHomePosition,
            int targetPositionToOpen,
            int targetPositionToClose, 
            int targetPositionToCloseStrongly, 
            int currentToClose,
            int currentToGoHome, 
            Map<String, Integer> paramsForCurrentToClamp,
            Map<String, Integer> paramsForCurrentToGoHomePosition,
            int minPosition,
            int maxPosition,
            int minCurrent,
            int maxCurrent) {
        super(moduleName, aTickMillis);
        this.hook1 = hook1;
        this.hook2 = hook2;
        this.hook3 = hook3;
        this.hook4 = hook4;
        this.initialized = false;
        this.timeoutForClosingHooks = timeoutForClosingHooks;
        this.timeoutForClampingHooks = timeoutForClosingHooksStrongly;
        this.timeoutForOpeningHooks = timeoutForOpeningHooks;
        this.timeoutForGoingToHomePosition = timeoutForGoingToHomePosition;
        this.targetPositionToOpen = targetPositionToOpen;
        this.targetPositionToClose = targetPositionToClose;
        this.targetPositionToClamp = targetPositionToCloseStrongly;
        this.currentToClamp = currentToClose;
        this.currentToGoHome = currentToGoHome;
        this.paramsForCurrentToClamp = paramsForCurrentToClamp;
        this.paramsForCurrentToGoHome = paramsForCurrentToGoHomePosition;
        this.minPosition = minPosition;
        this.maxPosition = maxPosition;
        this.minCurrent = minCurrent;
        this.maxCurrent = maxCurrent;

    }
    
    //for publication on the status bus for the GUI
    public int getFilterPresenceSensorValue() {
        return this.loader.getFilterPresenceSensor0().getDigitalValue();
    }
    
    

    public LoaderHookModule getHook1() {
        return hook1;
    }

    public LoaderHookModule getHook2() {
        return hook2;
    }

    public LoaderHookModule getHook3() {
        return hook3;
    }

    public LoaderHookModule getHook4() {
        return hook4;
    }

    public int getPosition() {
        return position;
    }

    //for the GUI
    public int getMinPosition() {
        return minPosition;
    }
    
    //for the GUI
    public int getMaxPosition() {
        return maxPosition;
    }

    public int getMinCurrent() {
        return minCurrent;
    }

    public int getMaxCurrent() {
        return maxCurrent;
    }
    
    
    

    //for the simulator
    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }
    //for the simulator
    public int getTargetPositionToClose() {
        return targetPositionToClose;
    }
    //for the simulator
    public int getTargetPositionToClamp() {
        return targetPositionToClamp;
    }
    //for the simulator
    public int getCurrentToClamp() {
        return currentToClamp;
    }
    //for the simulator
    public int getCurrentToGoHome() {
        return currentToGoHome;
    }

    public int getReadCurrent() {
        return readCurrent;
    }
    
    

    @ConfigChanger
    public long getTimeoutForClosingHooks() {
        return timeoutForClosingHooks;
    }

    public long getTimeoutForClampingHooks() {
        return timeoutForClampingHooks;
    }

    public long getTimeoutForOpeningHooks() {
        return timeoutForOpeningHooks;
    }

    @ConfigChanger
    public long getTimeoutForGoingToHomePosition() {
        return timeoutForGoingToHomePosition;
    }

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if hook is initialized.")
    public boolean isInitialized() {
        return initialized;
    }
    


    @ConfigChanger
    public void setTimeoutForClosingHooks(long timeoutForClosingHooks) {
        this.timeoutForClosingHooks = timeoutForClosingHooks;
    }

    @ConfigChanger
    public void setTimeoutForClampingHooks(long timeoutForClampingHooks) {
        this.timeoutForClampingHooks = timeoutForClampingHooks;
    }

    @ConfigChanger
    public void setTimeoutForOpeningHooks(long timeoutForOpeningHooks) {
        this.timeoutForOpeningHooks = timeoutForOpeningHooks;
    }

    @ConfigChanger
    public void setTimeoutForGoingToHomePosition(long timeoutForGoingToHomePosition) {
        this.timeoutForGoingToHomePosition = timeoutForGoingToHomePosition;
    }

    @ConfigChanger
    public void setMinPosition(int minPosition) {
        this.minPosition = minPosition;
    }

    @ConfigChanger
    public void setMaxPosition(int maxPosition) {
        this.maxPosition = maxPosition;
    }

    @ConfigChanger
    public void setTargetPositionToOpen(int targetPositionToOpen) {
        this.targetPositionToOpen = targetPositionToOpen;
    }

    @ConfigChanger
    public void setTargetPositionToClose(int targetPositionToClose) {
        this.targetPositionToClose = targetPositionToClose;
    }

    @ConfigChanger
    public void setTargetPositionToClamp(int targetPositionToClamp) {
        this.targetPositionToClamp = targetPositionToClamp;
    }

    @ConfigChanger
    public void setCurrentToClamp(int currentToClamp) {
        this.currentToClamp = currentToClamp;
    }
    
    @ConfigChanger
    public void setCurrentToGoHome(int currentToGoHome) {
        this.currentToGoHome = currentToGoHome;
    }
    
    
    
    

    public FcsEnumerations.LockStatus getLockStatus() {       
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.warning(name + ": interrupted in getLockStatus.");
                }

            }
            return lockStatus;
            
        } finally {
            lock.unlock();
        }        
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if LockStatus=LOCKED")
    public boolean isLocked() {
        return this.getLockStatus().equals(LockStatus.LOCKED);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if LockStatus=UNLOCKED")
    public boolean isUnlocked() {
        return this.getLockStatus().equals(LockStatus.UNLOCKED);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if LockStatus=UNLOCKED")
    public boolean isInError() {
        return this.getLockStatus().equals(LockStatus.ERROR);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if loader is empty.")
    public boolean isEmpty() {
        return loader.isEmpty();
    }
    
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if loader hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return loader.isHardwareReady();
    }
   
    @Override
    public void initModule() {
        this.hooksController = (EPOSController) this.getModule("hooksController");
        this.loader = (LoaderModule) this.getModule("loader");
        this.lockStatus = LockStatus.UNKNOWN;
        
        //listens to hooksController to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        this.listens((Observable) hooksController);
    }
    
     /**
     * What to do when the Modules we observe send there new values.
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        log.debug(name + ":processUpdate from source=" + source.toString() 
                + " ValueUpdate=" + v.getName());
        if (!(source instanceof CanOpenEPOS)) return;
        if (v.getValue() instanceof EmergencyMessage ) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            log.debug(name + ":EmergencyMessage received from CanOpenProxy=" 
                    + emcyMsg.toString());

            if (hooksController.getName().equals(emcyMsg.getDeviceName())) {
                String errCode = emcyMsg.getDeviceErrorCode();
                switch(errCode) {
                    case("00"):
                        log.debug(name + ":faultReset ?=" + emcyMsg.toString());
                        controllerInError=false;
                        errorMessage = emcyMsg.getDeviceErrorName();
                        
                    break;
                        
                    default:
                        log.debug(name + ":EmergencyMessage received for "
                                + "loader controller from CanOpenProxy=" + emcyMsg.toString());
                        controllerInError=true;
                        errorMessage = emcyMsg.getDeviceErrorName();
                    }              
                this.publishData();          
            }
            
        } else if (v.getValue() instanceof String) {
            CanOpenEPOS ctrl = (CanOpenEPOS) source;
            if (ctrl.getName().equals(hooksController.getName())) {
                String msgFromController = (String)v.getValue();
                if (msgFromController.equals("faultReset")) {
                    this.controllerInError=false;
                    this.errorMessage = null;
                    this.publishData();
                }
            }
        }
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if hook is open and at home position")
    public boolean isAtHomePosition() throws SDORequestException {
        return (position == 0);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if hook is closed and at clamped position")
    public boolean isAtClampedPosition() throws SDORequestException {
        return (this.targetPositionToClamp - 1000 <= this.position) && 
                       (this.targetPositionToClamp <= this.position + 1000);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if hook is at closed position.")
    public boolean isAtClosedPosition() throws SDORequestException {
        return (position == this.targetPositionToClose);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if hook is at open position.")
    public boolean isAtOpenPosition() throws SDORequestException {
        return (position == this.targetPositionToOpen);
    }
    
    /**
     * If loader is empty, go to home position otherwise go to clamped position.
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="If loader is empty, go to home position otherwise go to clamped position.",
            alias="homing")
    public void initializeHardware() throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        
        this.updateStateWithSensors();
        publishData();
     
        if (this.isInError()) {
            String msg = name + " in ERROR state. Can't initialize hardware.";
            log.error(msg);
            throw new FcsHardwareException(msg);
        }
        
        if(!hooksController.isInitialized()) {
                throw new FcsHardwareException(name 
                        + ": hooksController has to be initialized.");
        }
        
        if (this.isEmpty()) {
            this.goToHomePosition();
           
        } else {
            this.goToClampedPosition();

        }
        this.initialized = true;
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        //super.checkHardware();
        try {
            hooksController.initializeHardware();
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name + ":" + ex.getMessage());
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false,ex);
        } 
        return TreeWalkerDiag.HANDLING_CHILDREN;       
    }
    
    
    @Override
    public void tick() {
        //Don't execute here a method which takes a lock.
        publishData();
    }
    
    /**
     * In the initialisation phase, this close the clamp to be able to define the absolute position of the encoder.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Go to clamped position to intialize hardware.")
    public String goToClampedPosition() throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        
        if (!loader.isConnectedOnCamera()) throw new BadCommandException(name 
                + " Loader not connected - can't execute command goToClampedPosition.");
        
        updateStateWithSensors();
        
        if (this.isInError()) {
            String msg = name  + " in ERROR state - can't execute command goToClampedPosition.";
            log.error(msg);
            throw new FcsHardwareException(msg);
        }        
        
        if (isEmpty()) throw new BadCommandException(name 
                + "No filter in carrier -  can't execute command goToClampedPosition.");

        updatePosition();
        if (initialized && isAtClampedPosition()) {
            String msg = name + " is already at clamped position - nothing to do.";
            fcslog.warning(msg);
            return msg;
        }

        return this.executeAction(FcsEnumerations.MobileItemAction.GOTOCLAMPEDPOSITION, 
                timeoutForClampingHooks);
    }
    
    /**
     * In the initialisation phase, this open the clamp to be able to define the 
     * absolute position of the encoder.
     * September 2014 : this command is also used to open the clamp.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Go to home position to initialize hardware.")
    public String goToHomePosition() throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        
        updateStateWithSensors();
        
        if (this.isInError()) {
            String msg = name  + " in ERROR state - can't execute command goToHomePosition.";
            log.error(msg);
            throw new FcsHardwareException(msg);
        }
        if (!loader.isConnectedOnCamera()) throw new BadCommandException(name
                + " Loader hardware not connected on camera - "
                + "can't execute command goToHomePosition.");

        if (!initialized && !isEmpty()) throw new BadCommandException(name 
                + " There is a filter in carrier - can't execute command goToHomePosition.");
        
        if (initialized && !isEmpty() && !loader.isAutochangerHoldingFilter()) 
            throw new BadCommandException(name 
                + " There is a filter in carrier and Autochanger is not holding filter - "
                    + "can't execute command goToHomePosition. ");
 
        updatePosition();
        if (initialized && isAtHomePosition()) {
            String msg = name + " is already at home position - nothing to do.";
            fcslog.warning(msg);
            return msg;
        }

        return this.executeAction(FcsEnumerations.MobileItemAction.GOTOHOMEPOSITION, 
                timeoutForGoingToHomePosition);
    }
    

    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Close clamp.")
    public String close() throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {

        loader.checkPreConditionsForClosingHooks();
        if (this.isAtClosedPosition()) {
            String msg = name  + ": is already closed.";
            log.error(msg);
            throw new BadCommandException(msg);         
        }
        return  this.executeAction(FcsEnumerations.MobileItemAction.CLOSELOADERHOOKS, 
                timeoutForClosingHooks);
        
    }
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Clamp clamp.")
    public String clamp() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        
        loader.checkPreConditionsForClosingHooks();
        if (!isLocked()) throw new BadCommandException(name 
                + " has to be LOCKED first. Close hooks prior clamp.");  
        if (loader.isAutochangerHoldingFilter()) throw new BadCommandException(name 
                + " Autochanger is holding filter. Open autochanger latches before "
                + "clamping loader hooks.");
        return this.executeAction(FcsEnumerations.MobileItemAction.CLAMPLOADERHOOKS, 
                timeoutForClampingHooks);
    }
    
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Open clamp.")
    public String open() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
  
        loader.checkPreConditionsForOpeningHooks();
        return this.executeAction(FcsEnumerations.MobileItemAction.OPENLOADERHOOKS, 
                timeoutForOpeningHooks);
    }

    /**
     * Check if the action is completed.
     * This method is called after updateStateWithSensorsToCheckIfActionIsCompleted
     * where the different new values of current or position or sensors values 
     * have been updated.
     * @param action
     * @return 
     */
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        switch(action) {
           case GOTOCLAMPEDPOSITION :
               //TODO read the force sensor when it will be installed
               fcslog.debug("readCurrent=" + readCurrent + " currentToClamp=" + this.currentToClamp);
               return (this.currentToClamp - 5 <= this.readCurrent) && 
                       (this.readCurrent <= this.currentToClamp + 5);
               
           case GOTOHOMEPOSITION :
               fcslog.debug("readCurrent=" + readCurrent + " currentToGoHome=" + this.currentToGoHome);
               return (this.currentToGoHome - 5 <= this.readCurrent) && 
                       (this.readCurrent <= this.currentToGoHome + 5);               
                           
           case CLOSELOADERHOOKS : 
               return (this.position == this.targetPositionToClose);

           case CLAMPLOADERHOOKS : 
               //TODO read the force sensor
               return (this.currentToClamp - 5 <= this.readCurrent) && 
                       (this.readCurrent <= this.currentToClamp + 5);
               
           case OPENLOADERHOOKS :
               return (this.position == this.targetPositionToOpen);
        }
        return false;
        
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        this.updateStateWithSensors();
        if ((currentAction.equals(MobileItemAction.CLOSELOADERHOOKS))||
                (currentAction.equals(MobileItemAction.OPENLOADERHOOKS))) {
            position = this.hooksController.readPosition();            
        } else if ((currentAction.equals(MobileItemAction.GOTOHOMEPOSITION))||
                (currentAction.equals(MobileItemAction.GOTOCLAMPEDPOSITION)) ||
                (currentAction.equals(MobileItemAction.CLAMPLOADERHOOKS))){
            this.readCurrent = this.hooksController.readCurrent();
            //just to display on the GUI the position of the clamp
            position = this.hooksController.readPosition();
        }
        //TODO read the force sensor for the actions GOTOCLAMPEDPOSITION and CLAMP
    }

    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        
        
        switch(action) {  
            case GOTOCLAMPEDPOSITION :
                hooksController.enable();
                hooksController.changeMode(EposMode.CURRENT);
                hooksController.writeParameters(paramsForCurrentToClamp);
                hooksController.writeCurrent(this.currentToClamp);
                break;
                
           case GOTOHOMEPOSITION :
                hooksController.enable();
                hooksController.changeMode(EposMode.CURRENT);
                hooksController.writeParameters(paramsForCurrentToGoHome);
                hooksController.writeCurrent(this.currentToGoHome);
                break;
               
           case CLOSELOADERHOOKS :
                hooksController.enable();
                hooksController.changeMode(EposMode.PROFILE_POSITION);
                hooksController.writeParameters(EposMode.PROFILE_POSITION);
                hooksController.writeTargetPosition(this.targetPositionToClose);
                //hooksController.on();
                hooksController.writeControlWord("3F");
               break;
               
           case CLAMPLOADERHOOKS : 
               if (!(this.lockStatus.equals(LockStatus.LOCKED))) {
                   throw new BadCommandException(name + ": Hooks have to be LOCKED before being LOCKED STRONGLY");
               }
               hooksController.enable();
               hooksController.changeMode(EposMode.CURRENT);
               hooksController.writeParameters(paramsForCurrentToClamp);
               hooksController.writeCurrent(this.currentToClamp);
               break;
               
           case OPENLOADERHOOKS :
               hooksController.enable();
               hooksController.changeMode(EposMode.PROFILE_POSITION);
               hooksController.writeParameters(EposMode.PROFILE_POSITION);
               hooksController.writeTargetPosition(this.targetPositionToOpen);
               //hooksController.on();
               hooksController.writeControlWord("3F");
               break;
        }
    }
    
    @Override
    public void abortAction(MobileItemAction action, long delay) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        fcslog.debug(name + " is ABORTING action " + action.toString() 
                + " within delay " + delay);
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        fcslog.debug("Command running:" + currentCmdContext.getCommandName() );
        fcslog.info("coming from:" + currentCmdContext.getCommandOriginator());
        
        this.hooksController.off();
        
    }
    
    @Override
    public void quickStopAction(MobileItemAction action, long delay) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        fcslog.debug(name + " is STOPPING action " + action.toString() 
                + " within delay " + delay);
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        fcslog.debug("Command running:" + currentCmdContext.getCommandName() );
        fcslog.info("coming from:" + currentCmdContext.getCommandOriginator());
        
        this.hooksController.quickStop();
        
    }

    @Override
    public void postAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        
        //because we don't want to let the controller on power
        this.hooksController.off();
        fcslog.info(name + ":" + action.toString() + " completed - doing postAction.");
        
        switch(action) {
            case GOTOCLAMPEDPOSITION :
                if (!this.isLocked()) throw new ErrorInCommandExecutionException(name 
                            + ": check with sensors: clamp should be LOCKED."); 
                this.hooksController.defineAbsolutePosition(this.targetPositionToClamp);
                this.hooksController.off();
                updatePosition(); // for the GUI
                this.lockStatus = LockStatus.CLAMPED; //for the GUI
                this.publishData();

                break;
                
            case GOTOHOMEPOSITION :
                if (!this.isUnlocked()) throw new ErrorInCommandExecutionException(name 
                        + ": check with sensors: clamp should be UNLOCKED."); 
                this.hooksController.defineAbsolutePosition(0);
                this.hooksController.off();
                updatePosition(); // for the GUI
                this.publishData();
                break;
                
            case CLOSELOADERHOOKS : 
                if (!this.isLocked()) throw new ErrorInCommandExecutionException(name 
                        + ": check with sensors: clamp should be LOCKED."); 
                break;
                
            case CLAMPLOADERHOOKS :
                if (this.isLocked() && this.isAtClampedPosition()) {
                    this.lockStatus = FcsEnumerations.LockStatus.CLAMPED;
                    this.publishData();
                }
                break;
                
            case OPENLOADERHOOKS :
                if (!this.isUnlocked()) throw new ErrorInCommandExecutionException(name 
                        + ": check with sensors: clamp should be UNLOCKED.");
                break;
        }
    }
    
    /**
     * This methods updates the lockStatus in reading all the sensors.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Update clamp state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        loader.updateStateWithSensors();
    }

    /**
     * This methods updates the lockStatus from an array of hexaValues.
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void updateStateWithSensors(String[] readHexaValues) 
            throws FcsHardwareException {
        lock.lock();
            
        try {
            updatingState = true; 
            this.hook1.updateStateWithSensors(readHexaValues);
            this.hook2.updateStateWithSensors(readHexaValues);
            this.hook3.updateStateWithSensors(readHexaValues);
            this.hook4.updateStateWithSensors(readHexaValues);

            boolean inError = (this.hook1.getLockStatus() == LockStatus.ERROR) || 
                                    (this.hook2.getLockStatus() == LockStatus.ERROR) ||
                                    (this.hook3.getLockStatus() == LockStatus.ERROR) ||
                                    (this.hook4.getLockStatus() == LockStatus.ERROR);

            boolean locked = (this.hook1.getLockStatus() == LockStatus.LOCKED) && 
                                    (this.hook2.getLockStatus() == LockStatus.LOCKED) &&
                                    (this.hook3.getLockStatus() == LockStatus.LOCKED) &&
                                    (this.hook4.getLockStatus() == LockStatus.LOCKED);

            boolean unlocked = (this.hook1.getLockStatus() == LockStatus.UNLOCKED) && 
                                    (this.hook2.getLockStatus() == LockStatus.UNLOCKED) &&
                                    (this.hook3.getLockStatus() == LockStatus.UNLOCKED) &&
                                    (this.hook4.getLockStatus() == LockStatus.UNLOCKED);

            boolean inTravel = (this.hook1.getLockStatus() == LockStatus.INTRAVEL) && 
                                    (this.hook2.getLockStatus() == LockStatus.INTRAVEL) &&
                                    (this.hook3.getLockStatus() == LockStatus.INTRAVEL) &&
                                    (this.hook4.getLockStatus() == LockStatus.INTRAVEL);                

            if (inError) {
                this.lockStatus = LockStatus.ERROR;
            } else if (locked) {    
                this.lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                this.lockStatus = LockStatus.UNLOCKED;
            } else if (inTravel) {
                this.lockStatus = LockStatus.INTRAVEL;
            } else {
                this.lockStatus = LockStatus.UNKNOWN;
            }

            if (initialized && locked && isAtClampedPosition()) {
                this.lockStatus = LockStatus.CLAMPED;
            }
        
        } finally {
            
            updatingState = false;
            stateUpdated.signal(); 
            lock.unlock();
            this.publishData();
        } 
        
    }
    
     /**
     * Updates the field position of the carrier in reading the CPU of the controller.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Update carrier current in reading controller.")
    public void updateCurrent() throws BadCommandException, SDORequestException {
        try {
            this.readCurrent = hooksController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name + "=> ERROR IN READING CONTROLLER:" + ex.getMessage());
        }
        this.publishData();       
    }
    
    /**
     * Updates the position of the clamp in reading the CPU of the controller.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Update clamp position in reading controller.")
    public void updatePosition() throws SDORequestException {
        try {
            this.position = hooksController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name + "=> ERROR IN READING CONTROLLER:" + ex.getMessage());
        }
        this.publishData();       
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp info.")
    public String listHooks() {
        StringBuilder sb = new StringBuilder("Hooks status:");sb.append("\n");
        sb.append(this.hook1.toString());sb.append("/");sb.append(this.hook1.getLockStatus());sb.append("\n");
        sb.append(this.hook2.toString());sb.append("/");sb.append(this.hook2.getLockStatus());sb.append("\n");
        sb.append(this.hook3.toString());sb.append("/");sb.append(this.hook3.getLockStatus());sb.append("\n");
        sb.append(this.hook4.toString());sb.append("/");sb.append(this.hook4.getLockStatus());sb.append("\n");
        return sb.toString();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp sensors values.")
    public String listSensorsValues() {
        StringBuilder sb = new StringBuilder(name + "Sensors values=");sb.append("/");
        sb.append(this.hook1.listSensorsValues());sb.append("/");
        sb.append(this.hook2.listSensorsValues());sb.append("/");
        sb.append(this.hook3.listSensorsValues());sb.append("/");
        sb.append(this.hook4.listSensorsValues());sb.append("/");
        return sb.toString();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp sensors values.")
    public String listStatus() {
        StringBuilder sb = new StringBuilder("Clamp status:");sb.append(this.lockStatus);sb.append("\n");
        sb.append(this.hook1.getName());sb.append("/");sb.append(this.hook1.getLockStatus());sb.append("\n");
        sb.append(this.hook2.getName());sb.append("/");sb.append(this.hook2.getLockStatus());sb.append("\n");
        sb.append(this.hook3.getName());sb.append("/");sb.append(this.hook3.getLockStatus());sb.append("\n");
        sb.append(this.hook4.getName());sb.append("/");sb.append(this.hook4.getLockStatus());sb.append("\n");
        return sb.toString();
    }
    

    
    @Override
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp info.")
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/timeoutForClosingHooks=");sb.append(this.timeoutForClosingHooks);
        sb.append("/timeoutForClosingHooksStrongly=");sb.append(this.timeoutForClampingHooks);
        sb.append("/timeoutForOpeningHooks=");sb.append(this.timeoutForOpeningHooks);
        return sb.toString();
    }

     public StatusDataPublishedByLoaderClamp getStatusData() {
        StatusDataPublishedByLoaderClamp status = FcsUtils.createStatusDataPublishedByLoaderClamp(this);
        return status;
    }
     
     /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
     public void publishData() {
        StatusDataPublishedByLoaderClamp status = this.getStatusData();
        this.publish("loaderClamp", status);
     }







   


    
}
