
/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.CurrentCommandContext;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for the carrier in the loader.
 * @author virieux
 */
public class LoaderCarrierModule extends MobileItemModule {
    
    protected NumericSensor storagePositionSensor0;
    protected NumericSensor handoffPositionSensor0;
    protected NumericSensor storagePositionSensor1;
    protected NumericSensor handoffPositionSensor1;

    private int position;
    private int ssiPosition;
    private int handoffPosition; //maxPosition
    private int storagePosition; //minPosition
    private int speed;
    private int current;
    private final int maxSpeed;
    private final int maxCurrent;
    
    
    private long timeoutForGoingToHandOff;
    private long timeoutForGoingToStorage;

    private EPOSController carrierController;
    private LoaderModule loader;
    
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    /** true if the 2 position sensors value at Storage is 1 **/
    private boolean atStorage;
    
    /** true if the 2 position sensors value at Handoff is 1 **/
    private boolean atHandoff;
    
    /** true if the 2 position sensors value at Storage differ **/
    private boolean storageSensorsInError;
    
    /** true if the 2 position sensors value at Handoff differ **/
    private boolean handoffSensorsInError;
    
    /** true if the sensors value are not consistant with the position read on carrierController.**/
    private boolean noConsistant;
    
    /** true if the controller is in fault - false if the user does a faultReset**/
    private boolean controllerInFault;
    
    /** error message received in the emergency message sent by carrierController**/
//    private String  controllerErrorMessage; //it was for the GUI - no more useful
//    private String[]  controllerErrorHistory;
    
    private boolean limitSwitchDownInError;
    private boolean limitSwitchUpInError;
    private volatile boolean initialized;
    

    public LoaderCarrierModule(String moduleName, int aTickMillis,
            //String plutoGatewayName,
            int minPos,
            int maxPos,
            NumericSensor handoffPositionSensor,
            NumericSensor handoffPositionSensor1,
            NumericSensor storagePositionSensor, 
            NumericSensor storagePositionSensor1,
            long timeoutForGoingToHandOff,
            long timeoutForGoingToStorage,
            int maxCurrent,
            int maxSpeed

             ) {
        super(moduleName, aTickMillis);
        this.storagePositionSensor0 = storagePositionSensor;
        this.handoffPositionSensor0 = handoffPositionSensor;
        this.storagePositionSensor1 = storagePositionSensor1;
        this.handoffPositionSensor1 = handoffPositionSensor1;
        this.timeoutForGoingToHandOff = timeoutForGoingToHandOff;
        this.timeoutForGoingToStorage = timeoutForGoingToStorage;
        //this.plutoGatewayName = plutoGatewayName;
        this.storagePosition = minPos;
        this.handoffPosition = maxPos;
        this.maxCurrent = maxCurrent;
        this.maxSpeed = maxSpeed;
        this.atHandoff = false;
        this.atStorage = false;
        this.position = 0;
        this.ssiPosition = 0;
        this.storageSensorsInError = false;
        this.handoffSensorsInError = false;
        this.controllerInFault = false;
        this.limitSwitchDownInError = false;
        this.limitSwitchUpInError = false;
    }
    
    

    public int getPosition() {
        return position;
    }

    public int getSsiPosition() {
        return ssiPosition;
    }
    
    
    public int getSpeed() {
        return speed;
    }

    public int getCurrent() {
        return current;
    }

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if carrierController is in Fault.")
    public boolean isControllerInFault() {
        return controllerInFault;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if carrier is initialized and ready to receive commands.")
    public boolean isInitialized() {
        return initialized;
    }
    
    
    
    /**
     * 
     * @return max speed in rpm/mn (format decimal).
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return the max speed in rpm/mn (format decimal).")
    public int getMaxSpeed() {
        return maxSpeed;
    }

     /**
     * 
     * @return max current in mA (format decimal).
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return the max current in mA (format decimal).")
    public int getMaxCurrent() {
        return maxCurrent;
    }
    
    /**
     * 
     * @return handoff position in microns (format decimal).
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return the handoff position in microns (format decimal).")
    public int getHandoffPosition() {
        return handoffPosition;
    }
    
    /**
     * 
     * @return storage position in microns (format decimal).
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return the storage position in microns (format decimal).")
    public int getStoragePosition() {
        return storagePosition;
    }

    public NumericSensor getStoragePositionSensor0() {
        return storagePositionSensor0;
    }

    public NumericSensor getHandoffPositionSensor0() {
        return handoffPositionSensor0;
    }

    public NumericSensor getStoragePositionSensor1() {
        return storagePositionSensor1;
    }

    public NumericSensor getHandoffPositionSensor1() {
        return handoffPositionSensor1;
    }

    /**
     * 
     * @return timeout for going to Handoff action in millis (format decimal).
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return the timeout for going to Handoff in millis (format decimal).")
    public long getTimeoutForGoingToHandOff() {
        return timeoutForGoingToHandOff;
    }

     /**
     * 
     * @return timeout for going to Storage action in millis (format decimal).
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return the timeout for going to Storage in millis (format decimal).")
    public long getTimeoutForGoingToStorage() {
        return timeoutForGoingToStorage;
    }
    
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return false if the 2 redondant position sensors at Storage are equal.")
    public boolean isStorageSensorsInError() {
        return storageSensorsInError;
    }

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return false if the 2 redondant position sensors at HandOff are equal.")
    public boolean isHandoffSensorsInError() {
        return handoffSensorsInError;
    }

    public boolean isLimitSwitchDownInError() {
        return limitSwitchDownInError;
    }

    public boolean isLimitSwitchUpInError() {
        return limitSwitchUpInError;
    }
    
    

    /**
     * 
     * @param handoffPosition 
     */
    @ConfigChanger
    public void setHandoffPosition(int handoffPosition) {
        this.handoffPosition = handoffPosition;
    }
    
    /**
     * 
     * @param storagePosition 
     */
    @ConfigChanger
    public void setStoragePosition(int storagePosition) {
        this.storagePosition = storagePosition;
    }

    @ConfigChanger
    public void setTimeoutForGoingToHandOff(long timeoutForGoingToHandOff) {
        this.timeoutForGoingToHandOff = timeoutForGoingToHandOff;
    }

    @ConfigChanger
    public void setTimeoutForGoingToStorage(long timeoutForGoingToStorage) {
        this.timeoutForGoingToStorage = timeoutForGoingToStorage;
    }
    
    
    
    
    
    @Override
    public void initModule() {
        this.loader = (LoaderModule) this.getModule("loader");
        this.carrierController = (EPOSController) this.getModule("carrierController");
        
        //listens to carrierController to detect the controller's faultReset
        //or the emergency messages coming from the controller.
        this.listens((Observable) carrierController);
    }
    
    @Override
    public void tick() {
        this.publishData();
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();
        try {
            carrierController.initializeHardware();
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name + ":" + ex.getMessage());
        } catch (FcsHardwareException ex) {
            //throw new HardwareException(false,name+ ":" + ex.getMessage());
            throw new HardwareException(false,ex);
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;       
    }
    
    //done in LoaderModule
//    @Override
//    public void checkStarted() throws HardwareException {
//        if(!carrierController.isInitialized()) {
//            throw new HardwareException(true, "carrierController not ready, has to be intialized.");
//        }
//    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if loader hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return loader.isHardwareReady();
    }
    
    /**
     * Updates the boolean field empty and returns it.
     * @return empty
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="Read the sensors and return true if a filter is in the loader.")
    public boolean checkFilterPresence() throws FcsHardwareException {
        this.updateStateWithSensors();
        return !this.isEmpty();
    }
    

    
     /**
     * Returns the boolean field empty. 
     * If the empty boolean is being updated and waits 
     * for a response from a sensor, this methods waits until empty is updated.
     * If the field empty is not being updated, it returns immediatly the field empty.
     * 
     * @return empty
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if there is no filter in the loader. "
                        + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return loader.isEmpty();
    }
    
    
    
     /**
     * Returns the boolean field atStorage. 
     * If the atStorage boolean is being updated and waits 
     * for a response from a sensor, this methods waits until atStorage is updated.
     * If the field atStorage is not being updated, it returns immediatly the field atStorage.
     * 
     * @return atStorage
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the carrier is at storage position. "
                        + "This command doesn't read again the sensors.")
    public boolean isAtStoragePosition() {
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return atStorage;
            
        } finally {
            lock.unlock();
        }
    }
    
     /**
     * Returns the boolean field atHandoff. 
     * If the atHandoff boolean is being updated and waits 
     * for a response from a sensor, this methods waits until atHandoff is updated.
     * If the field atHandoff is not being updated, it returns immediatly the field atHandoff.
     * 
     * @return atHandoff
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the carrier is at HANDOFF position. "
                        + "This command doesn't read again the sensors.")
    public boolean isAtHandoffPosition() {
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return atHandoff;
            
        } finally {
            lock.unlock();
        }
    }
    
    /**
     * 
     * @return true if loader is connected on camera
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the loader is connected on the camera. "
                        + "This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        return loader.isConnectedOnCamera();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the autochanger is holding the filter. "
                        + "This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter() {
        return loader.isAutochangerHoldingFilter();
    }   


    /**
     * Move the carrier to Handoff position.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws FcsHardwareException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1,
            alias="goToHandoff",
            description="Move the carrier to Handoff position.")
    public String goToHandOff() throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        if (this.atHandoff) throw new BadCommandException(name + " is already at Handoff position.");
        loader.checkPreConditionsForCarrierMotion();
        return this.executeAction(FcsEnumerations.MobileItemAction.MOVELOADERCARRIERTOHANDOFF, 
                timeoutForGoingToHandOff);
    }

    /**
     * Move the carrier to storage position.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws FcsHardwareException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Move the carrier to Storage position.")
    public String goToStorage() throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        if (this.atStorage) throw new BadCommandException(name + " is already at Standby position.");
        loader.checkPreConditionsForCarrierMotion();
        return this.executeAction(FcsEnumerations.MobileItemAction.MOVELOADERCARRIERTOSTORAGE, 
                timeoutForGoingToStorage);
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        switch(action) {               
                           
           case MOVELOADERCARRIERTOHANDOFF : 
               return (this.position == this.handoffPosition);
               
           case MOVELOADERCARRIERTOSTORAGE :
               return (this.position == this.storagePosition);
        }
        return false;
    }
    
    /**
     * Updates the field position of the carrier in reading the CPU of the controller.
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Update carrier position in reading controller.")
    public void updatePosition() throws SDORequestException {
        try {
            this.position = carrierController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name + "=> ERROR IN READING CONTROLLER:" + ex.getMessage());
        }
        this.publishData(); 
        
    }
    
        /**
     * Updates the field position of the carrier in reading the CPU of the controller.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Update carrier current in reading controller.")
    public void updateCurrent() throws BadCommandException, SDORequestException {
        try {
            this.current = carrierController.readCurrent();
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name + "=> ERROR IN READING CONTROLLER:" + ex.getMessage());
        }
        this.publishData();       
    }
    

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() 
            throws Exception {
        try {
            carrierController.checkFault();
            this.position = this.carrierController.readPosition();
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name + "=> SDO ERROR IN READING CONTROLLER:" + ex.getMessage());
        }
        //for the GUI : is it the right place ?
        try {
            this.updateStateWithSensors();
            this.current = this.carrierController.readCurrent();
            this.speed = this.carrierController.readProfileVelocity();
        } catch (ShortResponseToSDORequestException | SDORequestException ex) {
            log.warning(name + "=> SDO ERROR IN READING CONTROLLER:" + ex.getMessage());
        }
        
    }
    
    /**
     * This methods updates the carrier state in reading all the carrier sensors.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Update clamp state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        loader.updateStateWithSensors();
    }
    
     /**
     * This methods updates the lockStatus from an array of hexaValues.
     * @param readHexaValues
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    protected void updateStateWithSensors(String[] readHexaValues) 
            throws FcsHardwareException {
        lock.lock();
            
        try {
            updatingState = true;

            this.handoffPositionSensor0.updateValue(readHexaValues);
            this.handoffPositionSensor1.updateValue(readHexaValues);
            this.storagePositionSensor0.updateValue(readHexaValues);
            this.storagePositionSensor1.updateValue(readHexaValues);

            //TODO compute a state 
            this.atStorage = storagePositionSensor0.getDigitalValue() == 1 
                    && storagePositionSensor1.getDigitalValue() == 1;
            this.atHandoff = handoffPositionSensor0.getDigitalValue() == 1 
                    && handoffPositionSensor1.getDigitalValue() == 1;
            this.storageSensorsInError = !(storagePositionSensor0.getDigitalValue() == storagePositionSensor1.getDigitalValue());
            this.handoffSensorsInError = !(handoffPositionSensor0.getDigitalValue() == handoffPositionSensor1.getDigitalValue());
            //TODO check if motion is allowed when positions sensors are in error.
        
        } finally {

            updatingState = false;
            stateUpdated.signal(); 
            lock.unlock();
            this.publishData();
            
        } 
        
    }



    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) 
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        carrierController.checkFault();
        
        if (action.equals(FcsEnumerations.MobileItemAction.MOVELOADERCARRIERTOHANDOFF)) {
            //TODO : check that if the filter is in the autochanger,the autochanger must hold the filter
            //and the loader clamp must be opened.
                carrierController.enable();
                carrierController.changeMode(EposMode.PROFILE_POSITION);
//                carrierController.writeParameters(EposMode.PROFILE_POSITION);
                carrierController.writeTargetPosition(this.handoffPosition);
                //carrierController.on();
                carrierController.writeControlWord("3F");

        } else if (action.equals(FcsEnumerations.MobileItemAction.MOVELOADERCARRIERTOSTORAGE)) {
                carrierController.enable();
                carrierController.changeMode(EposMode.PROFILE_POSITION);
//                carrierController.writeParameters(EposMode.PROFILE_POSITION);
                carrierController.writeTargetPosition(this.storagePosition);
                //carrierController.on();
                carrierController.writeControlWord("3F");


        } 
    }
    
    @Override
    public void abortAction(MobileItemAction action, long delay) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
                fcslog.debug(name + " is ABORTING action " + action.toString() 
                + " within delay " + delay);
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        fcslog.debug("Command running:" + currentCmdContext.getCommandName() );
        fcslog.info("coming from:" + currentCmdContext.getCommandOriginator());
        
        this.carrierController.off();
    }
    

    @Override
    public void quickStopAction(MobileItemAction action, long delay) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
                fcslog.debug(name + " is STOPPING action " + action.toString() 
                + " within delay " + delay);
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        fcslog.debug("Command running:" + currentCmdContext.getCommandName() );
        fcslog.info("coming from:" + currentCmdContext.getCommandOriginator());
        
        //doesn't work : we can't do a quickStop when in PROFILE_POSITION
        //this.carrierController.quickStop();
        this.carrierController.off();
        
    }
    
    

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        //because we don't want to let the controller on power
        this.carrierController.off();
        fcslog.info(name + ":" + action.toString() + " completed - doing postAction.");
        noConsistant = false;
        
        switch(action) {
            
            case MOVELOADERCARRIERTOSTORAGE :
                if (!this.isAtStoragePosition()) {
                    noConsistant=true;
                    throw new ErrorInCommandExecutionException(name 
                            + ": check with sensors: storage sensors don't confirm carrier position.");
                }
                break;
                
            case MOVELOADERCARRIERTOHANDOFF :
                if (!this.isAtHandoffPosition()) {
                    noConsistant=true;
                    throw new ErrorInCommandExecutionException(name 
                        + ": check with sensors: handoff sensors don't confirm carrier position.");
                }
                break;
            }
        
    }

    /**
     * Command for ENGINEERING mode.
     * Not used during INITIALIZATION.
     * @throws FcsHardwareException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Check if hardware is ready to be started.")
    public void initializeHardware() throws FcsHardwareException {
        
        if(!carrierController.isInitialized()) {
                throw new FcsHardwareException(name 
                        + ": carrierController has to be initialized.");
        }    
        updateStateWithSensors();
        try {
            updatePosition();
        } catch (SDORequestException ex) {
            String msg = name  + ": couldn't update position";
            log.error(msg);
            throw new FcsHardwareException(name,ex);
        }
        noConsistant=false;
        if (this.isAtHandoffPosition() && 
                !(this.position == this.handoffPosition||this.position > this.handoffPosition)) {
            noConsistant=true;
            throw new FcsHardwareException(name 
                        + ": handoff sensors don't confirm position read on carrierController.");
        }
        if (this.isAtStoragePosition() && !(this.position == this.storagePosition)) {
            noConsistant=true;
            throw new FcsHardwareException(name 
                        + ": storage sensors don't confirm position read on carrierController.");
        }
        this.initialized = true;
    }
    
    public StatusDataPublishedByLoaderCarrier getStatusData() {
        StatusDataPublishedByLoaderCarrier status = FcsUtils.createStatusDataPublishedByLoaderCarrier(this);
        return status;
    }
    
     /**
     * What to do when the Modules we observe send there new values.
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        log.debug(name + ":processUpdate from source=" + source.toString() 
                + " ValueUpdate=" + v.getName());
        if (!(source instanceof CanOpenEPOS)) return;
        if (v.getValue() instanceof EmergencyMessage ) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            log.debug(name + ":EmergencyMessage received from CanOpenProxy=" 
                    + emcyMsg.toString());

            if (carrierController.getName().equals(emcyMsg.getDeviceName())) {
                String errCode = emcyMsg.getDeviceErrorCode();
                switch(errCode) {
                    case("00"):
                        log.debug(name + ":faultReset ?=" + emcyMsg.toString());
                        controllerInFault=false;
                        
                    break;
                        
                    default:
                        log.debug(name + ":EmergencyMessage received for "
                                + "loader controller from CanOpenProxy=" + emcyMsg.toString());
                        controllerInFault=true;
                        //limitSwitchErrors
                        if (emcyMsg.getDeviceErrorCode().equalsIgnoreCase("ff06")) { 
                            //Negative Limit Switch Error
                            this.limitSwitchDownInError = true;
                        } else if (emcyMsg.getDeviceErrorCode().equalsIgnoreCase("ff07")) {
                            this.limitSwitchUpInError = true;
                        }
                    }              
                this.publishData();          
            }
            
        } else if (v.getValue() instanceof String) {
            CanOpenEPOS ctrl = (CanOpenEPOS) source;
            if (ctrl.getName().equals(carrierController.getName())) {
                String msgFromController = (String)v.getValue();
                if (msgFromController.equals("faultReset")) {
                    this.limitSwitchDownInError = false;
                    this.limitSwitchUpInError = false;
                    this.controllerInFault=false;
                    //this.controllerErrorMessage = null;
                    this.publishData();
                } else if (msgFromController.equals("checkFault")) {
                    this.controllerInFault = carrierController.isInError();
                }
            }
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        StatusDataPublishedByLoaderCarrier status = this.getStatusData();
        this.publish("loaderCarrier", status);
    }
   





    
    
}
