package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.singlefiltertest.BasicAutoChangerModule;
import java.util.List;
import java.util.Observable;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.Carousel;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.RunningWay;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This is a representation of the hardware of the carousel.
 * It receives commands from the FCSMAinModule and send back an acknowledge.
 * It publishes data on the status bus.
 * In engineering mode it can receive commands from the engineering console.
 * It has a latch, an engine which are connected to real or simulated hardware. 
 * @author virieux
 *
 */
public class CarouselModule extends Module implements Carousel {

    /**
     * 
     */
    private static final long serialVersionUID = -2376279469784152348L;
    protected static final Logger fcslog = FcsUtils.log;
    /*This actuator opens the clamp Xminus when the carousel is halted at standby position.*/
    private Actuator clampActuatorXminus;
    /*This actuator opens the clamp Xplus when the carousel is halted at standby position.*/
    private Actuator clampActuatorXplus;
    
    private BasicAutoChangerModule autochanger;
    
    private final int nbSockets;
    
    //because Groovy has issues with Array, what we get from the description file is a List<CarouselSocket>
    private static final CarouselSocket[] socketsModel = new CarouselSocket[0];
    private final CarouselSocket[] sockets;
    
    private BridgeToHardware bridge;

    public final static String publishedByCarouselOutputName = "publishedByCarousel";
    private volatile ModuleState state = ModuleState.HALTED;
    private boolean stopped = false;
    //private boolean initialized = false;

    public CarouselModule(String aName, int aTickMillis,
            int nbSockets, 
            List<CarouselSocket> socketsList) {
        super(aName, aTickMillis);
        this.nbSockets = nbSockets;
        this.sockets = socketsList.toArray(socketsModel);
    }
    




    //SETTERS and GETTERS
    public CarouselSocket[] getSockets() {
        return sockets;
    }

    @Override
    public double getPosition() {
        //return carouselMotor.getPosition();
        return 0;
    }

    public synchronized ModuleState getState() {
        return state;
    }

    public synchronized void setState(ModuleState state) {
        this.state = state;
    }

    /**
     * @return the nbSockets
     */
    public int getNbSockets() {
        return nbSockets;
    }

    /**
     * @return the clampActuatorXminus
     */
    public Actuator getClampActuatorXminus() {
        return clampActuatorXminus;
    }

    /**
     * @return the clampActuatorXminus
     */
    public Actuator getClampActuatorXplus() {
        return clampActuatorXplus;
    }

    /**
     * This method returns the clampX- which is at standby position.
     * It can returns null if there is no socketAtStandby halted at standby position.
     * 
     * @return 
     */
    public FilterClampModule getClampXminus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return (FilterClampModule) socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at standby position.
     * It can returns null if there is no socketAtStandby halted at standby position.
     * 
     * @return 
     */
    public FilterClampModule getClampXplus() {
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {
            return (FilterClampModule) socketAtStandby.getClampXplus();
        }
    }

    @Override
    public void initModule() {
        fcslog.info("[CarouselModule] Initializing the carousel module ");
        setState(ModuleState.HALTED);
        this.bridge =  (BridgeToHardware) getModule("bridge");
        this.clampActuatorXminus = (Actuator) getModule("clampActuatorXminus");
        this.clampActuatorXplus = (Actuator) getModule("clampActuatorXplus");
        this.autochanger = (BasicAutoChangerModule) getModule("autochanger");
    }

    public StatusDataPublishedByCarousel getStatusData() {
        return FcsUtils.createStatusDataPublishedByCarousel(this);
    }
    
     /**
     * Publish Data on status bus for trending data base and GUIs.
     * 
     */
     public void publishData() {
        StatusDataPublishedByCarousel status = this.getStatusData(); 
        this.publish("carousel", status);
     }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n Numbers of sockets: ");
        sb.append(nbSockets);
        for (int i = 0; i < nbSockets; i++) {
            sb.append("\n Socket number ");
            sb.append(i);
            sb.append(" / ");
            //sb.append(sockets[i].toString());
            sb.append(String.valueOf(sockets[i]));
        }
        //sb.append("\nClamps actuator  ").append(String.valueOf(clampActuatorXminus)).append("\n");
        return sb.toString();
    }

    /**
     * What to do when the Modules we observe send there new values : we notify our own observers
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        StatusDataPublishedByCarousel status = getStatusData();
        //sendToStatus(status);
        //this.publishCarouselData();
        this.publishData();
        setChanged();
        notifyObservers(new ValueUpdate(publishedByCarouselOutputName, status));
        fcslog.info(getName() + ": Filter at standby position : " + getFilterInStandbyName());

    }

    @Override
    public synchronized CarouselSocket getSocketAtStandby() {
        if (this.isRotating()) {
            return null;
        } else {
            for (CarouselSocket socket : sockets) {
                if (this.getPosition() == socket.getStandbyPosition()) {
                    return socket;
                }
            }
        }
        return null;
    }

    /**
     * This methods return the filter which is in the socketAtStandby at standby position.
     * When there is no filter at standby position it returns null.
     * 
     * @return filter at standby position or null if there is no filter at standby position.
     */
    public synchronized Filter getFilterAtStandby() {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return null;
        } else {

            return socketAtStandby.getFilter();

        }
    }

    public synchronized String getFilterInStandbyName() {

        Filter filterAtStandby = getFilterAtStandby();
        if (filterAtStandby == null) {
            return "none";
        } else {
            return filterAtStandby.getName();
        }

    }

    /**
     * This method let us know if the carousel is ready to receive a filter at
     * standby position : 
     *  - the carousel must not rotate
     *  - an empty socketAtStandby is at standby position.
     * @return true if the position of the carousel matches
     *              the position when one of its sockets is at standby position
     *              and this socketAtStandby is empty.
     *         false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command ( level=Command.ENGINEERING1, description="Returns true if Carousel is stopped and no filter is at STANDBY position", type=Command.CommandType.QUERY)
    public boolean isReadyToGrabAFilterAtStandby() throws FcsHardwareException {

        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }
    
    /**
     * 
     * @return true if a filter is clamped at STANDBY position and carousel is stopped
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command ( level=Command.ENGINEERING1, description="Returns true if a filter is clamped at STANDBY position", type=Command.CommandType.QUERY)
    public boolean isHoldingFilterAtStandby() throws FcsHardwareException {
        if (this.isRotating()) {
            return false;
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            return false;
        }
        if (socketAtStandby.isEmpty()) {
            return false;
        }
        if (socketAtStandby.isClampedOnFilter()) {
            return true;
        }
        return false;
        
        
    }

    public boolean isRotating() {
        //return carouselMotor.getEngineState().equals(EngineState.RUNNING);
        return false;
    }

    /**
     * 
     * @return true if the brake is released
     */
    @Override
    @Command ( level=Command.ENGINEERING1, description="Returns true if the brake is released", type=Command.CommandType.QUERY)
    public boolean isAbleToMove() {
        return true;
    }

    public boolean isLocked() {
        return true;
    }

    @Override
    @Command ( level=Command.ENGINEERING1, description="Engage the carousel brake to stop the rotation", type=Command.CommandType.ACTION)
    public String engageBrake() {
//        brake.lock();
        return getName() + " locked";

    }

    @Command ( level=Command.ENGINEERING1, description="Release the carousel brake", type=Command.CommandType.ACTION)
    public String releaseBrake() {
//        brake.unlock();
        return getName() + " unlocked";
    }
    
    
    /**
     * Read the clamp sensors one by one with RSDO command
     * @throws BadCommandException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromSDO() throws BadCommandException, FcsHardwareException {
        if (this.getSocketAtStandby() == null) { 
            throw new BadCommandException(getName() 
                    + " can't update Clamps State because there is no socket at standby.");
        }
        this.getSocketAtStandby().updateClampsStateWithSensors();
    }
    
    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one time.
     * Tested with success on single-filter-test in April 2013
     * @throws BadCommandException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO() throws BadCommandException, FcsHardwareException {
        if (this.getSocketAtStandby() == null) { 
            throw new BadCommandException(getName() 
                    + " can't update Clamps State because there is no socket at standby.");
        }
        PDOStorage pdoStore = this.bridge.readPDOs();
        fcslog.finest(name + ":pdoStore=" + pdoStore.toString());
        this.getSocketAtStandby().updateClampsStateWithSensors(pdoStore);
    }
    
    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one time.
     * Tested with success on single-filter-test in April 2013
     * @throws BadCommandException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command ( level=Command.ENGINEERING1, description="Read clamps sensors and update clamps state", type=Command.CommandType.QUERY)
    public void updateClampsStateWithSensors() throws BadCommandException, FcsHardwareException {
        updateClampsStateWithSensorsFromPDO();
    }

    /**
     * Rotates carousel for an angle given as a parameter.
     * This method can be launched by the user from the TestConsole.
     * It checks the value of the rotation angle entered by the user,
     * converts it in a double,
     * and send the result of the method rotate(double angle).
     * @param angle : rotation angle
     * 			The rotation angle can be positive or negative.
     * 			It must be a double and have a value between -360.0 and 360.0
     * @return the result of the command rotate(double angle)
     * @throws IllegalArgumentException, BadCommandException 
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command ( level=Command.ENGINEERING1, description="Rotate carousel for an angle", type=Command.CommandType.ACTION)
    @Override
    @Deprecated
    public String rotate(double angle) throws IllegalArgumentException, BadCommandException {


        //availables values for an angle : [-360,+360]
        if (angle < -360 || angle > 360) {
            throw new IllegalArgumentException("Please enter an angle between -360 and +360");
        }

        RunningWay way = null;
        double absAngle;
        if (angle >= 0) {
            absAngle = angle;
            way = RunningWay.POSITIVE;
        } else {
            absAngle = -angle;
            way = RunningWay.NEGATIVE;
        }
        double requiredPosition = addAngle(getPosition(), angle);
        return rotate(absAngle, way, requiredPosition);
    }

    /**
     * Rotate the carousel to a final position.
     * 
     * @param angle of the rotation : 0 <= angle <= 360
     * @param the final position has to be : 0 <= finalPosition <= 360
     * @param runningWay rotation way (enum)
     * @param finalPosition is the position the carousel must have
     * 			when the command is completed
     * The method rotate is delegated to the engine, which is an interface.
     * An engine can be a simulated engine or a real hardware driver.
     * @throws BadCommandException, IllegalArgumentException
     *
     */
    @Deprecated
    private String rotate(double angle, RunningWay runningWay, double finalPosition)
            throws BadCommandException, IllegalArgumentException {
        //assertion 0 <= angle <= 360
        //assertion 0 <= finalPosition <= 360
        if (angle < 0 || angle > 360) {
            throw new IllegalArgumentException(getName()
                    + " command rotate accepts angle only between 0 and 360)");
        }
        if (finalPosition < 0 || finalPosition > 360) {
            throw new IllegalArgumentException(getName()
                    + " command rotate accepts finalPosition only between 0 and 360)");
        }

        String message = null;

        if (isRotating()) {
            // Thread-safe because of encapsulation for carousel.state		
            //if (getClampsState() == ModuleState.RUNNING) {		
            message = "Carousel is rotating, stop it before sending a new rotate command.";
            fcslog.error(message);
            throw new BadCommandException(message);
        }

        if (!this.isAbleToMove()) {
            message = "Carousel is unable to move. Check lock or flip rail.";
            fcslog.error(message);
            throw new BadCommandException(message);
        }


        stopped = false;
        setState(ModuleState.RUNNING);

//        this.carouselMotor.setRequiredPosition(finalPosition);
//        this.sendToReply(new CommandReply("Please wait during carousel rotation...", CommandStatus.OK));
//        carouselMotor.move(angle, runningWay);
        message = "Rotating to required position: " + finalPosition;
        fcslog.info(message);

        while (isRotating()) {
            try {
                fcslog.info("...carousel rotating, please wait.....");
                fcslog.info(getName() + " WAITING TO BE AT REQUIRED POSITION=" + finalPosition);
                fcslog.info(state.toString());
                Thread.sleep(tickMillis);
            } catch (InterruptedException e) {
                return "Sleeping interrupted";
            }
        }
        if (!isRotating()) {
            fcslog.info("THREAD=" + Thread.currentThread().getName());
            setState(ModuleState.HALTED);
            fcslog.info("carousel is now stopped.");
            //fcslog.info("filter in standby position: " + getFilterInStandbyName());
            if (getPosition() == finalPosition) {
                message = "Command completed";
                fcslog.info(message);
            } else {
                message = "Command non completed: carousel is not stopped at the required position";
                fcslog.info(message);
            }
        }

        if (stopped) {
            message = "Command rotate carousel interrupted by command stop";
            fcslog.info(message);
        }

        if (!stopped && (!(getPosition() == finalPosition))) {
            message = "Command non completed: carousel is not stopped at the required position";
            fcslog.error(message);
            //TODO
            //throw new ErrorCommandException(message);
        }

        return message;

    }

    /**
     * What has to be done for each tick of the timer.
     * We have to publish on the status bus to refresh the GUI. 
     */
    @Override
    public void tick() {
        this.publishData();
    }

    /**
     * This method has no action on hardware.
     * This method updates the socketAtStandby at standby position when a filter is put on the carousel
     * at standby position.
     * A filter must be detected in the socketAtStandby at standby position (socketAtStandby not empty).
     * 
     * @param filter
     * @throws ErrorInCommandExecutionException 
     * @throws BadCommandException 
     */
    private synchronized void putFilterOnCarousel(Filter filter) throws ErrorInCommandExecutionException, BadCommandException, FcsHardwareException {

        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + ": there is no socket at standby position to put a filter in.");
        }


        socketAtStandby.putFilterOnSocket(filter);
        StatusDataPublishedByCarousel status = getStatusData();
        this.publishData();
        setChanged();
        notifyObservers(new ValueUpdate(publishedByCarouselOutputName, status));


    }


    /*
     * Removes a filter from carousel. This method has no action on hardware.
     * It updates the objects Socket and Filter.
     * A filter can be removed from Carousel only when the socketAtStandby 
     * is at standby position.
     */
    public synchronized void removeFilterFromCarousel(Filter filter) throws BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {

        if (filter == null) {
            throw new IllegalArgumentException(getName()
                    + " : removeFilterFromCarousel must have an argument not null");
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + " : "
                    + " : removeFilterFromCarousel can't remove the filter because there is no socket at standby position");
        }

        if (socketAtStandby.getFilter() == null) {
            throw new BadCommandException(getName() + " : "
                    + " : removeFilterFromCarousel can't remove the filter because there is no filter at standby position");
        }

        if (!socketAtStandby.getFilter().equals(filter)) {
            throw new BadCommandException(getName() + " : can't remove a filter which is not at standby position");
        }

        socketAtStandby.removeFilter();
        StatusDataPublishedByCarousel status = getStatusData();
        this.publishData();
        setChanged();
        notifyObservers(new ValueUpdate(publishedByCarouselOutputName, status));

    }

    //TODO exception if filter is not on Carousel ?
    public double getFilterPosition(Filter filter) {
        double filterPosition = 0;
        for (int ix = 0; ix < sockets.length; ix++) {
            if (sockets[ix].getFilter() == filter) {
                filterPosition = sockets[ix].getPosition();
            }
        }
        return filterPosition;
    }

    //TODO throw an exception if filter is not on carousel.
    public double getStandbyPositionForFilter(Filter filter) {
        double position = 0;
        for (CarouselSocket socket : sockets) {
            if (socket.getFilter() == filter) {
                position = socket.getStandbyPosition();
                return position;
            }

        }
        return position;
    }

    //TODO exception if filter is not on Carousel ?
    /**
     *
     * @param filter
     * @return the position of the filter in the sockets array
     * I'm not sure this method is useful actually.
     * Only used is FcsMainModule in method printFilters.
     */
    public int getSocketNumber(Filter filter) {
        int socketNumber = 0;
        for (int ix = 0; ix < sockets.length; ix++) {
            if (sockets[ix].getFilter() == filter) {
                socketNumber = ix;
            }
        }
        return socketNumber;
    }

    /**
     * This method rotates the carousel to put the given filter in front of the autochanger
     * (standby position).
     * This method can't be launched by the user from the console.
     * It computes the shortest way for the carousel. So the running way can be positive or
     * negative.
     * This could be more complicated if the carousel was not able to rotate
     * more than TBD turns in one way.
     * @param filter
     * @return 
     * @params filter : the filter can not be null and must be on the carousel
     * @throws BadCommandException
     */
    @Command ( level=Command.ENGINEERING1, description="Rotate carousel to put filter at standby position", type=Command.CommandType.ACTION) 
    protected String moveFilterToStandby(Filter filter) throws BadCommandException {

        //assert filter is on carousel
        if (!filter.isOnCarousel()) {
            throw new IllegalArgumentException("filter: " + filter.getName()
                    + " is not on carousel.");
        }
        String message = " [CarouselModule]";
        double requiredPosition = getStandbyPositionForFilter(filter);


        if (state == ModuleState.RUNNING) {
            throw new BadCommandException("Carousel is running, stop it before sending a new command.");
        }

        if (getPosition() == requiredPosition) {
            fcslog.info(filter.getName() + " is at STANDBY position on carousel.");
            return (filter.getName() + " is at STANDBY position on carousel.");
        }

//        //if carousel is locked, we have to ReleaseBrake it before running.		
//        if (this.brake.isLocked()) {
//            fcslog.info("Carousel is locked. Unlocking carousel");
//            releaseBrake();
//        }

        double angle = addAngle(getPosition(), getFilterPosition(filter));

        //
        RunningWay runningWay;
        if (angle < 180) {
            runningWay = RunningWay.NEGATIVE;
        } else {
            angle = 360 - angle;
            runningWay = RunningWay.POSITIVE;
        }
        //Release The clamps contact before the rotation
        releaseClampsContact();
        //rotate the carousel
        rotate(angle, runningWay, requiredPosition);
        //engage the clamps contact when it's stopped at standby position
        engageClampsContact();
        //lock the carousel to hold it in position
        return message += engageBrake();
    }

    /**
     * Print for every socket on carousel, the name of the filter it contains.
     */
    public void printSockets() {

        for (int ix = 0; ix < sockets.length; ix++) {
            fcslog.info("Socket number: " + ix + " socket position: " + sockets[ix].getPosition()
                    + " contains filter: " + sockets[ix].getFilter().getName());

        }
    }

    public String grabFilterAtStandby(Object filterName) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        FcsMainModule fcsMainModule = (FcsMainModule) this.getModule("main");
        fcsMainModule.controlFilterName(filterName);

        Filter filterToGrabbe = fcsMainModule.getFilterByName((String) filterName);

        // the entered filter name is correct
        fcslog.info("Filter to move : " + filterName);
        fcslog.info("Filter location : " + filterToGrabbe.getFilterLocation());
        return getName() + grabFilterAtStandby(filterToGrabbe);
    }

    /**
     * This method has to be executed when a filter has just been moved on the carousel 
     * at standby position by the autochanger. It updates the carousel socketAtStandby at standby or throw 
 an exception.
 
 This method tests if the clamps are Locked on a filter at standby position 
 The clamps Locks automaticaly, and the filter presence sensor tells us if the filter is here or not.
 If the clamps are Locked on a filter at standby position, this method updates the socketAtStandby 
 at standby position : it puts the filter on the socketAtStandby.
     * @param filter
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public synchronized String grabFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        if (filter == null) {
            throw new IllegalArgumentException(getName() + ": grabbeFilterAtStandby must receive a non null argument as a filter");
        }
        if (this.isRotating()) {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't grabbe a filter while rotating");
        }
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't be executed because there is no socket at standby position.");
        }

        socketAtStandby.updateClampsStateWithSensors();

//        fcslog.info("ClampX- / filter presence sensor = " + getClampXminus().isFilterEngaged());
//        fcslog.info("ClampX+ / filter presence sensor = " + getClampXplus().isFilterEngaged());

        if ((socketAtStandby.isEmpty())) {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't be executed because no filter is detected in socket at standby position.");
        }


        if ((socketAtStandby.isClampedOnFilter())) {
            fcslog.info(getName() + ": grabbing " + filter.getName() + " at standby position.");
            putFilterOnCarousel(filter);
            fcslog.info(getName() + ": " + filter.getName() + " is grabbed on carousel.");
            String ack = getName() + ": " + filter.getName() + " is grabbed at standby position";
            return ack;
        } else {
            throw new BadCommandException(getName() + ": grabbeFilterAtStandby can't be executed because the clamps are not CLAMPED ON FILTER.");
        }





    }

    public void ungrabFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        if (filter == null) {
            throw new IllegalArgumentException("Filter to ungrabbe should not be null");
        }
        //            if (!autochangerLatchesAreLocked())
        //                throw new BadCommandException("Carousel can't ungrabbe filter if it's not held by autochanger");
        CarouselSocket socketAtStandby = getSocketAtStandby();

        if (socketAtStandby == null) {
            throw new BadCommandException(getName() + ":A socket has to be halted at standby position for this operation.");
        }


        if (!isOnStandby(filter)) {
            throw new BadCommandException(getName() + ": filter has to be at standby position to be ejected");
        }

        if (!socketAtStandby.isClampedOnFilter()) {
            throw new BadCommandException(getName() + ": can't ungrabbe filter which is already unclamped.");
        }

        fcslog.info(getName() + ": ungrabbing " + filter.getName() + " at standby position.");

        //TODO test if the filter is held by the autochanger trucks.

        // The clamps can be opened.
        this.unlockClamps(filter);
        fcslog.info(getName() + ": socket at standby state : " + socketAtStandby.toString());


        fcslog.info(getName() + ": " + filter.getName() + " is ungrabbed from carousel.");
    }
    
    
    
    @Override
    @Command ( level=Command.ENGINEERING1, description="Release clamps at standby position to get ready to clamp again", type=Command.CommandType.ACTION) 
    public String releaseClamps() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

            CarouselSocket socketAtStandby = getSocketAtStandby();
            if (socketAtStandby == null) {
                throw new BadCommandException(getName() + 
                        ":Can't release clamps while a socket is not halted at standby position.");
            }

            socketAtStandby.updateClampsStateWithSensors();
            if (!socketAtStandby.isEmpty()) {
                throw new BadCommandException(getName() + 
                        ":Can't release clamps while "
                        + "a filter is in the socket at standby.");
            }

            if (socketAtStandby.isUnclampedEmpty()) {
                socketAtStandby.releaseClamps();
            } else {
                throw new BadCommandException(getName() + ":Can't release clamps when both clamps are not in state UNCLAMPEDEMPTY.");
            }

            socketAtStandby.updateClampsStateWithSensors();

            if (socketAtStandby.isReadyToClamp()) {
                return "Clamps are released and ready to clamp again.";
            } else {
                throw new ErrorInCommandExecutionException("Could not release clamps.");
            }
    }
    
    /**
     * This methods ReleaseBrakes the 2 clamps at standby position, without Thread.sleep
     * @param filter
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Override
    public synchronized String unlockClamps(Filter filter)
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {

        CarouselSocket socketAtStandby = getSocketAtStandby();
        if (socketAtStandby == null) {
            throw new BadCommandException("Can't unlock clamps while a socket is not halted at standby position.");
        }
        
        socketAtStandby.updateClampsStateWithSensors();
        
        if (!autochanger.isHoldingFilterAtStandby()) {
            throw new BadCommandException("Can't unlock clamps if the filter is not held by the autochanger.");
        }

        if (socketAtStandby.isClampedOnFilter()) {
            fcslog.info("Unlocking clamps at standby.");
            socketAtStandby.unlockClamps();

        } else {
            throw new BadCommandException("Can't unlock clamps when a filter is not clamped at standby.");
        }

        socketAtStandby.updateClampsStateWithSensors();


        //After the execution of the unlockClampsWithThreadSleep method in case of success.
        if (socketAtStandby.isUnclampedOnFilter()) {
            fcslog.info("Just about to remove filter from carousel");
            removeFilterFromCarousel(filter);
            fcslog.info("Command unlockClamps completed");
            return "Clamps unlocked";

        } else {
            fcslog.info(state);
            throw new ErrorInCommandExecutionException(getName() + ":Could not unlock filter at standby");

        }
    }
    
    //for ENGINEERING MODE
    @Command ( level=Command.ENGINEERING3, description="Unlock the clamps", 
            type=Command.CommandType.ACTION)
    public  String unlockClamps() throws BadCommandException, FcsHardwareException, ErrorInCommandExecutionException  {
        return unlockClamps(this.getFilterAtStandby());
    }

    public synchronized boolean isOnStandby(Filter filter) {
        return this.getPosition() == this.getStandbyPositionForFilter(filter);
    }

    /**
     * Add 2 angles : an angle is a double which value is between -360 and +360
     * @param angle1
     * @param angle2
     * @return result of the angles addition. Always > 0
     */
    public static double addAngle(double angle1, double angle2) {
        //if ( abs(angle1) > 360 || abs(angle2) > 360) throw new AssertionError("Adding ");

        double angle = angle1 + angle2;
        if (angle >= 360) {
            angle -= 360;
        }
        if (angle < 0) {
            angle += 360;
        }
        return angle;
    }

    public void releaseClampsContact() {
        //TODO;
    }

    public void engageClampsContact() {
        //TODO ;
    }
}
