/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class represents a model for the 2 Latches of the autochanger.
 * The Latches hold a filter on the trucks.
 * There is one latch on each side of the filter.
 * Because the ACTION  open (or close) for 2 latches is different from 
 * opening (or closing) one Latch, then the other one, it's more convenient 
 * to have a class to represent 2 Latches at a time.
 * @author virieux
 */
public class AutochangerTwoLatches extends MobileItemModule {
    
    AutochangerLatchModule latchXminus;
    AutochangerLatchModule latchXplus;
    
    volatile LockStatus lockStatus;
    volatile FilterPresenceStatus filterPresenceStatus;
    
    
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the latches state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    public AutochangerTwoLatches(String moduleName, int aTickMillis,
            AutochangerLatchModule latchXminus,
            AutochangerLatchModule latchXplus) {
        super(moduleName, aTickMillis);
        this.latchXminus = latchXminus;
        this.latchXplus = latchXplus;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if the 2 latches are initialized.")
    boolean isInitialized() {
        return latchXminus.isInitialized() && latchXplus.isInitialized();
    }

    boolean isInError() {
        return this.lockStatus == LockStatus.ERROR;
    }

    public LockStatus getLockStatus() {
        return lockStatus;
    }

    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }
    
    
    
    public EPOSController getLatchXminusController() {
        return latchXminus.getLatchController();
    }
    
     public EPOSController getLatchXplusController() {
        return latchXplus.getLatchController();
    }   

    @Override
    public boolean isHardwareReady() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        latchXminus.checkHardware();
        latchXplus.checkHardware();
        return TreeWalkerDiag.HANDLING_CHILDREN;       
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    /**
     * Update latches lockStatus from an array of hexa values read from the sensors.
     * @param readHexaValues 
     */
    void updateStateWithSensors(String[] readHexaValues) {
                lock.lock();
            
        try {
            updatingState = true; 
            this.latchXminus.updateStateWithSensors(readHexaValues);
            this.latchXplus.updateStateWithSensors(readHexaValues);

            boolean inError = (this.latchXminus.getLockStatus() == FcsEnumerations.LockStatus.ERROR) || 
                                    (this.latchXplus.getLockStatus() == FcsEnumerations.LockStatus.ERROR);

            boolean locked = (this.latchXminus.getLockStatus() == FcsEnumerations.LockStatus.LOCKED) && 
                                    (this.latchXplus.getLockStatus() == FcsEnumerations.LockStatus.LOCKED);

            boolean unlocked = (this.latchXminus.getLockStatus() == FcsEnumerations.LockStatus.UNLOCKED) && 
                                    (this.latchXplus.getLockStatus() == FcsEnumerations.LockStatus.UNLOCKED);

            boolean inTravel = (this.latchXminus.getLockStatus() == FcsEnumerations.LockStatus.INTRAVEL) && 
                                    (this.latchXplus.getLockStatus() == FcsEnumerations.LockStatus.INTRAVEL);                

            if (inError) {
                this.lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (locked) {    
                this.lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                this.lockStatus = LockStatus.UNLOCKED;
            } else if (inTravel) {
                this.lockStatus = LockStatus.INTRAVEL;
            } else {
                this.lockStatus = LockStatus.UNKNOWN;
            }
            
            //filter presence
            if (this.latchXminus.isEmpty() && this.latchXplus.isEmpty()) {
                filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            } else if (!this.latchXminus.isEmpty() && !this.latchXplus.isEmpty()) {
                filterPresenceStatus = FilterPresenceStatus.ENGAGED;
            } else {
                filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
            }
            
        
        } finally {
            
            updatingState = false;
            stateUpdated.signal(); 
            lock.unlock();
            this.publishData();
        } 
        
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    public StatusDataPublishedByAutochangerTwoLatches getStatusData() {
        return FcsUtils.createStatusDataPublishedByTwoLatches(this);
    }


    @Override
    public void publishData() {
        this.publish("autochangerLatches", getStatusData());
    }




    
}
