package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This class is used with the final hardware in which we have 
 * a clamp online to hold the filter at online position.
 * It's not used for Single Filter Test but is used for testbenchCPPM.
 * @author virieux
 */
public class AutoChangerModule extends Module implements HardwareController {
    
    private static final Logger fcslog = FcsUtils.log;
    
    private BridgeToHardware bridge;
    private final String plutoGatewayName;
    private PlutoGatewayModule plutoGateway;
    
    private final NumericSensor loaderConnectedSensor0;
    private final NumericSensor loaderConnectedSensor1;
    
    AutoChangerTrucksModule trucks;
    AutochangerTwoLatches latches;
    //AutochangerLatchModule latchXminus;
    CarouselModule carousel;
    LoaderModule loader;
    
    private boolean loaderConnected;
    
    private boolean loaderConnectedSensorsInError;
    
    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;    

    public AutoChangerModule(String name, int tickMillis, 
            String plutoGatewayName, 
            NumericSensor loaderConnectedSensor0, 
            NumericSensor loaderConnectedSensor1, 
            AutoChangerTrucksModule trucks,
            AutochangerTwoLatches latches) {
            //AutochangerLatchModule latch) {
        super(name, tickMillis);
        this.plutoGatewayName = plutoGatewayName;
        this.loaderConnectedSensor0 = loaderConnectedSensor0;
        this.loaderConnectedSensor1 = loaderConnectedSensor1;
        this.trucks = trucks;
        this.latches = latches;
//        this.latchXminus = latch;
        this.loaderConnected = false;
        this.loaderConnectedSensorsInError = false;
    }
    
    

    public NumericSensor getLoaderConnectedSensor0() {
        return loaderConnectedSensor0;
    }

    public NumericSensor getLoaderConnectedSensor1() {
        return loaderConnectedSensor1;
    }

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Return false if the 2 redondant loader sensors are equal.")
    public boolean isLoaderConnectedSensorsInError() {
        return loaderConnectedSensorsInError;
    }
    
    //for the simulator
    public int getHandoffPosition() {
        return this.trucks.getHandoffPosition();
    }
     //for the simulator
    public int getOnlinePosition() {
        return this.trucks.getOnlinePosition();
    }
    //for the simulator
    public int getStandbyPosition() {
        return this.trucks.getStandbyPosition();
    }  

    public AutochangerTwoLatches getLatches() {
        return latches;
    }
    

    public AutoChangerTrucksModule getTrucks() {
        return trucks;
    }

    public PlutoGatewayModule getPlutoGateway() {
        return plutoGateway;
    }
    
     /**
     * Returns the boolean field loaderConnected. 
     * If the loaderConnected boolean is being updated and waits 
     * for a response from a sensor, this methods waits until loaderConnected is updated.
     * If the field loaderConnected is not being updated, it returns immediatly the field loaderConnected.
     * 
     * @return atHandoff
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the loader is connected to the camera. "
                        + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return loaderConnected && !loaderConnectedSensorsInError;
            
        } finally {
            lock.unlock();
        }
    }
    
    @Override    
    public void initModule() {
        this.plutoGateway = (PlutoGatewayModule) this.getModule(plutoGatewayName);
        this.bridge = (BridgeToHardware) this.getModule("bridge");
        this.carousel = (CarouselModule) this.getModule("carousel");
        this.loader = (LoaderModule) this.getModule("loader");
    }
    
    

    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="Return true if all autochanger hardware is initialized and "
                    + "bridge is hardwareReady.")
    public boolean isHardwareReady() {       
        return bridge.isHardwareReady() && isInitialized();
    }
 
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized() &&
                trucks.isInitialized();
//                trucks.isInitialized() &&
//                //latches.isInitialized();
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        fcslog.debug(name + " checking hardware.");
        try {
            this.plutoGateway.initializeHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true,ex);
        } 
        trucks.checkHardware();
        //latches.checkHardware();
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    @Override
    public void checkStarted() throws HardwareException {
        fcslog.info(name + " BEGIN checkStarted");
        
        //check that all hardware is booted and identified.
        bridge.getTcpProxy().checkNewHardware();
        
        try {
            this.plutoGateway.initializeHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true,ex);
        }
        trucks.checkHardware();
    
    }

    /** 
     * This method is executed during a shutdown of the subystem to prevent a shutdown 
     * when some hardware is moving. Nothing to do here because all the moving hardware 
     * of the autochanger is represented by the class MobilItemModule where the method
     * checkStopped is implemented.
     * 
     * @throws HardwareException 
     */
    @Override
    public void checkStopped() throws HardwareException {
    }
    
    @Command (type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Check if the latches can be opened.")
    public boolean checkPreConditionsForOpeningLatches() throws SDORequestException, 
            BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        fcslog.info(name + " checking pre-conditions for opening latches");
        
        updateStateWithSensors();
       
        
//        if (!latches.isInitialized()) {
//            String msg = name  + ": latches are not intialized. PLease initialize hardware first.";
//            log.error(msg);
//            throw new BadCommandException(msg);
//        }
//                
//        if (latches.isInError()) {
//            String msg = name  + ": latches are in ERROR state - can't open latches.";
//            log.error(msg);
//            throw new FcsHardwareException(msg);
//        }
        
        if (isEmpty()) {
            String msg = name  + ": no filter in autochanger - can't open latches.";
            log.error(msg);
            throw new FcsHardwareException(msg);
        }
        
        if (!trucks.isAtStandbyPosition() && !trucks.isAtHandoffPosition()) {
            String msg = name  + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open clamp.";
            log.error(msg);
            throw new BadCommandException(msg);
        }
        
        if (trucks.isAtStandbyPosition() && !carousel.isHoldingFilterAtStandby()) {
            String msg = name  + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter "
                    + "- can't open latches.";
            log.error(msg);
            throw new BadCommandException(msg);
        }
        
        if (trucks.isAtHandoffPosition() && !loaderIsHoldingFilterAtHandoff()) {
            String msg = name  + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter "
                    + "- can't open latches.";
            log.error(msg);
            throw new BadCommandException(msg);
        }
        
        if (trucks.isAtStandbyPosition() && carousel.isHoldingFilterAtStandby()) {
            return true;
        } else return trucks.isAtHandoffPosition() && loaderIsHoldingFilterAtHandoff();
           
    }
    
    @Command (type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Check if the latches can be closed.")
    public boolean checkPreConditionsForClosingLatches() throws FcsHardwareException, BadCommandException {
        
        updateStateWithSensors();
       
        
//        if (!latches.isInitialized()) {
//            String msg = name  + ": latches are not intialized. PLease initialize hardware first.";
//            log.error(msg);
//            throw new BadCommandException(msg);
//        }
//        
//        if (latches.isInError()) {
//            String msg = name  + ": latches are in ERROR state - can't close latches.";
//            log.error(msg);
//            throw new FcsHardwareException(msg);
//        }
        
        if (isEmpty()) {
            String msg = name  + ": no filter in autochanger - can't open latches.";
            log.error(msg);
            throw new FcsHardwareException(msg);
        }
        
        if (!trucks.isAtStandbyPosition() && !trucks.isAtHandoffPosition()) {
            String msg = name  + ": autochanger is not "
                    + "at handoff position neither at standby - can't close latches.";
            log.error(msg);
            throw new BadCommandException(msg);
        } else return true;
    }

    boolean loaderIsHoldingFilterAtHandoff() throws BadCommandException, 
            FcsHardwareException {
        if (loaderConnected) {
            return this.loader.isHoldingAFilter();
        }
        return false;
    }
    
     /**
     * This methods updates the trucks, latches and online clamps state in reading all the sensors.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Update clamp state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        
        if (!plutoGateway.isBooted())
            throw new FcsHardwareException(name + ": plutoGateway not booted - can't read sensors");
        
        if (!plutoGateway.isInitialized())
            throw new FcsHardwareException(name + ": plutoGateway not initialized - can't read sensors");
        
        lock.lock();       
        try {
                updatingState = true; 
                this.plutoGateway.updateValues();
                String[] readHexaValues = this.plutoGateway.getHexaValues();
                fcslog.finest("readHexaValues[0]=" + readHexaValues[0]);
                fcslog.finest("readHexaValues[1]=" + readHexaValues[1]);

                //update of loaderConnectedSensors
                this.loaderConnectedSensor0.updateValue(readHexaValues);
                this.loaderConnectedSensor1.updateValue(readHexaValues);

                this.loaderConnected = loaderConnectedSensor0.getDigitalValue() == 1 
                        && loaderConnectedSensor1.getDigitalValue() == 1;

                this.loaderConnectedSensorsInError = !(loaderConnectedSensor0.getDigitalValue() 
                        == loaderConnectedSensor1.getDigitalValue());
                
                this.trucks.updateStateWithSensors(readHexaValues);
                //this.latches.updateStateWithSensors(readHexaValues);
        
        } finally {

            updatingState = false;
            stateUpdated.signal(); 
            lock.unlock();
            this.publishData();
            this.trucks.publishData();
            //this.latches.publishData();
        } 
    }

    boolean isAtHandoff() {
        return trucks.isAtHandoffPosition();
    }

    boolean isEmpty() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void goToHandOff() throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        trucks.goToHandOff();
    }

    void grabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void ungrabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void moveFilterToHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void grabFilterAtStandby(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    
    @Override
    public void tick() {
        //Don't execute here methods which take a lock.
        this.publishData();
    }
    
     public StatusDataPublishedByAutoChanger getStatusData() {
        StatusDataPublishedByAutoChanger status = FcsUtils.createStatusDataPublishedByAutoChanger(this);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        StatusDataPublishedByAutoChanger status = this.getStatusData();
        this.publish("autochangerGeneral", status);
    }   






    

}
