/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import java.util.Observable;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module.ValueUpdate;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.singlefiltertest.config.SingleFilterTestConfig;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * @author virieux
 */
public class SftMainModule extends MainModule {

    private final Filter dummyFilter;
    private SftCarouselModule carousel;
    private SftAutoChangerModule autochanger;
    private boolean filterLocated;
   
    

    public SftMainModule(String aName, int aTickMillis, BridgeToHardware bridge, 
            Filter dummyFilter) {
        super(aName, aTickMillis, bridge);
        this.dummyFilter = dummyFilter;
    }
    
    
    
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
     /**

     * @return the dummyFilter
     */
    public Filter getDummyFilter() {
        return dummyFilter;
    }
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    @Override
    public void initModule() {
        
        super.initModule();
        carousel = (SftCarouselModule) this.getModule("carousel");
        autochanger = (SftAutoChangerModule) this.getModule("autochanger");
        hardwareReady = false;
        filterLocated=false;
    }
    
    @Override
    
    public void tick() {
        //TODO this has to be done outside tick method
        //in the postStart() method in core-1.6.0.
        if (this.isHardwareReady() && !filterLocated) {
            try {
                this.locateFilter();
                filterLocated = true;
                fcslog.info("Filter has been located. Hardware is ready to receive commands.");
                fcslog.debug("DEBUGGING");
            } catch (BadCommandException | ErrorInCommandExecutionException | FcsHardwareException ex) {
                filterLocated = false;
                fcslog.error(ex);
                this.getSubsystem().raiseAlarm(ex.toString());
            }
        }
    }

    
    @Command ( level=Command.NORMAL, description="Read the sensors and update the location of the filter and the state of the carousel and the autochanger.", type=Command.CommandType.QUERY)  
    public void locateFilter() throws FcsHardwareException, ErrorInCommandExecutionException, BadCommandException {
        //TODO that is OK for the simulator not for real hardware
//        carousel.getSockets()[0].putFilterOnSocket(dummyFilter);
//        carousel.engageClampsContact();
        fcslog.debug("MODE DEBUG");
        carousel.updateClampsStateWithSensors();
        if (carousel.getClampXminus().getClampState().equals(FilterClampState.CLAMPEDONFILTER)
                || carousel.getClampXminus().getClampState().equals(FilterClampState.UNCLAMPEDONFILTER)) {
            carousel.getSockets()[0].putFilterOnSocket(dummyFilter);
        }
        autochanger.updateLatchesStateWithSensors();

 
        if (!autochanger.isTrucksEmpty()) {
            autochanger.setFilterOnTrucks(dummyFilter);
        }
        
        autochanger.updateTrucksLocationWithSensors();
        
    }
    
    @Command ( level=Command.ENGINEERING1, description="Check if the carousel is ready to unlock the clamps in a safe way for the filter.", type=Command.CommandType.QUERY, alias="checkUnlock")  
    public void checkPreConditionsForUnlockClamps() throws FcsHardwareException, BadCommandException, ErrorInCommandExecutionException {
        this.updateStateWithSensors();
        if (carousel.isHoldingFilterAtStandby() && (!autochanger.isHoldingFilterAtStandby())) {
            throw new ErrorInCommandExecutionException("Can't unlock clamps if filter is not held by autochanger.");
        }
    }


    @Command ( level=Command.ENGINEERING1, description="To display at a console the status of the hardware.", type=Command.CommandType.QUERY)
    public String displayData() {
        StringBuilder sb = new StringBuilder("Data published by ");
        sb.append(getSubsystem().getName()).append("\n  ");
        fcslog.info("Carousel status=" + carousel.toString());
        sb.append(carousel.toString());sb.append("\n  ");
        sb.append("clampXminus status=");sb.append(carousel.getClampXminus().toString());
        sb.append("\n  ");
        sb.append("clampXplus status=");sb.append(carousel.getClampXplus().toString());
        sb.append("\n  ");
        fcslog.info("autochanger status=" + autochanger.toString());
        sb.append(autochanger.toString());
        sb.append("\n  ");
        sb.append("latchXminus status=");sb.append(autochanger.getLatchXminus().toString());
        sb.append("\n  ");
        sb.append("latchXplus status=");sb.append(autochanger.getLatchXplus().toString());
        sb.append("\n  ");
        return sb.toString();
    }

    /**
     * Initial state for sequence1 :
     * autochanger is in highposition and trucks are empty
     * carousel holds a filter and its clamps are in CLAMPEDONFILTER state.
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     * @throws org.lsst.ccs.messaging.BadCommandException
     */
    @Command ( level=Command.ENGINEERING1, description="Move the hardware to the initial state : ", type=Command.CommandType.ACTION)
    public String goToInitialState() throws FcsHardwareException, ErrorInCommandExecutionException, BadCommandException {
        this.updateStateWithSensors();
        String ack = "Hardware is at initial state and ready for a sequence1";
        if (carousel.isHoldingFilterAtStandby()) {
                if (autochanger.isAtStandback()) {
                    return ack;
                } else if (autochanger.isAtStandby()) {
                    fcslog.info("=======> Autochanger moving back empty to online position.");
                    autochanger.ungrabFilterAtStandby();      
                    return ack;
                } else {
                    return "Autochanger trucks are in the midle, I don't know what to do.";
                }
   
        } else if (autochanger.isHoldingFilterAtStandback()) {
                if (!carousel.isReadyToGrabAFilterAtStandby()) {
                    carousel.releaseClamps();
                }
                fcslog.info("=======> Autochanger trucks moving back the filter to standby position.");
                autochanger.moveFilterToStandby(dummyFilter);
                
                // carousel checks if the filter is clamped at standby position
                fcslog.info("=======> Carousel is grabbing filter at standby position.");
                carousel.grabFilterAtStandby(getDummyFilter());

                //then the autochanger can unclamp the filter and go to swapout position
                fcslog.info("=======> Autochanger moving back empty to online position.");
                autochanger.ungrabFilterAtStandby();
                return ack;
                               
        } else if (autochanger.isHoldingFilterAtStandby()) {
                //in this case the carousel doesn't hold the filter
                //autochanger trucks hold the filter and move it to Online Position
                fcslog.info("=======> Autochanger trucks about to move loaded with the filter to standback position.");
                autochanger.moveFilterToStandback(dummyFilter);

                fcslog.info("=======> Carousel about to release Clamps.");
                carousel.releaseClamps();

                //then we go back to the initial position :
                //autochanger move back the filter in standby position
                fcslog.info("=======> Autochanger trucks moving back the filter to standby position.");
                autochanger.moveFilterToStandby(dummyFilter);


                // carousel checks if the filter is clamped at standby position
                fcslog.info("=======> Carousel is grabbing filter at standby position.");
                carousel.grabFilterAtStandby(getDummyFilter());

                //then the autochanger can unclamp the filter and go to swapout position
                fcslog.info("=======> Autochanger moving back empty to online position.");
                autochanger.ungrabFilterAtStandby();
                return ack;
            
        } else if (!autochanger.isAtStandback() && !autochanger.isAtStandby()) {
                return "Autochanger trucks are in the midle, I don't know what to do.";
        } else {
                return "Nobody holds the filter : I don't know how to go to initial state.";
        }
    }

    /**
     * This method is the main sequence of commands that we want to test on the Single Filter Test.
     * It executes this sequence of commands once.
     * From the initial position (autochanger trucks empty in the higher position) :
     * <UL> <LI> the autochanger trucks go down along the rails to standby position, 
     *      <LI> the autochanger grabs the filter at standby position, 
     *      <LI> the carousel ungrabs the filter at standby position,
     *      <LI> the autochanger moves the filter to the higher position,
     *      <LI> the carousel releases the clamps to be ready to clamp a filter again,
     *      <LI> the autochanger moves back the filter at standby position,
     *      <LI> the carousel checks if the filter is clamped at standby position,
     *      <LI> the autochanger ReleaseBrakes the filter latches,
      <LI> the autochanger trucks go empty to the higher position.
     * </UL>
     * @return a message for the operator
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.CanOpenException
     */
    @Command ( level=Command.NORMAL, description="The world famous sequence 1", 
            type=Command.CommandType.ACTION, alias="es1")    
    public String executeSequence1() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {

        //autochanger trucks go to standby position to grab the filter held by carousel
        fcslog.info("=======> Autochanger trucks about to move empty to standby position.");
        autochanger.grabFilterAtStandby(dummyFilter);
        
        if (haltRequired.get()) return getName() + " Execution of Sequence1 HALTED at step 1";

        // carousel clamps are opened and go in the READYTOCLAMP state
        // when autochanger moves up the filter
        fcslog.info("=======> Carousel is ungrabbing filter at standby position.");
        carousel.ungrabFilterAtStandby(dummyFilter);
        
        if (haltRequired.get()) return getName() + " Execution of Sequence1 HALTED at step 2";

        //autochanger trucks hold the filter are move it to Online Position
        fcslog.info("=======> Autochanger trucks about to move loaded with the filter to standback position.");
        autochanger.moveFilterToStandback(dummyFilter);
        

        fcslog.info("=======> Carousel about to release Clamps.");
        carousel.releaseClamps();
        
        if (haltRequired.get()) return getName() + " Execution of Sequence1 HALTED at step 3";

        //then we go back to the initial position :
        //autochanger move back the filter in standby position
        fcslog.info("=======> Autochanger trucks moving back the filter to standby position.");
        autochanger.moveFilterToStandby(dummyFilter);


        // carousel checks if the filter is clamped at standby position
        fcslog.info("=======> Carousel is grabbing filter at standby position.");
        carousel.grabFilterAtStandby(getDummyFilter());
        
        if (haltRequired.get()) return getName() + " Execution of Sequence1 HALTED at step 4";

        //then the autochanger can unclamp the filter and go to swapout position
        fcslog.info("=======> Autochanger moving back empty to online position.");
        autochanger.ungrabFilterAtStandby();

        fcslog.info("=======> Execution of Sequence1 completed.");
        return getName() + " Execution of Sequence1 completed";
    }

    /**
     * This method will execute nb times the Sequence1 : the parameter nb
     * is given by the user at the console.
     * @param nb : number of times we want to execute the Sequence1
     * @return 
     * @throws org.lsst.ccs.messaging.BadCommandException 
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.CanOpenException 
     */
    @Command ( level=Command.NORMAL, description="The world famous sequence 1 (with number of repetitions)", type=Command.CommandType.ACTION, alias="es1")    
    public String executeSequence1(
            @Argument (name="nExecutions", description="Number of executions")                    
            int nb) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {


        if ((nb < 1) || (nb > 20000)) {
            throw new IllegalArgumentException("nb has to be comprise between 1 and 20000");
        }

        for (int i = 1; i < nb+1; i++) {

            fcslog.info("===========================BEGIN execution Sequence1 nb=" + i + " ================");
            executeSequence1();
            fcslog.info("===========================END execution Sequence1 nb=" + i + " ================");
        }
        return getName() + ":" + nb + " executions of Sequence1 completed";
    }

    boolean autochangerLatchesAreLocked() {
        throw new UnsupportedOperationException("Not yet implemented");
    }



    public Filter getFilterAtStandby() {
        return carousel.getFilterAtStandby();
    }

    @Override
    public void processUpdate(Observable source, ValueUpdate v) {

        //Unuseful : this is already done in BasicAutoChangerModule and in CarouselModule
        //carousel.sendToStatus(carousel.getStatusData());       
        //autochanger.sendToStatus(autochanger.getStatusData());
    }
    
    public String printFilters() {
        StringBuilder sb = new StringBuilder(getName() + " printFilters \n ");
        sb.append(dummyFilter.getName());
        sb.append(" position: ").append(this.carousel.getFilterPosition(dummyFilter));
        sb.append(" socket number: ").append(this.carousel.getSocketNumber(dummyFilter));
        sb.append(" standby position: ").append(this.carousel.getStandbyPositionForFilter(dummyFilter));
        fcslog.info(sb.toString());
        return sb.toString();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n").append(carousel.toString());
        sb.append("\n").append(autochanger.toString());
        return sb.toString();
    }

    /**
     * This command updates the state of the carousel and of the autochanger in reading the sensors.
     * It has to be used in Engineering mode at the console when we send commands step by step.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws ErrorInCommandExecutionException
     * @throws BadCommandException
     */
    @Command ( level=Command.ENGINEERING1, description="Read the sensors and update the state of the carousel and the autochanger.", 
            type=Command.CommandType.QUERY)
    public void updateStateWithSensors() throws FcsHardwareException, ErrorInCommandExecutionException, BadCommandException {
        carousel.updateStateWithSensors();
        autochanger.updateStateWithSensors();
    }
    
    @Override
    public TreeWalkerDiag signal(Signal signal) {
//        fcslog.info("Received:" + signal.getLevel().toString());
        switch(signal.getLevel()) {
            case HALT :
                fcslog.info("HALT required");
                this.haltRequired.set(true);
                break;
            case INTERRUPT1 :
                break;
            case INTERRUPT2 :
                break;
            case RE_START :
                this.haltRequired.set(false);
                break;
            case STOP :
                fcslog.info("STOP required");
                this.haltRequired.set(true);
                break;
        }
        //return TreeWalkerDiag.STOP;
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }

    public SftCarouselModule getCarouselModule() {
        return carousel;
    }

    public SftAutoChangerModule getAutoChangerModule() {
        return autochanger;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, description="The GUIs need that for the initialization.")
    public SingleFilterTestConfig getFullState() {
        return FcsUtils.createSingleFilterTestConfig(this);
    }


    
}
