/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.NumericSensor;

/**
 * This is a model for the numeric sensors we have in single-filter-test and the loader,
 * to monitor the latches in the autochanger,
 * the autochanger trucks position in single-filter-test 
 * and the loader hooks.
 * 
 * 
 * This is also used in the loader test bench where the sensors are connected to a 
 * Pluto PLC for the camera protection system.
 * We read the sensors through a Gateway (CanOpenPlutoGateway) byte by byte.
 * So we need to know on which byte this sensor is coded.
 * 
 * @author virieux
 */
public class CanOpenNumericSensor extends NumericSensor {
    
     /**
     * This is the name of the DIO in the groovy file.
     * Used in the toString method, just for tracing purpose when the subsystem starts 
     * from groovy configuration.
     * 
     */
    String dioName;
    
    /**
     * Numero of input chanel on the device where this sensor is plugged.
     */
    int inputNumero;
    
     /**
     * Gateway byte number where the sensor is coded.
     * value = 1 or 2 in january 2014
     * value = 0,1,2,3,4,5 in december 2014
     */
    int byteNumero;

    public CanOpenNumericSensor(String aName, String dioName, int inputNumero) {
        super(aName);
        this.dioName = dioName;
        this.inputNumero = inputNumero;
    }

    public CanOpenNumericSensor(String aName, String dioName, int inputNumero, int byteNumero) {
        super(aName);
        this.dioName = dioName;
        this.inputNumero = inputNumero;
        this.byteNumero = byteNumero;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public int getByteNumero() {
        return byteNumero;
    }

    @ConfigChanger
    public void setByteNumero(int byteNumero) {
        this.byteNumero = byteNumero;
    }
    
    
    
   
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    public int getInputNumero() {
        return inputNumero;
    }

    @ConfigChanger
    public void setInputNumero(int inputNumero) {
        this.inputNumero = inputNumero;
    }
    

    /**
     * This methods updates the digital value for this sensor.
     * It takes as an argument the value in hexa sent by the CompactIOModule.
     * This hexa value represents the 8 values for the 8 channels.
     * To retrieve the value for this sensor, the hexa value is transformed in binary, 
     * then reverse and the digital value is the digit at the index= inputNumero.
     * exmaple if hex = 80, and inputNumero=8, digital value=1 otherwise digital value=0.
     * @param hex 
     */
    @Override
    public synchronized void updateValue(String hex) {

        fcslog.finest(name + " INPUT NO=" + inputNumero);
        
        String bin = Integer.toBinaryString(Integer.parseInt(hex, 16));
        
        //to add leading zeros on the left
        String s = String.format("%08d", Integer.parseInt(bin));
        fcslog.finest(name + " binary value=" + s);
        
        //to retrieve the digital value for this sensor
        //(the digits are given from left to right
        //1010000 => DIO[8]=1,DIO[7]=0,DIO[6]=1,DIO[5]=0,DIO[4]=0,DIO[3]=0,DIO[2]=0,DIO[1]=0
        //int myValue = Character.digit(s.charAt(8-inputNumero), 16);
        int myValue = Character.digit(s.charAt(7-inputNumero), 16);
        fcslog.finest(name + " Value updated=" + myValue);

        this.digitalValue = myValue;
    }
    
     /**
     * Update the field digitalValue from the bytes read on the gateway.
     * @param hexaValues 
     */
    @Override
    public synchronized void updateValue(String[] hexaValues) {
        //this.updateValue(hexaValues[byteNumero - 1]);
        this.updateValue(hexaValues[byteNumero]);
    }
    
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING1)
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/DIO module=");sb.append(this.dioName);
        sb.append("/Input numero=");sb.append(String.valueOf(this.inputNumero));
        sb.append("/byteNumero=");sb.append(String.valueOf(this.byteNumero));
        sb.append("/digitalValue=");sb.append(String.valueOf(this.digitalValue));
        return sb.toString();
    }




    
    
    
}
