/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.atomic.AtomicBoolean;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.config.CanOpenProxyConfig;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.tracers.Tracer;

/**
 * This is the Main Module for every software related to the Filters Exchanger :
 * - single-filter-test control-command
 * - scale 1 prototype control-command
 * - real Filters Exchanger
 * This class is designed to gather the commons methods to all these softwares.
 *
 * @author virieux
 */
public abstract class MainModule extends Module {
    
    protected static final Logger fcslog = FcsUtils.log;
    protected BridgeToHardware bridge;
    protected boolean hardwareReady;
    protected AtomicBoolean haltRequired;
    protected AtomicBoolean stopRequired;

    

    public MainModule(String aName, int aTickMillis, BridgeToHardware bridge) {
        super(aName, aTickMillis);
        this.hardwareReady = false;
        this.bridge = bridge;
    }

    public AtomicBoolean getHaltRequired() {
        return haltRequired;
    }

    public AtomicBoolean getStopRequired() {
        return stopRequired;
    }
    
    

    @Override
    public void initModule() {
        haltRequired = new AtomicBoolean(false);
        stopRequired = new AtomicBoolean(false);
        String msg = controlRealHardware() ? "Real Hardware" : "Simulation";
        fcslog.info(name + "==>" + msg);
    }
    
    @Override
    public void postStart() throws HardwareException  {
        log.fine(name + ":postStart begins.");
        try {
            this.updateStateWithSensors();
        } catch (FcsHardwareException | ErrorInCommandExecutionException | BadCommandException ex) {
            throw new HardwareException(true,ex.getMessage());
        }
        //this has to stay at the end of the method postStart dixit Bernard.
        log.fine(name + ":postStart ends.");
        startTicking();
    }
    
    /**
     * This methods updates the boolean hardwareReady from the hardware bridge.
     * But when the hardware is ready, it doesn't not ask hardware bridge again to avoid
     * too much messages on the log bus, could be replace by :
     * return this.bridge.isHardwareReady();
     * @return true if hardware is booted, identified and initialized
     */
    @Command ( level=Command.ENGINEERING1, description="Return true if hardware is booted, identified and initialized.", type=Command.CommandType.QUERY)
    public boolean isHardwareReady() {
        if (hardwareReady) {
            return this.hardwareReady;
        } else {
            this.hardwareReady = this.bridge.isHardwareReady();
            return this.hardwareReady;
        }
    }
    
    @Command ( type=Command.CommandType.QUERY, description="Return true if this main module controls real hardware.")
    public boolean controlRealHardware() {
        return this.bridge.isRealHardware();
    }
    
    /**
     * This method returns a String with the list of hardware that this bridge manages.
     * @return 
     */
    @Command ( level=Command.ENGINEERING1, description="Return the list of CANopen hardware "
            + "that this subsystem manages.", type=Command.CommandType.QUERY)
    public String listHardware() {
        return this.bridge.getTcpProxy().listHardware();
    }
    
    public void updateStateWithSensors() throws FcsHardwareException, 
            ErrorInCommandExecutionException, BadCommandException {
        getTcpProxy().publishData();
    }
    
     /**
     * For the Can Open Hardware GUI.
     * @return 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="The GUIs need that for the initialization.")
    public CanOpenProxyConfig getCanOpenHardwareFullState() {
        return FcsUtils.createCanOpenProxyConfig(bridge.getTcpProxy());
    }
    
    public CanOpenProxy getTcpProxy() {
        return this.bridge.getTcpProxy();
    }
    
    
    @Override
    public TreeWalkerDiag signal(Signal signal) {
        switch(signal.getLevel()) {
            case HALT :
                fcslog.debug("HALT required");
                this.haltRequired.set(true);
                break;
            case STOP :
                fcslog.debug("STOP required");
                this.stopRequired.set(true);
                break;
            case INTERRUPT1 :
                break;
            case INTERRUPT2 :
                break;
            case RE_START :
                this.haltRequired.set(false);
                this.stopRequired.set(false);
                break;
        }
        //we want the signal to go to all my children.
        return TreeWalkerDiag.GO;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
         description="For tests purpose - raise an alarm.")
    public void simulateAlarm(String alarmMsg) {
        log.debug(name + "An Alarm was raised:" + alarmMsg);
        this.getSubsystem().raiseAlarm(alarmMsg);
    }
    
    @Override
    public String toString() {
        return super.toString();
    }
    
    //To retrieve the FCS version number.
    //To use with a properties : org.lsst.ccs.traces=VERSION and option : -ea
    static {
        assert Tracer.version("$Rev$", MainModule.class, "org-lsst-ccs-subsystems-fcs");
    }
    




    
}
