/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.CurrentCommandContext;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.HardwareController;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This is the model for the loader in the Filter Exchange System. 
 * The loader is used to load a filter into the camera or to unload a filter from the camera.
 * @author virieux
 */
public class LoaderModule extends Module implements HardwareController {
    
    private static final Logger fcslog = FcsUtils.log;
    
    private final LoaderCarrierModule carrier;
    private final LoaderClampModule clamp;
    private BridgeToHardware bridgeToLoader;
    
    private final NumericSensor filterPresenceSensor0;
    private final NumericSensor filterPresenceSensor1;
    private final NumericSensor loaderOnCameraSensor0;
    private final NumericSensor loaderOnCameraSensor1;
    private NumericSensor autochangerHoldingFilterSensor;

    
    private final String plutoGatewayName;
    private PlutoGatewayModule plutoGateway;
    
    private boolean autochangerHoldingFilter;
    private boolean connectedOnCamera;
    private boolean empty;
    private boolean filterPresenceSensorsInError;
    private boolean loaderOnCameraSensorsInError;


    
    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

   
    public LoaderModule(String moduleName, int aTickMillis,
            LoaderCarrierModule carrier, 
            LoaderClampModule clamp,
            String plutoGatewayName,
            NumericSensor filterPresenceSensor0, 
            NumericSensor filterPresenceSensor1,
            NumericSensor loaderOnCameraSensor0,
            NumericSensor loaderOnCameraSensor1) {
        super(moduleName, aTickMillis);
        this.carrier = carrier;
        this.clamp = clamp;
        this.plutoGatewayName = plutoGatewayName;
        this.empty = false;
        this.connectedOnCamera = false;
        this.autochangerHoldingFilter = false;
        this.filterPresenceSensorsInError = false;
        this.loaderOnCameraSensorsInError = false;
        this.filterPresenceSensor0 = filterPresenceSensor0;
        this.filterPresenceSensor1 = filterPresenceSensor1;
        this.loaderOnCameraSensor0 = loaderOnCameraSensor0;
        this.loaderOnCameraSensor1 = loaderOnCameraSensor1;
    }
    
    Boolean isHaltRequired() {
        return ((MainModule)this.getModule("main")).getHaltRequired().get();       
    }
    
    Boolean isStopRequired() {
        return ((MainModule)this.getModule("main")).getStopRequired().get();
    }

    public NumericSensor getFilterPresenceSensor0() {
        return filterPresenceSensor0;
    }
    
    public NumericSensor getFilterPresenceSensor1() {
        return filterPresenceSensor1;
    }

    public NumericSensor getLoaderOnCameraSensor0() {
        return loaderOnCameraSensor0;
    }
    
    public NumericSensor getLoaderOnCameraSensor1() {
        return loaderOnCameraSensor1;
    }

    public NumericSensor getAutochangerHoldingFilterSensor() {
        return autochangerHoldingFilterSensor;
    }

    public boolean isFilterPresenceSensorsInError() {
        return filterPresenceSensorsInError;
    }

    public boolean isLoaderOnCameraSensorsInError() {
        return loaderOnCameraSensorsInError;
    }
    
 
    
    /**
     * To publish data we don't want to read again the sensors.
     * @return 
     */
    public boolean isEmptyToPublish() {
        return empty;
    }
    
     /**
     * To publish data we don't want to read again the sensors.
     * @return 
     */
    public boolean isOnCameraToPublish() {
        return connectedOnCamera;
    }
    
     /**
     * To publish data we don't want to read again the sensors.
     * @return 
     */
    public boolean isAutochangerHoldingFilterToPublish() {
        return this.autochangerHoldingFilter;
    }
    
    //for the Loader GUI
    public EPOSController getHooksController() {
        return (EPOSController) this.getModule("hooksController");
    }
    
    //for the Loader GUI
    public EPOSController getCarrierController() {
        return (EPOSController) this.getModule("carrierController");
    }
    
    @Override    
    public void initModule() {
        this.plutoGateway = (PlutoGatewayModule) this.getModule(plutoGatewayName);
        this.autochangerHoldingFilterSensor = (NumericSensor) this.getModule("autochangerHoldingFilterSensor");
        this.bridgeToLoader = (BridgeToHardware) this.getModule("bridge");
    }
    
    public boolean isHardwareReady() {       
        return bridgeToLoader.isHardwareReady() && isInitialized();
    }
    
    public boolean isInitialized() {
        return plutoGateway.isInitialized() &&
                ((EPOSController) this.getModule("hooksController")).isInitialized() &&
                ((EPOSController) this.getModule("carrierController")).isInitialized();
    }
    
     /**
     * Returns the boolean field empty. 
     * If the empty boolean is being updated and waits 
     * for a response from a sensor, this methods waits until empty is updated.
     * If the field empty is not being updated, it returns immediatly the field empty.
     * 
     * @return empty
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if there is no filter in the loader. "
                        + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return empty;
            
        } finally {
            lock.unlock();
        }
    }
    
     /**
     * Returns the boolean field atHandoff. 
     * If the atHandoff boolean is being updated and waits 
     * for a response from a sensor, this methods waits until atHandoff is updated.
     * If the field atHandoff is not being updated, it returns immediatly the field atHandoff.
     * 
     * @return atHandoff
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the loader is connected on the camera. This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return connectedOnCamera;
            
        } finally {
            lock.unlock();
        }
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the autochanger is holding the filter. This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter() {
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return autochangerHoldingFilter;
            
        } finally {
            lock.unlock();
        }
    }   
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="Return true if a filter is there and it is held by the clamp.")
    public boolean isHoldingAFilter() throws SDORequestException, BadCommandException, 
            FcsHardwareException {
        this.updateStateWithSensors();
        this.clamp.updatePosition();
        return !this.isEmpty() && this.clamp.isAtClampedPosition();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="Return true if the clamp or the carrier are in action.")
    public boolean isMoving() {
        return this.clamp.isMoving() || this.carrier.isMoving();
    }
    

    public LoaderClampModule getClamp() {
        return this.clamp;
    }

    public LoaderCarrierModule getCarrier() {
        return carrier;
    }
    
    public boolean isCarrierAtStoragePosition() {
        return carrier.isAtStoragePosition();
    }
    
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Initialize loader hardware.")
    public void initializeHardware() throws  HardwareException {
        checkHardware();               
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        
        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true,ex);
        }
        clamp.checkHardware();
        carrier.checkHardware();
        
        try {
            clamp.initializeHardware();
        } catch (BadCommandException | ErrorInCommandExecutionException | FcsHardwareException ex) {
            throw new HardwareException(true,ex);
        }

        if (carrier.isEmpty() && clamp.isLocked()) {
            String msg = name  + ": carrier is empty and clamp is LOCKED - can't start.";
            log.error(msg);
            throw new HardwareException(true,msg);
        }
        
        return TreeWalkerDiag.HANDLING_CHILDREN;   
    }

    /** 
     * CheckStarted is executed by completeInitialization
     * when a first start has failed. 
     * Because end user could have changed many things, we 
     * have to check again that all is correct.
     * @throws org.lsst.ccs.HardwareException
     **/
    @Override
    public void checkStarted() throws HardwareException {
        
        fcslog.info(name + " BEGIN checkStarted");
        
        //check that all hardware is booted and identified.
        bridgeToLoader.getTcpProxy().checkNewHardware();
        
        try {
            this.plutoGateway.initializeAndCheckHardware();
            updateStateWithSensors();
        } catch (FcsHardwareException ex) {
            throw new HardwareException(true,ex);
        }
        clamp.checkHardware();
        carrier.checkHardware();
        
        try {
            clamp.initializeHardware();
        } catch (BadCommandException ex) {
            log.warning(ex);
        } catch (ErrorInCommandExecutionException | FcsHardwareException ex) {
            throw new HardwareException(true,ex);
        }

        if (carrier.isEmpty() && clamp.isLocked()) {
            String msg = name  + ": carrier is empty and clamp is LOCKED - can't start.";
            log.error(msg);
            throw new HardwareException(true,msg);
        }
    }
    
    /**
     * This method is executed during a shutdown of the subystem to prevent a shutdown 
     * when some hardware is moving. Nothing to do here because all the moving hardware 
     * of the loader is represented by the class MobilItemModule where the method
     * checkStopped is implemented.
     * @throws org.lsst.ccs.HardwareException
     **/
    @Override
    public void checkStopped() throws HardwareException {
    }
    
    @Override
    public void tick() {
        //Don't execute here methods which take a lock.
        this.publishData();
        this.carrier.publishData();
        this.clamp.publishData();
    }
    

    


    @Command (type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Check if the carrier can move.")
    public boolean checkPreConditionsForCarrierMotion() throws SDORequestException, 
            BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        fcslog.info(name + " checking pre-conditions for carrier motion");
        
        updateStateWithSensors();
        
        if (!clamp.isInitialized()) {
            String msg = name  + ": clamp is not intialized. PLease initialize hardware first.";
            log.error(msg);
            throw new BadCommandException(msg);
        }
        
        if (clamp.isInError()) throw new BadCommandException(name 
                + " carrier can't move because clamp is in ERROR.");
        
        if (carrier.isEmpty()) return true;

        //here a filter is in the loader       
        if (!this.isConnectedOnCamera()) return true;
        
        if (this.isHoldingAFilter() && isAutochangerHoldingFilter()) {
            throw new BadCommandException(name
                + " carrier can't move because a filter is in the loader"
                + " and it's held by loader AND autochanger.");
            
        } else if (!this.isHoldingAFilter() && !isAutochangerHoldingFilter()) {
            throw new BadCommandException(name
                + " carrier can't move because the filter in the loader is not held"
                + " neither by loader neither by autochanger. "
                + "Close loader clamp or autochanger latches");
            
        } else return true;
           
    }
    

    @Command (type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Check if the clamp can be open.")
    public boolean checkPreConditionsForOpeningHooks() throws SDORequestException, 
            BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        fcslog.info(name + " checking pre-conditions for opening hooks");
        
        updateStateWithSensors();
        
        if (!isConnectedOnCamera()) throw new BadCommandException(name 
                + ": Loader not connected - can't open clamp.");
        
        if (!clamp.isInitialized()) {
            String msg = name  + ": clamp is not intialized. PLease initialize hardware first.";
            log.error(msg);
            throw new BadCommandException(msg);
        }
                
        if (clamp.isInError()) {
            String msg = name  + ": clamp is in ERROR state - can't open clamp.";
            log.error(msg);
            throw new FcsHardwareException(msg);
        }
        
        if (carrier.isEmpty()) return true;
        
        if (!carrier.isAtHandoffPosition()) {
            String msg = name  + ": carrier is loaded with a filter but is not "
                    + "at handoff position - can't open clamp.";
            log.error(msg);
            throw new BadCommandException(msg);
        }
        
        if (!isAutochangerHoldingFilter()) {
            String msg = name  + ": A filter is in the loader but autochanger and "
                    + "not holding it - can't open clamp.";
            log.error(msg);
            throw new BadCommandException(msg);
        } else return true;
           
    }
    
    @Command (type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Check if the clamp can be closed.")
    public boolean checkPreConditionsForClosingHooks() throws SDORequestException, 
            BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        fcslog.info(name + " checking pre-conditions for closing hooks");
        updateStateWithSensors();
        this.clamp.updatePosition();
        
        if (!isConnectedOnCamera()) throw new BadCommandException(name 
                + ": Loader not connected - can't execute close command.");

        if (!clamp.isInitialized()) {
            String msg = name  + ": clamp is not intialized. PLease initialize hardware first.";
            log.error(msg);
            throw new BadCommandException(msg);
        }
        
        if (clamp.isInError()) {
            String msg = name  + ": clamp is in ERROR state - can't execute command open.";
            log.error(msg);
            throw new FcsHardwareException(msg);
        }
        
        if (carrier.isEmpty()) {
            String msg = name  + ": no filter in carrier - can't execute close command for Module "
                    + clamp.getName();
            log.error(msg);
            throw new BadCommandException(msg);
            
        } 
        //at this point carrier is NOT empty
        if (this.clamp.isAtClampedPosition()) {
            String msg = name  + ": a filter is already clamped - can't execute close or clamp command for Module "
                    + clamp.getName();
            log.error(msg);
            throw new BadCommandException(msg);
          
        }
        
 
        return true;
    }
    
    /**
     * This method closes the clamps.
     * When this action is completed the filter is clamped softly on the loader and can't fall.
     * @return a message for the end user.
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Close hooks.")
    public String closeHooks() throws BadCommandException, ErrorInCommandExecutionException, 
            FcsHardwareException {
        return clamp.close();
    }
    
    /**
     * This method closes the clamps strongly.
     * When this action is completed, the filter is clamped softly on the loader and is tight fermely on the carrier.
     * 
     * @return a message for the end user.
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Clamp hooks.")
    public String clampHooks() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        return clamp.clamp();
    }
    
    /**
     * This method opens the clamp.
     * @return a message for the end user.
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command ( level=Command.ENGINEERING1, type=Command.CommandType.ACTION,
            description="Open the hooks")
    public String openHooks() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        return clamp.open();
    }


    
     /**
     * Moves the loader carrier to the handoff position.
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     */
    //TODO check the preconditions
    @Command (type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Move the loader carrier to HANDOFF position")
    public String goToHandoff() throws BadCommandException, FcsHardwareException, SDORequestException, ErrorInCommandExecutionException {
        return carrier.goToHandOff();
    }

    /**
     * Moves the loader carrier to the storage position.
     * @return 
     * @throws org.lsst.ccs.messaging.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException
     */
    @Command ( type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Move the loader carrier to STORAGE position")
    public String goToStorage() throws BadCommandException, FcsHardwareException, 
            SDORequestException, ErrorInCommandExecutionException {
        return carrier.goToStorage();
    }
    
     /**
     * This methods updates the carrier and clamp state in reading all the sensors.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Update clamp state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        
        if (!plutoGateway.isBooted())
            throw new FcsHardwareException(name + ": plutoGateway not booted - can't read sensors");
        
        if (!plutoGateway.isInitialized())
            throw new FcsHardwareException(name + ": plutoGateway not initialized - can't read sensors");
        
        lock.lock();       
        try {
                updatingState = true; 
                this.plutoGateway.updateValues();
                String[] readHexaValues = this.plutoGateway.getHexaValues();
                fcslog.finest("readHexaValues[0]=" + readHexaValues[0]);
                fcslog.finest("readHexaValues[1]=" + readHexaValues[1]);
                this.filterPresenceSensor0.updateValue(readHexaValues);
                this.filterPresenceSensor1.updateValue(readHexaValues);
                this.loaderOnCameraSensor0.updateValue(readHexaValues);
                this.loaderOnCameraSensor1.updateValue(readHexaValues);
                this.autochangerHoldingFilterSensor.updateValue(readHexaValues);
                this.clamp.updateStateWithSensors(readHexaValues);
                this.carrier.updateStateWithSensors(readHexaValues);

                this.empty = filterPresenceSensor0.digitalValue == 0 
                        && filterPresenceSensor1.digitalValue == 0;
                this.connectedOnCamera = loaderOnCameraSensor0.getDigitalValue() == 1 
                        && loaderOnCameraSensor1.getDigitalValue() == 1;
                this.autochangerHoldingFilter = autochangerHoldingFilterSensor.getDigitalValue() == 1;

                this.filterPresenceSensorsInError = !(filterPresenceSensor0.getDigitalValue() 
                        == filterPresenceSensor1.getDigitalValue());
                this.loaderOnCameraSensorsInError = !(loaderOnCameraSensor0.getDigitalValue() 
                        == loaderOnCameraSensor1.getDigitalValue());
        
        } finally {

            updatingState = false;
            stateUpdated.signal(); 
            lock.unlock();
            this.publishData();
            this.clamp.publishData();
            this.carrier.publishData();
        } 
    }
    
        /**
     * This command can be launched when a filter is in the loader
     * and we want to put in inside the camera.
     * @return
     * @throws BadCommandException
     * @throws FcsHardwareException
     * @throws ErrorInCommandExecutionException
     * @throws InterruptedException 
     */
    @Command (type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Load a filter from the loader into the camera.")
    public String loadFilterInCamera() throws BadCommandException, 
            FcsHardwareException, ErrorInCommandExecutionException, 
            InterruptedException {
        if (!isConnectedOnCamera())
            throw new BadCommandException(name 
                    +":loader has to be connected to camera first.");
        if (!isHoldingAFilter()) 
            throw new BadCommandException(name 
                    +":loader is not holding a filter : can't load a filter into camera.");
        if (!isCarrierAtStoragePosition()) //TODO : what do we do in this case ?
            throw new BadCommandException(name 
                    +":carrier loader is not at storage position : can't load a filter into camera.");
        
        //go to Handoff position with the filter
        carrier.goToHandOff();
        if (isHaltRequired()||isStopRequired()) {
            String msg = name + ": received HALT or STOP command.";
            return msg;
        }
        //wait until the autochanger hold the filter
        while (!isAutochangerHoldingFilter()) {
            fcslog.debug(name + " waiting until autochanger holds the filter...");
            Thread.sleep(300);
            updateStateWithSensors();
        }
        
        if (isAutochangerHoldingFilter()) {
            openHooks();
        }
        if (isHaltRequired()||isStopRequired()) {
            String msg = name + ": received HALT or STOP command.";
            return msg;
        }
        
        // empty carrier go to Storage
        return goToStorage(); 
    }
    
    /**
     * This command can be launched when the loader is empty 
     * and we want to take a filter from the camera.
     * A filter must be at handoff position and held by autochanger.
     * @return
     * @throws BadCommandException
     * @throws FcsHardwareException
     * @throws InterruptedException
     * @throws SDORequestException
     * @throws ErrorInCommandExecutionException 
     */
    @Command (type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Unload a filter from the camera into the loader.")
    public String unloadFilterFromCamera() throws BadCommandException, 
            FcsHardwareException, InterruptedException, SDORequestException, 
            ErrorInCommandExecutionException {
        
        updateStateWithSensors();
        if (!isConnectedOnCamera())
            throw new BadCommandException(name +":loader has to be connected to "
                    + "camera first.");
        if (isHoldingAFilter()) 
            throw new BadCommandException(name +":loader is holding a filter : "
                    + "can't unload a filter from camera.");
        
        //wait for the autochanger to put a filter at handoff position
        //in loader test bench it's just a switch to put on.
        while (!isAutochangerHoldingFilter()) {
            fcslog.debug(name + " waiting until autochanger holds the filter...");
            Thread.sleep(300);
            updateStateWithSensors();
        }
        
        //go tho Handoff position - carrier empty
        goToHandoff();    
        if (isHaltRequired()||isStopRequired()) {
            String msg = name + ": received HALT or STOP command.";
            return msg;
        } 
        
        //at handoff position a filter should be here.
        updateStateWithSensors();
        if (isEmpty())
            throw new ErrorInCommandExecutionException(name 
                    +": loader presence filter sensor doesn't detect a filter"
                    + " - can't go on.");
        closeHooks();
        
        //wait for the autochanger to unlock the filter at handoff position
        //in loader test bench it's just a switch to put off.(switch A22)
        while (isAutochangerHoldingFilter()) {
            fcslog.debug(name + " waiting until autochanger releases the filter...");
            Thread.sleep(300);
            updateStateWithSensors();
        }
        
        //close more fermely the hooks to hold the filter.
        clampHooks();
        if (isHaltRequired()||isStopRequired()) {
            String msg = name + ": received HALT or STOP command.";
            return msg;
        }
        
        //carrier holding filter goes to storage position.
        return goToStorage();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, description="List and display info on the hooks.")
    public String listHooks() {
        return this.clamp.listHooks();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display hooks sensors values.")
    public String listSensorsValues() {
        return this.clamp.listSensorsValues();
    }
    
    public StatusDataPublishedByLoader getStatusData() {
        StatusDataPublishedByLoader status = FcsUtils.createStatusDataPublishedByLoader(this);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    public void publishData() {
        StatusDataPublishedByLoader status = this.getStatusData();
        this.getSubsystem().publishStatus("loaderGeneral", status);
    }
    
    //for the simulation 
    public void test() {       
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        fcslog.debug("Command running:" + currentCmdContext.getCommandName() );
        fcslog.info("coming from:" + currentCmdContext.getCommandOriginator());
        this.getSubsystem().interruptActionThread();
    }




    
    

    
}
