
package org.lsst.ccs.subsystems.fcs;


import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import java.util.HashMap;
import java.util.Map;
import java.util.Observable;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterLocation;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenException;

/**
 * The main module in FCS.
 * Commands can be sent to the FcsMainModule from the CCS Master Control Module (MCM)
 * in normal operation mode or from a Console (org.lsst.ccs.subsystem.console.TestConsole).
 * 
 * <P/>Example of commands that can be sent to the FcsMainModule :
 * <BR/>
 * <TT>
 * Console> invoke fcs-test MoveFilterToOnline <i>filtername</i>
 * </TT>
 * <P/>The main goals of the FcsMainModule is :
 * <UL>
 *  <LI>Receive commands from MCM for the whole subsystem.</LI>
 *  <LI>Execute commands received from MCM in invoking methods on the Carousel Module 
 *  on the Autochanger Module and on Loader Module.
 *  </LI>
 *  <LI>Handle the logic of the whole subsystem.</LI>
 *  <LI>Publish on the status bus the status of the whole subsystem.</LI>
 * </UL>
 * @author virieux
 *
 */
public class FcsMainModule extends MainModule {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 7669526660659959402L;
        
	
	private CarouselModule carousel;
	private AutoChangerModule autochanger;
        private LoaderModule loader;
	
	private Double standbyPosition;

	/**
	 * A map to store the filters that FCS manages, by name.
	 * String = filter name
	 */
	private Map<String, Filter> filters = new HashMap<String, Filter>();
	
	private static final String ack = "[FCSMainModule]";	

    public FcsMainModule(String aName, int aTickMillis, BridgeToHardware bridge) {
        super(aName, aTickMillis, bridge);
    }
	
	/**
         * In the initialisation, we want to do some software initialization,
         * and some hardware initialization.
         * We publish the initial data for each piece of hardware : carousel
         * and autochanger.
         */
        @Override
	public void initModule()  {
		fcslog.info("[FcsMainModule] Initializing the FCS Main module ");
                carousel = (CarouselModule) this.getModule("carousel");
		autochanger = (AutoChangerModule) this.getModule("autochanger");
                loader = (LoaderModule) this.getModule("loader");

	}
	

	/**
	 * What to do when the Modules we observe (carousel and autochanger) update
         * their data to publish : we publish on status bus.
	 */
        //Unuseful
	public void processUpdate(Observable source, ValueUpdate v) {
            
                //publishData(v.getName(), v.getValue());
                getSubsystem().publishStatus(v.getName(), v.getValue());
	}
	
	//for tests purpose mainly
        public String printFilters() {
	
            StringBuilder mess = new StringBuilder(ack);
            mess.append(" printFilters %n ");

            for(String key : filters.keySet()) {
                mess.append(filters.get(key).getName());
                mess.append(" ");
                mess.append(filters.get(key).getFilterLocation());
		if (filters.get(key).isOnCarousel()) {
                    mess.append(" position: ").append(this.carousel.getFilterPosition(filters.get(key)));
                    mess.append(" socket number: ").append(this.carousel.getSocketNumber(filters.get(key)));
		    mess.append(" standby position: ").append(this.carousel.getStandbyPositionForFilter(filters.get(key)));
		}
                mess.append(" %n ");
            }
	
            fcslog.info(mess.toString());
            return mess.toString();
	
        }

        /**
         * This method can be invoked from the Console to display at the Console
         * status data of the whole subsystem.
         * @return 
         */
        public String displayData() {
            StringBuilder sb = new StringBuilder("Data published by ");
            sb.append(getSubsystem().getName()).append("\n  ");
            sb.append(carousel.toString()).append("\n  ");
            sb.append(autochanger.toString());
            return sb.toString();
        }

    @Override
        public String toString() {
            StringBuilder sb = new StringBuilder(getName());
            sb.append(carousel.toString());
            sb.append(autochanger.toString());
            return sb.toString();
        }
	

	/**
	 * Move a filter to online position.
	 * The filter must be in the camera. 
	 * It can be on the carousel or on the autochanger.
	 * This method can be called by the end-user at the console.
	 * @param filterName name of the filter to move
	 * @return message for the user
	 * @throws InterruptedException, IllegalArgumentException
	 * @throws BadCommandException 
	 * @throws ErrorInCommandExecutionException 
	 */
	public String moveFilterToOnline(Object filterName) 
		throws InterruptedException, IllegalArgumentException, BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {
	
		controlFilterName(filterName);
		
		// the entered filter name is correct
		fcslog.info("Filter to move : " + filterName);
		fcslog.info("Filter location :" + getFilterByName((String) filterName).getFilterLocation());
		return ack + moveFilterToOnline(getFilterByName((String) filterName));

	}
	
	/**
	 * Move a filter to standby position.
	 * The filter must be in the camera. 
	 * The filter can be on carousel or on autochanger.
	 * This method can be called by the end-user at the console.
	 * @param filterName name of the filter to move
	 * @return message for the user
	 * @throws InterruptedException, IllegalArgumentException
	 * @throws BadCommandException 
	 * @throws ErrorInCommandExecutionException 
	 */	
	public String moveFilterToStandby (Object filterName) 
		throws InterruptedException, IllegalArgumentException, BadCommandException, ErrorInCommandExecutionException {
		
		controlFilterName(filterName);
		
		// the entered filter name is correct
		fcslog.info("Filter to move : " + filterName);
		fcslog.info("Filter location : " + getFilterByName((String) filterName).getFilterLocation());				
		return ack + moveFilterToStandby(getFilterByName((String) filterName));
			
	}

//TODO IN REAL LIFE : rotate carousel, identify each filter on each socket
	private void locateFilters() {

                carousel.getSockets()[0].putFilterOnSocket(filters.get("filterU"));
                carousel.getSockets()[1].putFilterOnSocket(filters.get("filterG"));
                carousel.getSockets()[2].putFilterOnSocket(filters.get("filterR"));
                carousel.getSockets()[3].putFilterOnSocket(filters.get("filterI"));
                carousel.getSockets()[4].putFilterOnSocket(filters.get("filterZ"));

                carousel.engageClampsContact();

                filters.get("filterY").setFilterLocation(FilterLocation.OUT);

	}

	
	/**
	 * A method to move a filter in the camera to online position. 
         * It envolves the carousel Module and the autochanger Module.
         * If the filter we want to move is on the autochanger, the autochanger has to move it to online position.
         * If the filter is on the carousel, and the autochanger truck is loaded with another filter, the autochanger has 
         * to move back the other filter on the carousel at standby position. 
         * If the filter is on the carousel and the autochanger is empty, we have to rotate the carousel 
         * in order to put the required filter at standby position, then the autochanger grabbes the filter on the carousel 
         * and moves it to online position.
	 * @param filter filter to move
	 * @return message for the user
	 * @throws InterruptedException 
	 * @throws InterruptedException 
	 * @throws BadCommandException if the carousel is already rotating or the autochanger is already moving when 
         * the command is sent : we want to receive a command stop before executing a new command.
	 */
	private String moveFilterToOnline(Filter filter) throws InterruptedException, BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {

                //The carousel and the autochanger must not move, otherwise they have to be stopped before a new command.
//                if (carousel.isRotating() || autochanger.isMoving()) 
//                    throw new BadCommandException(getName()
//                        + " Please wait the end of the last command or send a stop command.");
//
//
//		//The filter we want to move is on autochanger.
//		if (filter == autochanger.getFilterOnTrucks()) {
//			return autochanger.moveFilterToOnline(filter);
//		} 
//
//                //The filter we want to move is not on the autochanger.
//
//                //The filter we want to move is on carousel.
//		if (filter.isOnCarousel()) {
//
//                        //a filter is held by the the autochanger
//                        //we have to put it back on the carousel before moving the carousel
//			if (!autochanger.isTrucksEmpty()) {
//				Filter filterOnTrucks = autochanger.getFilterOnTrucks();
//				// another filter in on the autochanger : we have to put it on carousel
//                                // the standby position on carousel must be empty
//                                if (!carousel.isReadyToGrabAFilterAtStandby())
//                                    throw new ErrorInCommandExecutionException(getName() + "/Carousel is not ready to grabbe a filter at standby position");
//				autochanger.moveFilterToStandby(filterOnTrucks);
//				carousel.grabFilterAtStandby(filterOnTrucks);
//				autochanger.ungrabFilterAtStandby();
//			}
//									
//			carousel.moveFilterToStandby(filter);
//			autochanger.grabFilterAtStandby(filter);
//			carousel.ungrabFilterAtStandby(filter);
//			autochanger.moveFilterToOnline(filter);
//                        carousel.releaseClamps();
//			autochanger.closeOnlineClamp();
//                        return(getName() + ": command moveFilterToOnline completed / filter on line: " + filter.getName());
//		}
		
		//assert : we should not go there
		String message = "Panic : filter is neither on carousel, neither on autochanger";
		fcslog.error(message);
		return message;
	}

	/**
	 * Move a filter in the camera to standby positionon the carousel.
         * If the filter is on the carousel, the command is sent to the carousel,
         * if the filter is on the autochanger the command is sent to the autochanger.
         * 
	 * @param filter : filter to move. It can be on the carousel or on the autochanger.
	 * @return message
	 * @throws BadCommandException
	 * @throws InterruptedException
	 * @throws ErrorInCommandExecutionException 
	 */
	private String moveFilterToStandby(Filter filter) throws BadCommandException, InterruptedException, ErrorInCommandExecutionException {
		String message = null;
		
//		if (filter.isOnCarousel()) {						
//			message = carousel.moveFilterToStandby(filter);
//			
//		} else if (filter.isOnAutoChanger()) {
//			message = autochanger.moveFilterToStandby(filter);;
//		
//		}
		return message;		
	}



	public Filter getFilterAtStandby() {
		return carousel.getFilterAtStandby();
	}
		
	/**
         * This method controls if the filter name given at the console by an operator is correct.
         * 
         * @param o the object we receive from the command line at the Console.
         * @throws IllegalArgumentException in this case :
         *   <UL>
         *      <LI>the name is not a String,</LI>
         *      <LI>the name is unknown in the map filters,</LI>
         *      <LI>the name is known in the map filters, but the filter location is unknown,</LI>
         *      <LI>the name is known in the map filters, but the filter is out of the camera.</LI>
         *   </UL>
         */
	protected void controlFilterName(Object o) throws IllegalArgumentException {
					
			if (!(o instanceof String)) {
				throw new IllegalArgumentException("Incorrect filter name : " + o);
			}
		
			String filterName = (String) o;
			
			if (!(filters.containsKey(filterName))) {
				throw new IllegalArgumentException( "Unknown filter name : " + o);
			}
						
			if (getFilterByName(filterName).isUnknown()) { 				
				throw new IllegalArgumentException("Filter: " + o + " is at unknown location.");			
			} 
			
			if (this.getFilterByName(filterName).isOut()) {
				throw new IllegalArgumentException( "Filter: " + o + " is out of the camera.");
			}
	}


	public Filter getFilterByName(String filterName) {
		return filters.get(filterName);
	}

        /**
         * 
         * @return true if the socket at standby position on the carousel is empty (contains no filter)
         */
	public boolean isEmptyStandBy() {
		return false;
	}

        /**
         * 
         * @return true if there is no filter at online position. This can be because :
         * <UL>
         *  <LI>the autochanger truck is not at online position.</LI>
         *  <LI>or the autochanger truck is at online position but it's empty.</LI>
         * </UL>
         */
	public boolean isEmptyOnline() {
		return true;
	}
	


//	protected Filter getFilterOnline() {
//		return autochanger.getFilterOnline();
//	}


	public Map<String, Filter> getFilters() {
		return filters;
	}

	public void setFilters(Map<String, Filter> filters) {
		this.filters = filters;
	}

	public Double getStandbyPosition() {
		return standbyPosition;
	}

	public void setStandbyPosition(Double standbyPosition) {
		this.standbyPosition = standbyPosition;
	}
        
        /**
         * Load a filter from the loader to the camera.
         * At the end of this operation the filter is grabbed on the autochanger at Handoff position.
         * @param aFilter 
         */
        public void loadFilter(Filter aFilter) throws BadCommandException, 
                ErrorInCommandExecutionException, FcsHardwareException {
            //TODO check the conditions for this action
            //the camera is in the position "horizontal"
            //the auto changer trucks are empty at "Hand-Off" position
            //the carousel socket at "Stand-by" position is empty
            //The Loader carrier is retracted at "Storage" position and the filter is clamped
            
            if (!carousel.isReadyToGrabAFilterAtStandby())
                throw new BadCommandException(getName() + " can't load a filter when socket at STANDBY position is not empty and ready.");
            
            if (!(autochanger.isAtHandoff() && autochanger.isEmpty()))
                throw new BadCommandException(getName() + " autochanger is not empty at STANDOFF position; can't load a filter.");
            
            connectLoader();
            
            if (!(loader.isHoldingAFilter() && loader.isCarrierAtStoragePosition()))
                throw new BadCommandException(getName() + " can't load filter because loader is not holding a filter at storage position.");
            
            loader.goToHandoff();
            autochanger.goToHandOff();

            autochanger.grabFilterAtHandoff(aFilter);
            aFilter.setFilterLocation(FilterLocation.ONAUTOCHANGER);
            loader.openHooks();
            loader.goToStorage();
            
            disconnectLoader();
            
                    
        }
        
        /**
         * Unload a filter from the camera to the loader.
         * The camera has to be at horizontal position.
         * The filter has to be on Carousel at STANDBY.
         * The trucks of the autochanger have to be empty at STANDOFF position.
         * @param aFilter 
         * @throws org.lsst.ccs.messaging.BadCommandException 
         * @throws org.lsst.ccs.messaging.ErrorInCommandExecutionException 
         * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
         */
        public void unloadFilter(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
            //TODO check the conditions for this action
            //Conditions in which this operation is authorized :
            //the camera is in the position "horizontal"
            //the auto changer trucks are empty at the "Hand-Off" position
            //the carousel socket at standby position has the filter to extract
            //The Loader carrier is empty, in storage position, the loader clamps are unclamped
            if (!aFilter.isOnCarousel()) 
               throw new BadCommandException(getName() + " " + aFilter.getName() + " is not on the carousel, can't unload it.");
            if (!carousel.isOnStandby(aFilter))
               throw new BadCommandException(getName() + " " + aFilter.getName() + " is not on the carousel at standby position, can't unload it.");
            if (!(autochanger.isAtHandoff() && autochanger.isEmpty()))
                throw new BadCommandException(getName() + " autochanger is not empty at STANDOFF position; can't unload a filter.");
            
            this.connectLoader();
            fcslog.info("=======> Autochanger trucks about to move empty to standby position.");
            autochanger.grabFilterAtStandby(aFilter);
            // carousel clamps are opened and go in the READYTOCLAMP state
            // when autochanger moves up the filter
            fcslog.info("=======> Carousel is ungrabbing filter at standby position.");
            carousel.ungrabFilterAtStandby(aFilter);
            //autochanger trucks hold the filter and move it to Online Position
            fcslog.info("=======> Autochanger trucks about to move loaded with the filter to handoff position.");
            autochanger.moveFilterToHandoff(aFilter);
            fcslog.info("=======> Carousel about to release Clamps.");
            carousel.releaseClamps();

                       
            loader.goToHandoff();
            
            loader.closeHooks();
                    
            autochanger.ungrabFilterAtHandoff(aFilter);
                      
            loader.clampHooks();
            
            loader.goToStorage();
            aFilter.setFilterLocation(FilterLocation.OUT);
            
            this.disconnectLoader();           
        }

    private void connectLoader() {
        //this.bridge.getTcpProxy().connectLoader();
    }

    private void disconnectLoader() {
        //this.bridge.getTcpProxy().disconnectHardware();
    }

    @Override
    public void updateStateWithSensors() throws FcsHardwareException, ErrorInCommandExecutionException, BadCommandException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }


}
