
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.singlefiltertest.BasicAutoChangerModule;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterLocation;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.ClampsOrLatchesDisagreeException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;


/**
 * This is a socket on the carousel : there is 5 sockets on a carousel.
 * When a filter is on the carousel, it is attached at a socket.
 * A socket has a clamp on each side : 
 * - clampX- (clampXminus)
 * - clampX+ (clampXplus)
 * clampX+ and clampX- are the same except for the method isLocked()
 * See FilterClampXminusModule and FilterClampXplusModule
 * 
 * The main job of this class CarouselSocket is to synchronize the actions on the 2 clamps.
 * The state of the CarouselSocket is the state of the clamps when both clamps state are 
 * identical. 
 * To synchonize an action (updateSateWithSensor, unlock or release) on both clamps, we use a CyclicBarrier :
 * we start 2 threads and they wait for each other action completion with a CyclicBarrier.
 * 
 * During an action on both clamps, we want to lock the CarouselSocket so no other thread can 
 * try to access this object before the completion of action. We do that with a Lock and Condition.
 * (see java.util.concurrent).
 * 
 * CarouselSocket extends MobileItemModule because we want to lock or release the clamps when the carousel is at standby position
 * and we want to wait until the action is completed. We know if the action is completed in reading the clamps sensors. 
 * So we use the general mechanism provided by the MobileItemModule.
 * 
 * @author virieux
 *
 */
public class CarouselSocket extends MobileItemModule {

    private BridgeToHardware bridge;

    private CarouselClampModule clampXminus;

    private CarouselClampModule clampXplus;
    
    private EPOSController clampActuatorXminus;
    private EPOSController clampActuatorXplus;

    /**
     * socket position (angle) in degrees on the carousel
     * 
     */
    private double position;

    /**
     * carousel position (angle) when this socket is at standby position
     * should be 360 - position, but could be different
     */
    private double standbyPosition;

    /**
     * When the socket holds a filter, this field is initialized whith this filter.
     * When the socket is empty, this field is null.
     */
    private Filter filter;
    
    volatile boolean updatingClamps = false;
//    volatile boolean itemMoving = false;
    private volatile FilterClampState clampsState = FilterClampState.UNDEFINED;
    
    //When we update the 2 clamps, we want to take a lock on the socket until
    //the end of the update, so any other thread has to wait for this update.
//    final Lock lock = new ReentrantLock();
    final Condition stateUpdated = lock.newCondition();
//    final Condition motionCompleted = lock.newCondition();
    
    //TODO : there is also a timeout for each clamp, it's a lot of timeouts !
    //here we could take the maximum of the timeout of the clampXminus and clampXplus.
    //this is done in initModule 
    private long timeoutForUnlocking = 4000;
    private long timeoutForReleasing = 4000;

    public CarouselSocket(String moduleName, int aTickMillis, 
            CarouselClampModule clampXminus, CarouselClampModule clampXplus, 
            double position, double standbyPosition) {
        super(moduleName, aTickMillis);
        this.clampXminus = clampXminus;
        this.clampXplus = clampXplus;
        this.position = position;
        this.standbyPosition = standbyPosition;


    }

    public CarouselClampModule getClampXminus() {
        return clampXminus;
    }

    public void setClampXminus(CarouselClampModule clampXminus) {
        this.clampXminus = clampXminus;
    }

    public CarouselClampModule getClampXplus() {
        return clampXplus;
    }

    public void setClampXplus(CarouselClampModule clampXplus) {
        this.clampXplus = clampXplus;
    }


    public double getPosition() {
            return position;
    }

    public Filter getFilter() {
            return filter;
    }

    private void setFilter(Filter filter) {
            this.filter = filter;
    }

    public double getStandbyPosition() {
            return standbyPosition;
    }

    
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/
    @Override
    public void initModule() {
        this.timeoutForReleasing = java.lang.Math.max(this.clampXminus.timeoutForReleasing, this.clampXplus.timeoutForReleasing);
        this.timeoutForUnlocking = java.lang.Math.max(this.clampXminus.timeoutForUnlocking, this.clampXplus.timeoutForUnlocking);
        this.bridge =  (BridgeToHardware) getModule("bridge");
        this.clampActuatorXminus = (EPOSController) getModule("clampActuatorXminus");
        this.clampActuatorXplus = (EPOSController) getModule("clampActuatorXplus");
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return ((MainModule) this.getModule("main")).isHardwareReady();
    }
    
    
    /**
     * Returns the state of the clamps. 
     * If the state is being updated and waiting 
     * for a response from a sensor, this methods waits until the state is updated.
     * If the state is not being updated, it returns immediatly the state.
     * 
     * @return state
     * 
     */
    public FilterClampState getClampsState() {
        
        lock.lock();
        try {
            while(updatingClamps) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.warning(name + " was interrupted during getClampsState.");
                }

            }
            return clampsState;
            
        } finally {
            lock.unlock();
        }
    }
    
    
    
    /**
     * This method updates the state of the 2 clamps in reading the values sent by the sensors.
     * The values of the sensors are read clamp by clamp with read SDO commands.
     * 
     * @throws HardwareError 
     */
    @Deprecated
    public void updateClampsStateWithSensorsFromSDO() throws FcsHardwareException {

        
        lock.lock();
        this.updatingClamps = true;
        try {
            clampXminus.updateStateWithSensorsFromSDO();
            clampXplus.updateStateWithSensorsFromSDO();
            
            if (clampXminus.getClampState().equals(clampXplus.getClampState()))  {
                    this.clampsState = clampXminus.getClampState();
                    //System.out.println("XXXXXXXXSocket at standby : clamps are updated");

            } else throw new FcsHardwareException("Error in filter presence detection at standby position : "
                        + "the clamps don't agree.");
            
            
        } finally {
            this.updatingClamps = false;
            this.stateUpdated.signal();         
            lock.unlock();
        }
    }
    
    public void updateClampsStateWithSensors(PDOStorage pdoStorage) throws FcsHardwareException {
        lock.lock();
        this.updatingClamps = true;
        try {
            clampXminus.updateStateWithSensors(pdoStorage);
            clampXplus.updateStateWithSensors(pdoStorage);
            
            if (clampXminus.getClampState().equals(clampXplus.getClampState()))  {
                    this.clampsState = clampXminus.getClampState();
                    fcslog.info(name + ":XXXXXXXXSocket at standby : clamps are updated");

            } else throw new ClampsOrLatchesDisagreeException("Error in carousel at standby position : "
                        + "the clamps don't agree.", getName());
            
            
        } finally {
            this.updatingClamps = false;
            this.stateUpdated.signal();         
            lock.unlock();           
        }
        publishData();
    }
    
    public void updateClampsStateWithSensors() throws FcsHardwareException, BadCommandException {
        
        fcslog.debug("updateClampsStateWithSensors/" + name + "/bridge=" + this.bridge.toString());
        updateClampsStateWithSensors(this.bridge.readPDOs());
    }
    


    /*
     * The carousel socket is empty if there is no filter in the socket.
     * That appends when clamps of both side of the socket are empty.
     * A clamp is empty when no filter is engaged in the clamp.
     * @throws HardwareError
     * If the 2 clamps are not in the same state this method throws
     * a HardwareError : this should not append except if 
     * the sensors are broken.
     */
    public boolean isEmpty() throws FcsHardwareException {
            return (getClampsState().equals(FilterClampState.READYTOCLAMP)) 
                    || (getClampsState().equals(FilterClampState.UNCLAMPEDEMPTY));      
    }
   

    public boolean isClampedOnFilter() {
        return getClampsState().equals(FilterClampState.CLAMPEDONFILTER);
    }
    
    public boolean isUnclampedOnFilter() {
        return getClampsState().equals(FilterClampState.UNCLAMPEDONFILTER);
    }

    public boolean isUnclampedEmpty() {
        return getClampsState().equals(FilterClampState.UNCLAMPEDEMPTY);
    }
    
    public boolean isReadyToClamp() {
        return getClampsState().equals(FilterClampState.READYTOCLAMP);
    }
    
     /**
     * Releases the 2 clamps of the socket.
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws HardwareError 
     */
    //TODO 
    public String releaseClamps() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        updateClampsStateWithSensors();
        
        fcslog.info("Checking conditions for release clamp " + getName() + " on socket at standby position.");
        
        if (!getClampsState().equals(FilterClampState.UNCLAMPEDEMPTY)) 
            throw new BadCommandException("Can't release clamps if socket is not unclamped and empty.");
        
        
        
        fcslog.info("Releasing clamp " + getName() + " on socket at standby position.");
        
        return this.executeAction(MobileItemAction.RELEASECLAMPS, timeoutForReleasing);
    }
    

    
    
        /**
     * This method unclamp the 2 clamps at standby position.
     * It turn ON the actuators and wait for the end of the unlocking task.
     * The unlock is completed when the clamps sensors notice that the clamps are UNLOCKED.
     * So this method reads the sensors with the method updateClampsStateWithSensors until the clamps are UNLOCKED or the 
     * duration of the task is greater than the timeout (timeoutForUnlocking)
     * To read the sensors at a regular time schedule we use the ScheduledThreadPoolExecutor (scheduler.scheduleAtFixedRate)
     * from the java.util.concurrent package.
     * @return a message for the end user
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     */
    public String unlockClamps() throws FcsHardwareException, BadCommandException, ErrorInCommandExecutionException {

        updateClampsStateWithSensors();
        
        if (!getClampsState().equals(FilterClampState.CLAMPEDONFILTER)) 
            throw new BadCommandException(name + ":Can't unlock clamps if isn't clamped on filter.");
       
        
        
       
        BasicAutoChangerModule autochanger = (BasicAutoChangerModule) this.environment.getComponentByName("autochanger");
        if (autochanger == null) throw new BadCommandException("NO AUTOCHANGER");
        
        if (!autochanger.isHoldingFilterAtStandby()) throw new BadCommandException("CANNOT UNLOCK CLAMPS if FILTER is not HELD by autochanger.");
        
        return this.executeAction(MobileItemAction.UNLOCKCLAMPS, timeoutForUnlocking);
    }
    

    
    

    
    
    /**
     * Software removing
     * @throws BadCommandException 
     */
    public void removeFilter() throws BadCommandException {
        if (this.filter == null) {
                throw new BadCommandException("Carousel socket : there is no filter to remove at standby position");
        }
        this.filter = null;
    }

    public synchronized void putFilterOnSocket(Filter filter) {
        setFilter(filter);
        filter.setFilterLocation(FilterLocation.ONCAROUSEL);
    }

    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/Filter in socket : ");
        if (getFilter() == null) {
            sb.append(" NO FILTER").append("\n");
        } else {
            sb.append(getFilter().getName()).append("\n");
        }
        return sb.toString();
    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.UNLOCKCLAMPS)) {
            return this.getClampsState().equals(FilterClampState.UNCLAMPEDONFILTER);
        } else if (action.equals(MobileItemAction.RELEASECLAMPS)) {
            return this.getClampsState().equals(FilterClampState.READYTOCLAMP);
        } else {
            throw new IllegalArgumentException("Action on clamps on socket must be UNLOCKCLAMPS or RELEASECLAMPS");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        try {
            this.updateClampsStateWithSensors();
        } catch (ClampsOrLatchesDisagreeException ex) {
            fcslog.info(getName() + "a little delai between the update of the 2 clamps : have to wait ...");
        }
    }

    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        if (action.equals(MobileItemAction.UNLOCKCLAMPS)) {
//              ce qui marchait avant que je change tout dans 
            //CanOpenEPOS
//            this.clampXminus.getActuator().on();
//            this.clampXplus.getActuator().on();
            //ne se termine jamais
            if (!this.clampActuatorXminus.isEnabled()) this.clampActuatorXminus.enable();
            if (!this.clampActuatorXplus.isEnabled()) this.clampActuatorXplus.enable();
//          TODO: A tester en septembre 2015 au CPPM
            if (!clampActuatorXminus.isEnabled()) throw new BadCommandException(name 
                + " ControllerXminus has to be enabled first.");
            if (!clampActuatorXplus.isEnabled()) throw new BadCommandException(name 
                + " ControllerXplus has to be enabled first.");
            this.clampActuatorXminus.writeCurrent(this.clampXminus.getCurrentToUnlock());
            this.clampActuatorXplus.writeCurrent(this.clampXplus.getCurrentToUnlock());
            this.clampActuatorXminus.setOn(true);
            this.clampActuatorXplus.setOn(true);
        } else if (action.equals(MobileItemAction.RELEASECLAMPS)) {
            this.clampActuatorXminus.off();
            this.clampActuatorXplus.off();
        } else throw new IllegalArgumentException("Action on clamps on socket must be UNLOCKCLAMPS or RELEASECLAMPS");
    }

    @Override
    public void postAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        //TODO complete this
        fcslog.info(name + " stopAction : nothing to be done.");
    }

    @Override
    public void publishData() {
        ((CarouselModule)this.getModule("carousel")).publishData();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }






}
