package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.singlefiltertest.BasicAutoChangerModule;
import org.lsst.ccs.subsystems.fcs.common.Thermometer;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.CurrentCommandContext;
import org.lsst.ccs.HardwareException;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * An abstract class which extends Module to model a clamp that holds a filter on the carousel.
 * It implements pattern clampState.
 * See <a href="CarouselClampState.png"> the clampState machine diagram for a carousel clamp.</a>
 * Each clamp on the carousel is coupled with a sensor which detects the presence of a filter.
 * This abstract class provides a method to lock, unlock or release the clamps, and to compute the clampState 
 * of the clamp. 
 * The clampState of the clamp is computed, updated and published on the status bus each tick of the timer.
 * 
 * 
 * @author virieux
 */
public class CarouselClampModule extends MobileItemModule {
    
    //to be able to read sensors from PDO
    private BridgeToHardware bridge;
      
    /**
     * The actuator which controls this clamp.
     */
    //private ClampActuatorModule actuator;
    public EPOSController  actuator;
    
     /**
     * The filterPresenceSensor detects where is the filter within the clamp.
     * @See enum PresenceFilterStateOnClamp
     */
    private final Sensor14bits filterPresenceSensor;
    
    
    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private final Sensor14bits lockSensor;
    
   /**
     * The thermometer mesures the temperature inside the clamp.
    */
    private final Thermometer thermometer;

    protected FilterClampState clampState;
    protected FilterPresenceStatus filterPresenceStatus;
    protected int currentToUnlock;
    protected int currentToMaintainUnlocked;
      
    private LockStatus lockStatus;
    
    protected double temperature;
    
    /**
     * If the lock sensor returns a value between 0 and lockSensorValueA:
     * the sensor is unlocked.
     * If the lock sensor returns a value between lockSensorValueA and lockSensorValueB:
     * we don't know.
     * if the sensor returns a value between lockSensorValueB and ValueC:
     * the clamp is locked.
     * If the sensor returns a value greater than valueC the sensor is in ERROR.
     */
    private int lockSensorValueA;
    private int lockSensorValueB;
    private int lockSensorValueC;
    private int lockSensorOffset;
    
    /**
     * The min and max value for the sensors.
     * Needed by the GUI.
     */
    private static final int lockSensorMinValue = Sensor14bits.minValue;
    private static final int lockSensorMaxValue = Sensor14bits.maxValue;
    
    private static final int filterPositionMinValue = Sensor14bits.minValue;
    private static final int filterPositionMaxValue = Sensor14bits.maxValue;
    
    
    /**
     * if the filter presence sensor returns a value between 0 and filterPositionValueA : 
     * the sensor is in error.
     */
    //@ConfigChanger( doc= "if the filter presence sensor returns a value between 0 and filterPositionValueA : the sensor is in error.")
    private int filterPositionValueA;
    
    /**
     * if filter presence sensor returns a value between filterPositionValueA and filterPositionValueB : 
     * the filter is engaged and lockable.
     */
    private int filterPositionValueB;
    
     /**
     * if filter presence sensor returns a value between filterPositionValueB and filterPositionValueC : 
     * we don't know. And if the value is greater than valueC, there's no filter.
     */   
    private int filterPositionValueC;
    private int filterPositionOffset;
    
    private boolean initialized = false;
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    protected long timeoutForUnlocking = 4000;
    protected long timeoutForReleasing = 4000;

    public CarouselClampModule(String aName, 
            int aTickMillis, 
            Sensor14bits filterPresenceSensor, 
            Sensor14bits lockSensor, 
            Thermometer thermometer, 
            int filterPositionValueA, int filterPositionValueB, int filterPositionValueC, int filterPositionOffset,
            int lockSensorValueA, int lockSensorValueB, int lockSensorValueC, int lockSensorOffset,
            int timeoutForUnlocking, int timeoutForReleasing,
            int currentToUnlock, int currentToMaintainUnlocked
            ) {
        super(aName,aTickMillis);
        this.filterPresenceSensor = filterPresenceSensor;
        this.lockSensor = lockSensor;
        this.thermometer = thermometer;
        this.filterPositionValueA = filterPositionValueA;
        this.filterPositionValueB = filterPositionValueB;
        this.filterPositionValueC = filterPositionValueC;
        this.filterPositionOffset = filterPositionOffset;
        this.lockSensorValueA = lockSensorValueA;
        this.lockSensorValueB = lockSensorValueB;
        this.lockSensorValueC = lockSensorValueC;
        this.lockSensorOffset = lockSensorOffset;
        this.timeoutForUnlocking = timeoutForUnlocking;
        this.timeoutForReleasing = timeoutForReleasing;
        this.currentToUnlock = currentToUnlock;
        this.currentToMaintainUnlocked = currentToMaintainUnlocked;
    }






    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**
     * @return ************************************************************************************************/


    //public ClampActuatorModule getActuator() {
    public EPOSController getActuator() {
        return actuator;
    }

    //because this is used for construction from groovy file description
    //public void setActuator(ClampActuatorModule actuator) {
    public void setActuator(EPOSController  actuator) {
        this.actuator = actuator;
    }

    public int getCurrentToUnlock() {
        return currentToUnlock;
    }
    
    
    
    
    /**
     * @param valueA the filterPositionValueA to set
     */   
    @ConfigChanger
    public void setFilterPositionValueA(int valueA) {
        this.filterPositionValueA = valueA;
    }

    /**
     * @param valueB the filterPositionValueB to set
     */
    @ConfigChanger
    public void setFilterPositionValueB(int valueB) {
        this.filterPositionValueB = valueB;
    }

    /**
     * @param valueC the filterPositionValueC to set
     */
    @ConfigChanger
    public void setFilterPositionValueC(int valueC) {
        this.filterPositionValueC = valueC;
    }






    /**
     * @return the lockSensorValueA
     */
    public int getLockSensorValueA() {
        return lockSensorValueA;
    }

    /**
     * @param aValue the lockSensorValueA to set
     */
    @ConfigChanger
    public void setLockSensorValueA(int aValue) {
        this.lockSensorValueA = aValue;
    }

    /**
     * @return the lockSensorValueB
     */
    public int getLockSensorValueB() {
        return lockSensorValueB;
    }

    /**
     * @param aValue the lockSensorValueB to set
     */
    @ConfigChanger
    public void setLockSensorValueB(int aValue) {
        this.lockSensorValueB = aValue;
    }

    public int getLockSensorValueC() {
        return lockSensorValueC;
    }
    
    @ConfigChanger
    public void setLockSensorValueC(int lockSensorValueC) {
        this.lockSensorValueC = lockSensorValueC;
    }

    public int getFilterPositionMaxValue() {
        return filterPositionMaxValue;
    }



    public int getFilterPositionMinValue() {
        return filterPositionMinValue;
    }



    public int getLockSensorMaxValue() {
        return lockSensorMaxValue;
    }



    public int getLockSensorMinValue() {
        return lockSensorMinValue;
    }

    @ConfigChanger
    public void setLockSensorOffset(int lockSensorOffset) {
        this.lockSensorOffset = lockSensorOffset;
    }

    /**
     * @return the filterPositionValueA
     */
    public int getFilterPositionValueA() {
        return filterPositionValueA;
    }

    /**
     * @return the filterPositionValueB
     */
    public int getFilterPositionValueB() {
        return filterPositionValueB;
    }

    /**
     * @return the filterPositionValueC
     */
    public int getFilterPositionValueC() {
        return filterPositionValueC;
    }

    public int getLockSensorOffset() {
        return lockSensorOffset;
    }

    public int getFilterPositionOffset() {
        return filterPositionOffset;
    }

    public void setFilterPositionOffset(int filterPositionOffset) {
        this.filterPositionOffset = filterPositionOffset;
    }
    
    
    
     /**
     * For encapsulation : we want to access to filterPresenceSensor in sub-packages.
     * @return the filterPresenceSensor
     */
    public Sensor14bits getFilterPresenceSensor() {
        return filterPresenceSensor;
    }
    
        /**
     * @forSpring
     * @return the lockSensor
     */
    public Sensor14bits getLockSensor() {
        return lockSensor;
    }

    public Thermometer getThermometer() {
        return thermometer;
    }

    /**
     * @return the lockStatus
     */
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * @param lockStatus the lockStatus to set
     */
    public void setLockStatus(LockStatus lockStatus) {
        this.lockStatus = lockStatus;
    }



    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public double getTemperature() {
        return temperature;
    }

    public long getTimeoutRelease() {
        return timeoutForReleasing;
    }
    
    @ConfigChanger
    public void setTimeoutRelease(long timeoutRelease) {
        this.timeoutForReleasing = timeoutRelease;
    }

    public long getTimeoutUnlock() {
        return timeoutForUnlocking;
    }

    @ConfigChanger
    public void setTimeoutUnlock(long timeoutUnlock) {
        this.timeoutForUnlocking = timeoutUnlock;
    }
    
   

    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/
    
    @Override
    public void initModule() {
        this.bridge =  (BridgeToHardware) getModule("bridge");
        this.clampState = FilterClampState.UNDEFINED;
        this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
        this.lockStatus = LockStatus.UNKNOWN;
        this.temperature = 0;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return ((MainModule) this.getModule("main")).isHardwareReady();
    }
    
    @Override
    public TreeWalkerDiag checkHardware() throws HardwareException {
        EposMode mode;
        try {
            mode = actuator.readMode();
        } catch (SDORequestException | ShortResponseToSDORequestException ex) {
            String msg = name + " ERROR while reading EPOS controller mode :" + ex.getMessage();
            throw new HardwareException(false,msg);
        }
        if (!mode.equals(EposMode.CURRENT)) 
            throw new HardwareException(false,name + " is not in CURRENT mode.");
        return TreeWalkerDiag.GO;
    }
    

    
        /**
     * Returns the clampState of the clamp. 
     * If the clampState is being updated and waiting 
     * for a response from a sensor, this methods waits until the clampState is updated.
     * If the clampState is not being updated, it returns immediatly the clampState.
     * 
     * @return clampState
     * 
     */

    public FilterClampState getClampState() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.warning(name + ": getClampState interrupted while waiting for update.");
                }

            }
            return clampState;
            
        } finally {
            lock.unlock();
        }
    }

     /**
     * This methods returns true if the clamp is locked.
     * In the simulator, it is overriden.
     * @return true if the clamp is locked
     */
    @Command ( level=Command.ENGINEERING1, description="Returns true if the clamp is locked", type=Command.CommandType.QUERY)
    public boolean isLocked() {
        return (this.getLockStatus().equals(LockStatus.LOCKED));
    }
    
    /**
     * 
     * @return true if filter is engaged on the clamp
     */
    @Command ( level=Command.ENGINEERING1, description="Returns true if filter is engaged on the clamp : filter presence sensors sees it ", type=Command.CommandType.QUERY)
    public synchronized boolean isFilterEngaged() {
        return (this.filterPresenceStatus.equals(FilterPresenceStatus.LOCKABLE));
    }


    
    /**
     * Update the field lockPresenceStatus in reading the filter presence sensor. 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateFilterPresenceStatus() throws FcsHardwareException {
            //READ filter presence sensor
            this.filterPresenceSensor.updateValue();
            int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
            int mechaValue = newFilterPresenceSensorValue - this.filterPositionOffset;
            if (mechaValue < this.filterPositionValueA) {
                    this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    fcslog.error(getName() + " ERROR new read value FOR FILTER POSITION SENSOR  = " 
                            + newFilterPresenceSensorValue);
                    throw new SensorValueOutOfRangeException("FILTER POSITION SENSOR", 
                            this.filterPresenceSensor.getName(),CarouselClampModule.filterPositionMinValue, 
                            this.filterPositionValueA, mechaValue);
                
            } else if (mechaValue < this.filterPositionValueB) {
                this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;     
                
            } else if (mechaValue < this.filterPositionValueC) {
                this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
                
            } else {
                this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            } 
    }
    
    private void updateFilterPresenceStatus(PDOStorage pdoStorage) throws FcsHardwareException {
        int newFilterPresenceSensorValue = filterPresenceSensor.updateValue(pdoStorage);
            int mechaValue = newFilterPresenceSensorValue - this.filterPositionOffset;
            if (mechaValue < this.filterPositionValueA) {
                    this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    fcslog.error(getName() + " ERROR new read value FOR FILTER POSITION SENSOR  = " 
                            + newFilterPresenceSensorValue);
                    throw new SensorValueOutOfRangeException("FILTER POSITION SENSOR", 
                            this.filterPresenceSensor.getName(),CarouselClampModule.filterPositionMinValue, 
                            CarouselClampModule.filterPositionMaxValue, mechaValue);
                
            } else if (mechaValue < this.filterPositionValueB) {
                this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;     
                
            } else if (mechaValue < this.filterPositionValueC) {
                this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
                
            } else {
                this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            } 
    }
    
    /**
     * Update the field lockStatus in reading the lock sensor. 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateLockStatus() throws FcsHardwareException {
        
        //READ lock sensor
        this.lockSensor.updateValue();

        int newLockSensorValue = lockSensor.getValue();
        int mechaValue = newLockSensorValue - this.lockSensorOffset;

        //fcslog.info(getName() + " NEWVALUE FOR LOCK SENSOR=" + newLockSensorValue);

        if ( mechaValue < lockSensorValueA) {
            this.lockStatus = LockStatus.UNLOCKED;

        } else if (mechaValue <= lockSensorValueB){
            this.lockStatus = LockStatus.UNKNOWN;


        } else if (mechaValue <= lockSensorValueC) {
            this.lockStatus = LockStatus.LOCKED;

        } else {
            this.lockStatus = LockStatus.ERROR;
                fcslog.error(getName() + " ERROR new read value FOR LOCK SENSOR  = " + newLockSensorValue);
                throw new SensorValueOutOfRangeException("ERROR in reading LOCK SENSOR", 
                        this.lockSensor.getName(),CarouselClampModule.lockSensorMinValue, CarouselClampModule.lockSensorMaxValue, mechaValue);
        }
        
    }
    
    /**
     * Update lockStatus from parameter pdoStorage
     * @param pdoStorage
     * @throws HardwareError 
     */
    private void updateLockStatus(PDOStorage pdoStorage) throws FcsHardwareException {
        int newLockSensorValue = lockSensor.updateValue(pdoStorage);
        int mechaValue = newLockSensorValue - this.lockSensorOffset;

        //fcslog.info(getName() + " NEWVALUE FOR LOCK SENSOR=" + newLockSensorValue);

        if ( mechaValue < lockSensorValueA) {
            this.lockStatus = LockStatus.UNLOCKED;

        } else if (mechaValue <= lockSensorValueB){
            this.lockStatus = LockStatus.UNKNOWN;


        } else if (mechaValue <= lockSensorValueC) {
            this.lockStatus = LockStatus.LOCKED;

        } else {
            this.lockStatus = LockStatus.ERROR;
                fcslog.error(getName() + " ERROR new read value FOR LOCK SENSOR  = " + newLockSensorValue);
                throw new SensorValueOutOfRangeException("ERROR in reading LOCK SENSOR", 
                        this.lockSensor.getName(),CarouselClampModule.lockSensorMinValue, CarouselClampModule.lockSensorMaxValue, mechaValue);
        }
    }
    
    
    /**
     * This method updates the clamp clampState regarding the value returned by the 
     * filter presence sensor and the value returned by the method isLocked().
     * When the update is completed, it sends a signal to threads waiting to get
     * the new value of clampState.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Deprecated
    public void updateStateWithSensorsFromSDO() throws FcsHardwareException {
        
        lock.lock();
        updatingState = true;   
        
        try {

            this.updateFilterPresenceStatus();
            
            this.updateLockStatus();

            this.clampState = this.computeClampState();
                  
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }
        
        this.publishData();

    }
    
     /**
     * This method updates the clamp clampState regarding the value returned by the 
     * filter presence sensor and the value returned by the method isLocked().
     * When the update is completed, it sends a signal to threads waiting to get
     * the new value of clampState.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.messaging.BadCommandException
     */
    @Command ( level=Command.ENGINEERING1, description="Read clamp sensors and update clampState", type=Command.CommandType.QUERY, alias ="updState")
    public void updateStateWithSensors() throws FcsHardwareException, BadCommandException {
        updateStateWithSensors(this.bridge.readPDOs());
    }
    
    
    //void updateStateWithSensorsFromSDO(HashMap<String, String> sensorsValues) throws HardwareError {
    void updateStateWithSensors(PDOStorage pdoStorage) throws FcsHardwareException {
        lock.lock();
          
        
        try {
            updatingState = true; 

            this.updateFilterPresenceStatus(pdoStorage);
            
            this.updateLockStatus(pdoStorage);

            this.clampState = this.computeClampState();
                  
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();

            
        }
        
        this.publishData();
    }
    
     /**
     * Publish Data on status bus for trending data base and GUIs.
     * 
     */
     public void publishData() {
        StatusDataPublishedByClamp status = this.getStatusData(); 
        //if we change the string "carouselClamp", it has to be changed also in the GUI.
        this.getSubsystem().publishStatus("carouselClamp", status);
     }
   

    
    
    
    /**
     * Compute the global state of the clamp given the lock sensor and the presence filter sensor state.
     * This has to be overriden for the clamp X-.
     * @return clamp state
     */
    public FilterClampState computeClampState() {
        
        
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.ERROR) 
                || (this.lockStatus.equals(LockStatus.ERROR))) 
            return  FilterClampState.ERROR;

        // a filter is in the socket
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.LOCKABLE)) {

            if (this.lockStatus.equals(LockStatus.LOCKED)) {
                return FilterClampState.CLAMPEDONFILTER;

            } else if (this.lockStatus.equals(LockStatus.UNLOCKED)) {
                return  FilterClampState.UNCLAMPEDONFILTER;
            } else {
                return FilterClampState.UNDEFINED;
            }
        }

        // no filter
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.NOFILTER)) {
            if (this.lockStatus.equals(LockStatus.LOCKED)) {
                return  FilterClampState.READYTOCLAMP;

            } else if (this.lockStatus.equals(LockStatus.UNLOCKED)) {
                return  FilterClampState.UNCLAMPEDEMPTY;

            } else {
                return  FilterClampState.ERROR;
            }
        }

        // NOT_LOCKABLE (UNDEFINED)
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.NOT_LOCKABLE)) {
            return FilterClampState.UNDEFINED;
        }
        
        return FilterClampState.UNDEFINED;

    }
    
   

    @Override
    public void tick() {
        publishData();
    }


    public StatusDataPublishedByClamp getStatusData() {
        StatusDataPublishedByClamp status = FcsUtils.createStatusDataPublishedByClamp(this);
        return status;
    }
    
    /**
     * The clamps on the carousel are locked automaticaly when the filter comes
     * at the standby position. To be able to lock automaticaly again, it has to be
     * released after each time it has been unlocked. 
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     */
    @Command ( level=Command.ENGINEERING3, description="Release clamp in order to get ready to clamp a filter again", type=Command.CommandType.ACTION)
    public String release() throws FcsHardwareException, BadCommandException, ErrorInCommandExecutionException {
        fcslog.info("Checking conditions for release clamp " + getName() + " on socket at standby position.");
        
        updateStateWithSensors();
        if (!getClampState().equals(FilterClampState.UNCLAMPEDEMPTY)) 
            throw new BadCommandException("Can't release a clamp if isn't unclamped and empty.");
        
        
        
        fcslog.info("Releasing clamp " + getName() + " on socket at standby position.");
        
        return this.executeAction(MobileItemAction.RELEASE, timeoutForReleasing);
        
    }
    
     /**
     * Unlock the clamp when a filter is locked by the clamp.
     * This is used only in engineering mode.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command ( level=Command.ENGINEERING3, description="Unlock the clamp", type=Command.CommandType.ACTION)
    public String unlock() 
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        
        fcslog.info(getName() + ": "
                + "UNLOCK State1 = " + clampState.toString());
        updateStateWithSensors();
        
        if (!getClampState().equals(FilterClampState.CLAMPEDONFILTER)) 
            throw new BadCommandException("Can't unlock a clamp if it is already unlocked.");
       
        
        
       
        BasicAutoChangerModule autochanger = (BasicAutoChangerModule) this.environment.getComponentByName("autochanger");
        if (autochanger == null) throw new BadCommandException("NO AUTOCHANGER");
        
        if (!autochanger.isHoldingFilterAtStandby()) throw new BadCommandException("CANNOT UNLOCK A CLAMP if FILTER is not HELD by autochanger.");
        
        return this.executeAction(MobileItemAction.UNLOCK, timeoutForUnlocking);
        
    } 
    
    
    @Override
    public void startAction(MobileItemAction action) throws SDORequestException, BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        if (action.equals(MobileItemAction.UNLOCK)) {
            this.actuator.enable();
            this.actuator.writeCurrent(this.currentToUnlock);
            this.actuator.setOn(true);
        } else if (action.equals(MobileItemAction.RELEASE)) {
            this.actuator.off();            
        } else throw new IllegalArgumentException("Action on clamp must be UNLOCK or RELEASE");
        
    }
    
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.UNLOCK)) {
            return this.getClampState().equals(FilterClampState.UNCLAMPEDONFILTER);
        } else if (action.equals(MobileItemAction.RELEASE)) {
            return this.getClampState().equals(FilterClampState.READYTOCLAMP);
        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK or RELEASE");
        }
    }
    
    @Override
    public void postAction(MobileItemAction action) {
    }
    
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws FcsHardwareException, BadCommandException {
        this.updateStateWithSensors();
    }
    
    /**
     * This methods read the thermometer, update the field temperature and returns the value sent
     * by the thermometer;
     * @return 
     */
    public double updateTemperature() throws FcsHardwareException {
        double temp = this.thermometer.readTemperature();
        this.temperature = temp; 
        return temp;
    }
    
    @Override
    public String toString() {
//        StringBuilder sb = new StringBuilder(super.toString());
//        sb.append("Name=");sb.append(this.getName());
        //sb.append("/actuator=");sb.append(String.valueOf(this.actuator.getName()));
        StringBuilder sb = new StringBuilder(this.getName());
        sb.append("/filterPresenceSensor=");sb.append(this.filterPresenceSensor.getName());
        sb.append("#valueA=");sb.append(this.filterPositionValueA);
        sb.append("#valueB=");sb.append(this.filterPositionValueB);
        sb.append("#valueC=");sb.append(this.filterPositionValueC);
        sb.append("/lockSensor=");sb.append(this.lockSensor.getName());
        sb.append("#valueA=");sb.append(this.lockSensorValueA);
        sb.append("#valueB=");sb.append(this.lockSensorValueB);
        sb.append("#valueC=");sb.append(this.lockSensorValueC);
        return sb.toString();
    }

    //TODO check with Guillaume what is to be done to stop the actions.
    @Override
    public void abortAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        fcslog.debug("Command running:" + currentCmdContext.getCommandName() );
        fcslog.info("coming from:" + currentCmdContext.getCommandOriginator());
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

   










    
	
}
