/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.HardwareException;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.messaging.BadCommandException;
import org.lsst.ccs.messaging.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.ShortResponseToSDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for online clamp which holds a filter
 * when it is at ONLINE position.
 * The online clamps are part of the autochanger subsystem.
 * There is 3 online clamps.
 * @author virieux
 */
public class AutochangerOnlineClampModule extends MobileItemModule {
    
    AutoChangerModule autochanger;
    EPOSController controller;
    final NumericSensor lockSensor;
    final NumericSensor lockSensorC;//lock sensor complementary 
    final NumericSensor unlockSensor;
    final NumericSensor unlockSensorC;//unlock sensor complementary
    
    Map<String,Integer> paramsForCurrentToLock;//not used - to delete ?
    Map<String,Integer> paramsForCurrentToUnlock;//not used - to delete ?
    int currentToLock,currentToUnlock,currentToClamp;
    int holdTime = 2000;
    int travelTime = 5000;
    
    boolean locked, unlocked, inError, inTravel;
    boolean lockSensorInError, unlockSensorInError;
    
    FcsEnumerations.LockStatus lockStatus;
    private boolean controllerInFault;

    private volatile boolean initialized;
    private boolean controllerConfigured;
    
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;

    public AutochangerOnlineClampModule(String moduleName, int aTickMillis, 
            EPOSController onlineClampController,
            NumericSensor lockSensor0,
            NumericSensor lockSensor1,
            NumericSensor unlockSensor0,
            NumericSensor unlockSensor1,
            int currentToLock, 
            int currentToUnlock,
            int currentToClamp,
            Map<String, Integer> paramsForCurrentToLock, 
            Map<String, Integer> paramsForCurrentToUnlock,
            int holdTime, 
            int travelTime) {
        super(moduleName, aTickMillis);
        this.controllerConfigured = false;
        this.controller = onlineClampController;
        this.lockSensor = lockSensor0;
        this.lockSensorC = lockSensor1;
        this.unlockSensor = unlockSensor0;
        this.unlockSensorC = unlockSensor1;
        this.paramsForCurrentToLock = paramsForCurrentToLock;
        this.paramsForCurrentToUnlock = paramsForCurrentToUnlock;
        this.currentToLock = currentToLock;
        this.currentToUnlock = currentToUnlock;
        this.currentToClamp = currentToClamp;
        this.holdTime = holdTime;
        this.travelTime = travelTime;
    }

    public int getCurrentToLock() {
        return currentToLock;
    }

    @ConfigChanger
    public void setCurrentToLock(int currentToLock) {
        this.currentToLock = currentToLock;
    }

    public int getCurrentToUnlock() {
        return currentToUnlock;
    }

    @ConfigChanger
    public void setCurrentToUnlock(int currentToUnlock) {
        this.currentToUnlock = currentToUnlock;
    }

    public int getCurrentToClamp() {
        return currentToClamp;
    }

    @ConfigChanger
    public void setCurrentToClamp(int currentToClamp) {
        this.currentToClamp = currentToClamp;
    }

    public int getHoldTime() {
        return holdTime;
    }

    @ConfigChanger
    public void setHoldTime(int holdTime) {
        this.holdTime = holdTime;
    }

    public int getTravelTime() {
        return travelTime;
    }

    @ConfigChanger
    public void setTravelTime(int travelTime) {
        this.travelTime = travelTime;
    }

    public EPOSController getController() {
        return controller;
    }
    
    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    public NumericSensor getLockSensor() {
        return lockSensor;
    }

    public NumericSensor getUnlockSensor() {
        return unlockSensor;
    }

    public NumericSensor getUnlockSensorC() {
        return unlockSensorC;
    }

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if lockSensor and lockSensorC retun the same value.")    
    public boolean isLockSensorInError() {
        return lockSensorInError;
    }

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if unlockSensor and unlockSensorC retun the same value.")
    public boolean isUnlockSensorInError() {
        return unlockSensorInError;
    }

    

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if LockStatus=ERROR, this means that "
                    + "lockSensor or unlockSensor is in ERROR or that"
                    + "unlockSensor and lockSensor return non consistant values.")
    public boolean isInError() {
        return this.getLockStatus().equals(FcsEnumerations.LockStatus.ERROR);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if LockStatus=LOCKED.")
    public boolean isLocked() {
        return this.getLockStatus().equals(FcsEnumerations.LockStatus.LOCKED);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if LockStatus=UNLOCKED.")
    public boolean isUnlocked() {
        return this.getLockStatus().equals(FcsEnumerations.LockStatus.UNLOCKED);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if LockStatus=INTRAVEL.")
    public boolean isInTravel() {
        return this.getLockStatus().equals(FcsEnumerations.LockStatus.INTRAVEL);
    }    

    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if controller is in fault.")
    public boolean isControllerInFault() {
        return controllerInFault;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, 
            description="Returns true if clamp is initialized.")
    public boolean isInitialized() {
        return initialized;
    }
    
    @Override
    public void initModule() {
        super.initModule();
        this.autochanger = (AutoChangerModule) this.getModule("autochanger");
        this.lockStatus = FcsEnumerations.LockStatus.UNKNOWN;
        if (controller == null) {
            fcslog.error(name + "==>>> onlineController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(name + "==>>> null onlineClampController - fix groovy description file.");
        } else {
            //listens to my Controller to detect the controller's faultReset
            //or the emergency messages coming from the controller.
            this.listens((Observable) controller);
        }
    }
    
    @Override
    public void tick() {
        this.publishData();
    }

    @Command(type=Command.CommandType.QUERY, 
            level=Command.ENGINEERING1, 
            description="Returns true if controller is initialized and configured.")
    @Override
    //tested on CPPM testbench in september 2015
    public boolean isHardwareReady() {
        return ((MainModule) this.getModule("main")).isHardwareReady() 
                && controller.isInitialized() 
                && controllerConfigured;
    }
    
    @Command(type=Command.CommandType.QUERY, 
            level=Command.ENGINEERING1, 
            description="Check if the onlineClamp can be locked.")   
    public void checkConditionForUnlocking() throws FcsHardwareException {
        //TODO : check trucks position
        if (!this.isHardwareReady()) 
            throw new FcsHardwareException(name + ":controller not ready - can't be unlocked");
    }
    
    @Command(type=Command.CommandType.QUERY, 
            level=Command.ENGINEERING1, 
            description="Check if the onlineClamp can be locked.")
    public void checkConditionForLocking() throws FcsHardwareException {
        //TODO : check trucks position
        if (!this.isHardwareReady()) 
            throw new FcsHardwareException(name + ":controller not ready - can't locked");
    }
    
     /**
     * This method is called during INITIALIZATION phase.
     * It configures the controller for the digital inputs and outputs.
     * It changes it to CURRENT mode if it's not in this mode and it writes in the CPU 
     * of the controller the values of configuration for CURRENT mode parameters.
     * @return
     * @throws HardwareException 
     */
    @Override
    //tested on CPPM testbench in september 2015
    public TreeWalkerDiag checkHardware() throws HardwareException {
        super.checkHardware();
        
        fcslog.debug(name + " checking hardware.");

        try {
            controller.initializeAndCheckHardware();
            configureController();
            //check if is in mode CURRENT else changeMode(EposMode.CURRENT)
            controller.writeParameters(EposMode.CURRENT);
        } catch (FcsHardwareException ex) {
            throw new HardwareException(false,ex);
            
        } catch (ErrorInCommandExecutionException ex) {
            throw new HardwareException(true,ex);
        }
        return TreeWalkerDiag.HANDLING_CHILDREN;
        
    }
    
    @Command(type=Command.CommandType.QUERY, 
        level=Command.ENGINEERING1, 
        description="Read in the CPU of the controller parameters for mode CURRENT."
                + "If a parameter has a different value than in configuration, "
                + "throws an exception.")
    //tested on CPPM testbench in september 2015
    public void checkControllerBeforeAction() throws HardwareException, FcsHardwareException {

        EposMode modeInCPU = controller.readMode();
        if (!modeInCPU.equals(EposMode.CURRENT)) {
            String msg = name + " Controller is not in mode CURRENT";
            fcslog.error(msg);
            throw new FcsHardwareException(msg);
        }
        boolean parametersOK = true;
        controller.checkParameters(EposMode.CURRENT);
        controller.checkFault();
        if (!parametersOK) {
            String msg = name + " Some parameter values are not"
                + " the same in CPU and configuration system.";
            fcslog.error(msg);
            throw new FcsHardwareException(msg);
            
        } else {
            log.debug(name + ":controller parameters are OK - ready for action.");
        }
        
    }
    
    
    @Command(type=Command.CommandType.ACTION, 
            level=Command.ENGINEERING1, 
            description="Configure controller.")  
    //tested on CPPM testbench in september 2015
    public void configureController() throws SDORequestException, HardwareException {
        try {
            controllerConfigured = false;
            controller.activateBrake();
            controller.shutdown();
            this.configureDigitalInputOfOnlineClamps();
            this.configureDigitalOutputOfOnlineClamps();
            controllerConfigured = true;
        } catch (ShortResponseToSDORequestException ex) {
            log.warning(name+ex.getMessage());
        }
    }
    
     /**
     * This methods updates the whole autochanger state in reading all the sensors.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Update latch state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        autochanger.updateStateWithSensors();
    }
    
    /**
     * This methods updates lockStatus from the values return by the sensors.
     * This values are given in an array of hexa values as arguments of the method.
     * @param hexaValues 
     */
     @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
             description="Update state in reading sensors.")
    public void updateStateWithSensors(String[] hexaValues) {
        lock.lock();
        try {
            updatingState = true;            
            this.lockSensor.updateValue(hexaValues);
            this.lockSensorC.updateValue(hexaValues);
            this.unlockSensor.updateValue(hexaValues);
            this.unlockSensorC.updateValue(hexaValues);

 
            locked = this.lockSensor.getDigitalValue() == 1
                        && this.lockSensorC.getDigitalValue() == 0;
            unlocked = this.unlockSensor.getDigitalValue() == 1
                        && this.unlockSensorC.getDigitalValue() == 0;
            inTravel = !locked && !unlocked;
            lockSensorInError = this.lockSensor.getDigitalValue() 
                    == this.lockSensorC.getDigitalValue();
            unlockSensorInError = this.unlockSensor.getDigitalValue() 
                    == this.unlockSensorC.getDigitalValue();
            inError = lockSensorInError || unlockSensorInError || locked && unlocked; 
            
            if (inError) {
                lockStatus = FcsEnumerations.LockStatus.ERROR;
            } else if (locked) {
                lockStatus = FcsEnumerations.LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = FcsEnumerations.LockStatus.UNLOCKED;
            } else if (inTravel) {
                lockStatus = FcsEnumerations.LockStatus.INTRAVEL;    
            }
            
        } finally {

            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
            this.publishData();
        }
       
    }    
    

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Override
    //TODO modify this from what has been learned in september 2015 from testLock and testUnlock
    public void startAction(FcsEnumerations.MobileItemAction action) throws BadCommandException, 
            ErrorInCommandExecutionException, FcsHardwareException {
        switch(action) {  
            case LOCK_ONLINECLAMP :
                controller.enable();
                controller.changeMode(EposMode.CURRENT);
                //TODO checks these parameters during initialization phase.
                //onlineClampController.writeParameters(paramsForCurrentToLock);
                controller.releaseBrake();
                controller.writeCurrent(this.currentToLock);
                break;
                
           case UNLOCK_ONLINECLAMP :
                controller.enable();
                controller.changeMode(EposMode.CURRENT);
                //TODO checks these parameters during initialization phase.
                //onlineClampController.writeParameters(paramsForCurrentToUnlock);
                //first apply a force on the clamp, and then releaseBrake
                //don't do it in the reverse order !!!!
                controller.writeCurrent(this.currentToClamp);
                controller.releaseBrake();
                slowDownCurrent();
                
                controller.writeCurrent(this.currentToUnlock);
                break;
                }
    }
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="Lock online clamp for CPPM test bench.")
    //tested on CPPM testbench in september 2015
    public String testLock() throws SDORequestException, ShortResponseToSDORequestException, 
            BadCommandException, ErrorInCommandExecutionException, InterruptedException,
            FcsHardwareException, HardwareException {
        fcslog.debug(name + "BEGIN testLock");
        
        checkConditionForLocking();
        fcslog.debug(name + ":condition for locking OK.");
        
        //commented in march 2015 because we want to enable the controller in all the cases.
//                if (!onlineClampController.isEnabled()) 
//                    throw new BadCommandException(name + "controller has to be ENABLED first.");
        
        controller.enable();
        checkControllerBeforeAction();
        //now the parameters are OK
        

        controller.releaseBrake();

        controller.writeCurrent(this.currentToLock);
        fcslog.debug(name+": sent current to controller:" + currentToLock);
        //il y aura des capteurs mais en attendant, on attend. 
        fcslog.debug(name+": sleeping for:" + travelTime);
        fcslog.debug(name+":time="+System.currentTimeMillis());
        Thread.sleep(travelTime);
        fcslog.debug(name+":time="+System.currentTimeMillis());

        controller.writeCurrent(this.currentToClamp);
        fcslog.debug(name+"==> current sent to controller=" + currentToClamp);
        fcslog.debug(name+": sleeping for:" + holdTime);
        fcslog.debug(name+":time="+System.currentTimeMillis());
        Thread.sleep(holdTime);
        fcslog.debug(name+":time="+System.currentTimeMillis());

        //the good order is activateBrake, then writeCurrent in this case.
        //because we want to put the pression, active brake and then set current to 0.             
        controller.activateBrake();
        fcslog.debug(name+": controller brake activated.");
        Thread.sleep(1000);

        controller.writeCurrent(0);
        fcslog.debug(name+"==> current sent to controller=" + 0);
        controller.disable();//added in sept2015- to be tested
        fcslog.debug(name + "END testLock");
        return name+" LOCKED";
    }   
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
        description="Unlock online clamp for CPPM test bench.")
    //tested on CPPM testbench in september 2015
    public String testUnlock() throws SDORequestException, BadCommandException, 
            ShortResponseToSDORequestException, ErrorInCommandExecutionException, 
            InterruptedException, FcsHardwareException, HardwareException {
        fcslog.debug(name + "BEGIN testUnlock");
        
        checkConditionForUnlocking();
        fcslog.debug(name + ":controller parameters are OK.");

        controller.enable();
        //controller.changeMode(EposMode.CURRENT); //decided with Patrick in sept 2015
        //the mode CURRENT is set during initialization in checkHardware
        //and it is checked before each action in checkControllerBeforeAction
        checkControllerBeforeAction();
        //now the parameters are OK
        //first apply a force on the clamp, and then releaseBrake
        //don't do it in the reverse order !!!!
        controller.writeCurrent(this.currentToClamp);
        controller.releaseBrake();
        Thread.sleep(1000);
        slowDownCurrent();

        controller.writeCurrent(this.currentToUnlock);
        Thread.sleep(travelTime+3000);

        controller.writeCurrent(0);
        Thread.sleep(10000);

        //controller.activateBrake();//done in disable
        controller.disable();
        fcslog.debug(name + "END testUnlock");
        return name+" UNLOCKED";
                
    }

    @Override
    //tested on CPPM testbench in september 2015
    public void abortAction(FcsEnumerations.MobileItemAction action, 
            long delay) throws BadCommandException, 
            ErrorInCommandExecutionException, 
            FcsHardwareException {
        //TODO check if correct with Patrick
        controller.quickStop();
        controller.activateBrake();
        controller.disable();
    }

    @Override
    public void postAction(FcsEnumerations.MobileItemAction action) 
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
                switch(action) {  
            case LOCK_ONLINECLAMP :               
                controller.writeCurrent(this.currentToClamp);
                try {
                    Thread.sleep(holdTime);
                } catch (InterruptedException ex) {
                    throw new ErrorInCommandExecutionException(name 
                            + " was interrupted during action " + action.toString());
                }
                controller.activateBrake();
                controller.writeCurrent(0);
                controller.disable(); //added in sept2015 to be consistant with trucks
                //to be tested on CPPM test bench
                break;
                
           case UNLOCK_ONLINECLAMP :
 
                
                break;
                }
    }
    
    public void slowDownCurrent() throws BadCommandException, SDORequestException, 
            ShortResponseToSDORequestException, FcsHardwareException {
        int step = currentToClamp/5;
        for (int curr=currentToClamp-step; curr>0; curr=curr-step) {
            fcslog.debug("current="+curr);
            controller.writeCurrent(curr);
            //this is for CPPM test bench only - for the scale 1 prototype,
            //we will do it for the 3 controllers
            try {
                Thread.sleep(500);
            } catch (InterruptedException ex) {
                throw new BadCommandException(name 
                        + " was interrupted during action testLock." + ex.getMessage());
            }
            //read actualCurrent to check if it actualCurrent = curr.
            int actualCurrent = controller.readCurrent();
            if (!(actualCurrent == curr)) log.error(name + ": CURRENT ERROR in"
                    + " slowDownCurrent actualCurrent=" +actualCurrent+",curr="+curr);
        
        }
        fcslog.debug("current="+0);
        controller.writeCurrent(0);
    }
    
     /**
     * This method is used to configure the controllers of the autochanger online clamps.
     *          * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                * Digital Inputs activated for OnlineClamps controllers are:
                * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                * QuickStop     Input=01        Mask=Enabled    Polarity=HighActive     HE_Sensor
                * General_A     Input=02        Mask=Enabled    Polarity=HighActive     HE_Sensor\
                * General_B     Input=03        Mask=Enabled    Polarity=HighActive     Check Brake Status      
                *********************************************
                * 1 - Configure the Digital Input Functionnality index 0x2070 sindex input_ID
                * Functionnality tab
                * Value = 15    General Purpose A
                * Value = 14    General Purpose B
                * Value = 13    General Purpose C
                * Value = 12    General Purpose D
                * Value = 11    General Purpose E
                * Value = 10    General Purpose F
                * Value = 9     General Purpose G
                * Value = 8     General Purpose H
                * Value = 7     General Purpose I
                * Value = 6     General Purpose J
                * Value = 5     QuickStop
                * Value = 4     Drive enable
                * Value = 3     Position marker
                * Value = 2     Home switch
                * Value = 1     Positive Limit switch
                * Value = 0     Negative Limit switch
                msg = 'Set DigIn_1 Quick Stop,wsdo,%s,2070,01,02,0005\n'  % (str(cobID))                        
                self.com_hal.executeOperation(client,msg)
                msg = 'Set DigIn_2 General_A,wsdo,%s,2070,02,02,8000\n'  % (str(cobID))                 
                self.com_hal.executeOperation(client,msg)
                msg = 'Set DigIn_3 Genral_B,wsdo,%s,2070,02,02,4000\n'  % (str(cobID))                  
                self.com_hal.executeOperation(client,msg)
                ****************************************************************
                * 2 - Configure the Digital Input Polarity 
                * index 0x2071 subindex 0x03
                * Value = 0     High Active
                * Value = 1     Low Active
                *****************************************************************
                * 3 - Configure the Digital Input Mask 
                * index 0x2071 subindex 0x02
                * Value = 0     Functionnality state will not be displayed
                * Value = 1     Functionnality state will be displayed
                *                 * Digital Input Functionnality tab              value
                * Bit_15        General Purpose A               1       
                * Bit_14        General Purpose B               1
                * Bit_13        General Purpose C               0
                * Bit_12        General Purpose D               0
                * Bit_11        General Purpose E               0
                * Bit_10        General Purpose F               0
                * Bit_9         General Purpose G               0
                * Bit_8         General Purpose H               0
                * Bit_7         General Purpose I               0
                * Bit_6         General Purpose J               0
                * Bit_5         QuickStop                       1
                * Bit_4         Drive enable                    0
                * Bit_3         Position marker                 0
                * Bit_2         Home switch                     0
                * Bit_1         Positive Limit switch           0
                * Bit_0         Negative Limit switch           0
                **********************************************************************
                * 4 - Configure the Digital Input Execution Mask 
                * index 0x2071 subindex 0x04
                * Digital Input Excecution Mask                         Value
                * Bit_15 to Bit_6       reserved                        0       
                * Bit_5                 QuickStop                       1
                * Bit_4                 Drive enable                    1               
                * Bit_3                 Position marker                 0
                * Bit_2                 Home switch                     0
                * Bit_1                 Positive Limit switch           0
                * Bit_0                 Negative Limit switch           0
                * **********************************************************************
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="To configure the online clamps controllers.")
    public void configureDigitalInputOfOnlineClamps() throws SDORequestException {
        //1-Configure Digital Input Fonctionnality
        //QuickStop
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalInput1, "0005");
        //General_A
        //writeParameterInHexa(CanOpenEPOS.Parameter.ConfigurationOfDigitalInput2, "8000");
        //General_B
        //writeParameterInHexa(CanOpenEPOS.Parameter.ConfigurationOfDigitalInput3, "4000");
        
        //2-Configure the Digital Input Polarity
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityPolarity, "0");
        
        //3-Configure the Digital Input Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityMask, "C20");
        
        //4 - Configure the Digital Input Execution Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalInputFonctionnalityExecutionMask, "30");
        
    }
    
    /**
     * This methods is used to configure the autochanger onlineClamps EPOS controllers.
     *          * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                * Digital Outputs activated for OnlineClamps controller are:
                * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
                * Ready/Fault   Output=01       Mask=Enabled    Polarity=HighActive     
                * General_A     Output=04       Mask=Enabled    Polarity=HighActive     
                *********************************************
                * 1 - Configure the Digital Output Functionnality 
                * index 0x2079 subindex input_ID
                * Functionnality tab
                * Value = 15            General Purpose Out_A
                * Value = 14            General Purpose Out_B
                * Value = 13            General Purpose Out_C
                * Value = 12            General Purpose Out_D
                * Value = 11            General Purpose Out_E
                * Value = 10..8         not used
                * Value = 7..3          reserved
                * Value = 2             Holding Brake
                * Value = 1             Position compare
                * Value = 0             Ready/Fault
                *********************************************************************************
                * 2 - Configure the Digital Output Functionnality Mask 
                * index 0x2078 subindex 0x02
                * Value = 0     functionnality not activatd
                * Value = 1     functionnality activated
                *********************************************************************************
                * Digital Output Functionnality Mask            Value
                * Bit_15        General Purpose Out_A           1
                * Bit_14        General Purpose Out_B           0
                * Bit_13        General Purpose Out_C           0
                * Bit_12        General Purpose Out_D           0
                * Bit_11        General Purpose Out_E           0
                * Bit_10        General Purpose Out_F           0
                * Bit_9         General Purpose Out_G           0
                * Bit_8         General Purpose Out_H           0
                * Bit_7         reserved                        0
                * Bit_6         reserved                        0
                * Bit_5         reserved                        0
                * Bit_4         reserved                        0
                * Bit_3         reserved                        0
                * Bit_2         Holding Brake                   0
                *********************************************************************************
                * 3 - Configure the Digital Output Functionnality Polarity 
                * index 0x2078 subindex 0x03
                * Value = 0     associated output not change    => HighActive
                * Value = 1     associated output inverted      => LowActive
                *********************************************************************************
     * @throws SDORequestException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, 
            description="To configure the online clamps controllers.")
    public void configureDigitalOutputOfOnlineClamps() throws SDORequestException {
        //1 - Configure the Digital Output Functionnality
        //Ready/Fault
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.ConfigurationOfDigitalOutput1, "0");
        //General_A
        //gives an error : CAN OPEN DEVICE ERROR CODE=6090031 index=2079 subindex=04 error name=Value too high Error
        //writeParameterInHexa(Parameter.ConfigurationOfDigitalOutput4, "8000"); //pas bon
        
        //2 - Configure the Digital Output Functionnality Mask
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityMask, "8001");
        
        //3 - Configure the Digital Output Functionnality Polarity
        controller.writeParameterInHexa(EPOSEnumerations.Parameter.DigitalOutputFonctionnalityPolarity, "8001");        
    }
    
     /**
     * What to do when the Modules we observe send there new values.
     * This class observes its controller to publish data when its controller is 
     * in fault or after a faultReset. 
     * Needed to update the GUI.
     * @param source
     * @param v
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        log.debug(name + ":processUpdate from source=" + source.toString() 
                + " ValueUpdate=" + v.getName());
        if (!(source instanceof CanOpenEPOS)) return;
        if (v.getValue() instanceof EmergencyMessage ) {
            EmergencyMessage emcyMsg = (EmergencyMessage) v.getValue();
            log.debug(name + ":EmergencyMessage received from CanOpenProxy=" 
                    + emcyMsg.toString());

            if (controller.getName().equals(emcyMsg.getDeviceName())) {
                String errCode = emcyMsg.getDeviceErrorCode();
                switch(errCode) {
                    case("00"):
                        log.debug(name + ":faultReset ?=" + emcyMsg.toString());
                        controllerInFault=false;
                        //controllerErrorMessage = emcyMsg.getDeviceErrorName();                       
                    break;
                        
                    default:
                        log.debug(name + ":EmergencyMessage received for "
                                + "my controller from CanOpenProxy=" + emcyMsg.toString());
                        controllerInFault=true;
                        //controllerErrorMessage = emcyMsg.getDeviceErrorName();
                    }              
                this.publishData();          
            }
            
        } else if (v.getValue() instanceof String) {
            CanOpenEPOS ctrl = (CanOpenEPOS) source;
            if (ctrl.getName().equals(controller.getName())) {
                String msgFromController = (String)v.getValue();
                if (msgFromController.equals("faultReset")) {
                    this.controllerInFault=false;
                    //this.controllerErrorMessage = null;
                    this.publishData();
                }
            }
        }
    }    

    public StatusDataPublishedByAutochangerOnlineClamp getStatusData() {
        StatusDataPublishedByAutochangerOnlineClamp status = FcsUtils.createStatusDataPublishedByOnlineClamp(this);
        return status;
    }
    
    @Override
    public void publishData() {
        this.getSubsystem().publishStatus(name, getStatusData());
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) 
            throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        controller.quickStop();
    }


   
    
}
