/*
 * To change this template, choose Tools | Templates
 * and on the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.BasicAutoChangerModule;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutoChangerTrucksLocation;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceInLatchStatus;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.FilterLatchModule;
import org.lsst.ccs.subsystems.fcs.TruckModule;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;

/**
 * This class is for the Single Filter Test autochanger.
 * The autochanger in Single Filter Test has no flip rail and
 * no online clamp.
 * 
 * To move the autochanger trucks in the Single Filter Test we have 
 * 2 pneumatic cylinders each moved by an actuator : 
 * when we want the truck to be in stanby position, the standby actuator has to be ON
 * and the standback actuator has to be OFF,
 * when we want the truck to be in standback position, the standback actuator has to be ON 
 * and the standby actuator has to be OFF.
 *
 * @author virieux
 */
public class SftAutoChangerModule extends BasicAutoChangerModule {

    
    protected SftTruckMotor sftTrucksMotor;
   
    
    private long timeToConfortAPosition; //in millisecond
    private long timeToGoToStandby; //in millisecond
    private long timeToGoToStandback; //in millisecond

    public SftAutoChangerModule(String moduleName, int aTickMillis,
             FilterLatchModule latchXminus, 
             FilterLatchModule latchXplus, 
             TruckModule truckXminus, 
             TruckModule truckXplus, 
             String railsSensorsDIOName, 
             String filterSensorsDIOName,
             SftTruckMotor sftTrucksMotor, 
            long timeToConfortAPosition, 
            long timeToGoToStandby, 
            long timeToGoToStandback
            ) {
        super(moduleName, aTickMillis, latchXminus, latchXplus, truckXminus, truckXplus, railsSensorsDIOName, filterSensorsDIOName);
        this.sftTrucksMotor = sftTrucksMotor;
        this.timeToConfortAPosition = timeToConfortAPosition;
        this.timeToGoToStandby = timeToGoToStandby;
        this.timeToGoToStandback = timeToGoToStandback;
    }
    


    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/

    public SftTruckMotor getSftTrucksMotor() {
        return sftTrucksMotor;
    }

    public void setSftTrucksMotor(SftTruckMotor sftTrucksMotor) {
        this.sftTrucksMotor = sftTrucksMotor;
    }

    
    //Used by SimuActuatorModule
    public long getTimeToConfortAPosition() {
        return timeToConfortAPosition;
    }

    //Used by SimuActuatorModule
    public long getTimeToGoToStandback() {
        return timeToGoToStandback;
    }

    //Used by SimuActuatorModule
    public long getTimeToGoToStandby() {
        return timeToGoToStandby;
    }

    @ConfigChanger
    public void setTimeToConfortAPosition(long timeToConfortAPosition) {
        this.timeToConfortAPosition = timeToConfortAPosition;
    }

    @ConfigChanger
    public void setTimeToGoToStandby(long timeToGoToStandby) {
        this.timeToGoToStandby = timeToGoToStandby;
    }

    @ConfigChanger
    public void setTimeToGoToStandback(long timeToGoToStandback) {
        this.timeToGoToStandback = timeToGoToStandback;
    }
    
    
    
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

 
    
    @Override
    public boolean isMovingToStandback() {
        return this.sftTrucksMotor.isMovingToStandback();
    }

    @Override
    public boolean isMovingToStandby() {
        return this.sftTrucksMotor.isMovingToStandby();
    }
       
    
    @Override
    public boolean isMoving() {
        return isMovingToStandback() || isMovingToStandby();
    }
  
    @Override
    public String toString() {
        return getName();
    }
 


    @Override
    public void locateTrucks() {
        throw new UnsupportedOperationException("Not supported yet.");
    }




  
    @Override
    public String goToStandback() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {
        
        checkPreConditionsForTrucksMotion();
        updateTrucksLocationWithSensors();
        if (this.isAtStandback()) {
            return getName() + " trucks already at standback location";
        }
        
        if (this.getTrucksLocation().equals(AutoChangerTrucksLocation.UNKNOWN)) {
            fcslog.error(getName() + " Cannot execute goToStandback : Trucks are between standback and standby location.");
            throw new BadCommandException(getName() + " Cannot execute goToStandback : Trucks are between standback and standby location.");
        }
        
        if (this.getTrucksLocation().equals(AutoChangerTrucksLocation.ERROR)) {
            fcslog.error(getName() + " Cannot execute goToStandback : ERROR in reading rails sensors");
            //TODO : if trucks at ERROR position in normal mode, put the subsystem in error.
            throw new BadCommandException(getName() + " Cannot execute goToStandback : ERROR in reading rails sensors.");
        }
              
        
        
        if (this.isAtStandby()) {
            
            //if both the carousel and the autochanger holds the filter we can't move.
            if (!this.isTrucksEmpty() && this.getLatchesState() == FcsEnumerations.LockStatus.LOCKED) {
                this.carousel.updateClampsStateWithSensors();
                if (carousel.isHoldingFilterAtStandby()) {
                    throw new BadCommandException(getName() 
                            + " can't move filter to standback because the carousel is holding the filter at standby.");
            }
        }
            
            // to confort the position at standby
            // here the Thread.sleep can't be avoid
            fcslog.debug(getName() + ": The position is being conforted");
            sftTrucksMotor.moveToStandby();
            try {
                
                Thread.sleep(timeToConfortAPosition);
            } catch (InterruptedException ex) {
                fcslog.debug(ex);
            }
        }
        fcslog.info(getName() + " IS GOING TO STANDBACK POSITION");
        sftTrucksMotor.moveToStandback();
        
        try {
            //fcslog.debug(getName() + ": Waiting for the trucks to go to STANDBACK");
            Thread.sleep(timeToGoToStandback);
        } catch (InterruptedException ex) {
            fcslog.debug(ex);
        }
        this.sftTrucksMotor.off();

        
        updateTrucksLocationWithSensors();
        
        if (this.isAtStandback()) {          
            fcslog.debug("====> " + getName() + "  IS AT STANDBACK POSITION");
            return getName() + " IS AT STANDBACK POSITION";
        } else {
            throw new ErrorInCommandExecutionException(getName() + "could not go to STANDBACK POSITION");
        }
    }

    /**
     * This methods moves the trucks from standback position to standby position.
     * It first checks the preconditions for a motion of the trucks.
     * Then it checks if the trucks are in standback position otherwise throws a BadCommandException.
     * If the trucks are at standback, it first conforts the position (standbackMotor.on) then it moves
     * the trucks to the standby position (standbackMotor.off then standbyMotor.on).
     * The motor is stopped after the temporisation has expired.
     * The command is completed when the position sensors at standby says the trucks are in standby.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */
    @Override
    public String goToStandby() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException{
        
        
        checkPreConditionsForTrucksMotion();
        
        updateTrucksLocationWithSensors();
        
        if (this.isAtStandby()) {
            return getName() + "trucks already at standby position";
        }
        
        if (this.getTrucksLocation().equals(AutoChangerTrucksLocation.UNKNOWN)) {
            fcslog.error(getName() + " Cannot execute goToStandby : Trucks are between standback and standby location.");
            throw new BadCommandException(getName() + " Cannot execute goToStandby : Trucks are between standback and standby location.");
        }
        
        if (this.getTrucksLocation().equals(AutoChangerTrucksLocation.ERROR)) {
            fcslog.error(getName() + " Cannot execute goToStandby : ERROR in reading rails sensors");
            //TODO : if trucks at ERROR position in normal mode, put the subsystem in error.
            throw new BadCommandException(getName() + " Cannot execute goToStandby : ERROR in reading rails sensors.");
        }
        
        // to confort the position
        if (this.isAtStandback()) {
            
            sftTrucksMotor.moveToStandback();
            
            try {
                fcslog.debug(getName() + ": Waiting for the position to be conforted");
                Thread.sleep(timeToConfortAPosition);
            } catch (InterruptedException ex) {
                fcslog.error(ex);
            }
        }
        fcslog.info(getName() + " IS GOING TO STANDBY POSITION");
        sftTrucksMotor.moveToStandby();
        try {
            fcslog.debug(getName() + ": Waiting for the trucks to go to STANDBY");
            Thread.sleep(timeToGoToStandby);
        } catch (InterruptedException ex) {
            fcslog.error(ex);
        }
        this.sftTrucksMotor.off();
        //setMovingToStandby(false);
        
        updateTrucksLocationWithSensors();

        if (this.isAtStandby()) {
            
            fcslog.debug("====> " + getName() + "  IS AT STANDBY POSITION");            
            return getName() + " IS AT STANDBY POSITION";
        } else {
            throw new ErrorInCommandExecutionException(getName() + " could not go to STANDBY POSITION");
        }
 
    }

    @Override
    public String moveFilterToStandback(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {
        if (isTrucksEmpty()) {
            throw new BadCommandException(getName() + " unable to moveFilterToStandback: Autochanger is empty");
        }
        
        //TODO this test has to be done in method goToStandback
//        CarouselModule carousel = (CarouselModule) getModule("carousel");
//        carousel.updateClampsStateWithSensors();
//        if (carousel.isHoldingFilterAtStandby()) {
//            throw new BadCommandException(getName() 
//                    + "can't move filter to standback because the carousel is holding the filter at standby.");
//        }

        goToStandback();
        this.updateLatchesStateWithSensors();
        publishDataAndNotifyObservers();
        return getName() + ": " + aFilter.getName() + " is at standback position.";
    }

    @Override
    public String moveFilterToStandby(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {
        if (isTrucksEmpty()) {
            throw new BadCommandException(getName() + "/moveFilterToStandby: Autochanger is empty");
        }
        //CarouselModule carousel = (CarouselModule) getModule("carousel");
        carousel.updateClampsStateWithSensors();
        
        if (!carousel.isReadyToGrabAFilterAtStandby()) {
            throw new BadCommandException(getName() + 
                    "can't move a filter to STANDBY because carousel is not ready to grab a filter.");
        }
        goToStandby();
        this.updateLatchesStateWithSensors();
        publishDataAndNotifyObservers();
        return getName() + ": " + aFilter.getName() + " is at standby position.";
    }

    @Override
    public String goToPosition(double trucksPositionOnline) throws BadCommandException {
        throw new UnsupportedOperationException("Not in single-filter-test.");
    }

    @Override
    public double getTrucksPosition() {
        throw new UnsupportedOperationException("Not supported in single-filter-test.");
    }

    @Override
    public String moveFilterToOnline(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {
        return moveFilterToStandback(filter);
    }
    
    @Override
    @Command ( level=Command.ENGINEERING1, description="Read sensors and update latches and trucks state", type=Command.CommandType.QUERY)
    public void updateStateWithSensors() throws FcsHardwareException, ErrorInCommandExecutionException {
        this.updateLatchesStateWithSensors();       
        this.updateTrucksLocationWithSensors();
        
        if (this.getLatchXminus().getPresenceStatus().equals(FilterPresenceInLatchStatus.NOFILTER) 
                && this.getLatchXplus().getPresenceStatus().equals(FilterPresenceInLatchStatus.NOFILTER)) {
            this.setTrucksEmpty(true);
            this.setFilterOnTrucks(null);
            
        }
    }


 



    
    
    
    



}
