/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This is a pale copy of Bernard's commandWord dispenser pattern.
 * The CommandDispenser provides methods and internal classes to stored 
 * the commands we send to the tcpProxy and to wait for the response.
 * It is used by the method call of the tcpProxy to stored the initial commands 
 * and then to retrieve the response.
 * The response is updated by the tcpProxy readingThread when it arrives from 
 * the tcp Socket.
 * The CommandToRegister object has to be removed from the map by the client classes. 
 * 
 * @author virieux
 */
public class CommandDispenser {
    private static final Logger fcslog = FcsUtils.log;

    public static class TokenLostException extends Exception {
        public final String token ;

        public TokenLostException(String token) {
            this.token = token;
        }

        @Override
        public String toString() {
            return super.toString() + " " + token ;
        }
    }
    
    protected static class CommandToRegister {
        String initialCommand;
        volatile String response;
        final Lock lock = new ReentrantLock();
        final Condition responseArrived = lock.newCondition();
        
        public CommandToRegister(String aTextCommand) {
            this.initialCommand = aTextCommand;
            this.response = null;
        }
        
        public void updateResponse(String theResponse) {
            lock.lock();
            try {
                this.response = theResponse;
                responseArrived.signal();
            } finally {
                lock.unlock();
            }
        }
      
        
        /**
         * Return the response to the initial commandWord.
         * If the response is not yet arrived, it waits for it.
         * @param timeout we don't want to wait the response more than timeout
         * @return this.response when it's arrived
         * @throws org.lsst.ccs.subsystems.fcs.errors.CanOpenCallTimeoutException
         */
        public String getResponse(long timeout) throws CanOpenCallTimeoutException  {            
            
           lock.lock();

           try {
                while (this.response == null) {
                       try {
                           boolean responseOK = responseArrived.await(timeout,TimeUnit.MILLISECONDS);
                           //TODO check what does it mean that await answers false.
                           if (!responseOK) {
                               throw new CanOpenCallTimeoutException("waiting time too long before return the response for command:" + initialCommand);
                           }
                           if (this.response == null) {
                               fcslog.error("ERROR expired timeout while waiting for " + initialCommand);
                               throw new CanOpenCallTimeoutException("while waiting for response to command:" + initialCommand);
                           }                  
                   
                       }   catch (InterruptedException ex) {
                           fcslog.error("Interrupted while waiting for a response for command :  " + initialCommand, ex);
                       }
                }
                fcslog.finest("Return response: " + response + " for command: " + this.toString());
                return this.response;
            } finally {
                lock.unlock();
            }
        }
        
        @Override
        public String toString() {
            return initialCommand + "/" + response;
        }
    }

    private final HashMap<String, CommandToRegister> map = new HashMap<>() ;

    /**
     * registers a commandWord and returns a token to be used to run the invocation code
     * @param command
     * @return
     */
    public synchronized String register(String command) {
        if(command == null) {
            throw new IllegalArgumentException("null Command") ;
        }
        String key;
        String nodeID;
        String[] words = command.split(",");
        String commandWord = words[0];
//        if (commandWord.startsWith("srtr")) {
//            key = "srtr";
//        } else if (commandWord.startsWith("sync")){
//            key = "sync";
//        } else {
        if (words.length > 1) {
            nodeID = words[1];
            key = commandWord + nodeID;
        } else {
            key = commandWord;
        }
//        }
        
        fcslog.finest("command word:" + commandWord);      
        fcslog.finest("key: " + key);
        fcslog.finest("Command to register: " + command);
        map.put(key, new CommandToRegister(command)) ;
        return key ;
    }

    /**
     * registers the response at a commandWord and the token is given by the calling code.
     * @param token
     * @param response
     * @return
     */
    public String registerResponse(String token ,String response) {
        fcslog.finest("Registering token response: " + token + " " + response);
        if(response == null) {
            throw new IllegalArgumentException("null Response") ;
        }
        if (!isTokenUsed(token)) {
            throw new IllegalArgumentException("invalid Token: " + token);
        }
        CommandToRegister commandToRegister;
        //the actions on the map have to be synchronized.
        synchronized(this) {
            commandToRegister = map.get(token);
        }
        //we can't update the response in the synchronized block because the
        //thread waiting for the response has acquired a lock on the object CommandRegister.
        commandToRegister.updateResponse(response);        
        return token ;
    }
    
    //public synchronized String getCommandResponse(String token) {
    public String getCommandResponse(String token, long timeout) throws CanOpenCallTimeoutException {
        if (!isTokenUsed(token)) 
            throw new IllegalArgumentException("invalid Token: " + token); 
        
        //fcslog.debug("getCommandResponse for token=" + token);
        //fcslog.debug(map);
        CommandToRegister commandToRegister;
        synchronized(this) {
            commandToRegister = map.get(token);
        }
        String responseToSend = commandToRegister.getResponse(timeout);
        return responseToSend;
    }
    
    public synchronized void remove(String token) {
        fcslog.finest("removing token = " + token);
        map.remove(token);
    }
    
    


    /**
     * tells if the token is known.
     * <B>beware</B> : in a multihreaded environment there may be race condition
     * where this method returns true but where  an Exception is fired upon call of
     * callback Method.
     * @param token
     * @return
     */
    public synchronized boolean isTokenUsed(String token) {
    //public boolean isTokenUsed(String token) {
        if(token == null) return false ;
        return map.containsKey(token) ;
    }

}
