/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.CurrentCommandContext;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * This is the model for the loader in the Filter Exchange System. 
 * The loader is used to load a filter into the camera or to unload a filter from the camera.
 * @author virieux
 */
public class LoaderModule extends Module {
    
    private static final Logger fcslog = FcsUtils.log;
    
    private final LoaderCarrierModule carrier;
    private final LoaderClampModule clamp;
    private AutoChangerModule autochanger;
    
    private NumericSensor filterPresenceSensor;
    private NumericSensor loaderOnCameraPresenceSensor;
    private NumericSensor autochangerHoldingFilterSensor;
    private final String plutoGatewayName;
    private PlutoGatewayModule plutoGateway;
    
    private boolean empty;
//    private boolean atStorage;
//    private boolean atHandoff;
    private boolean connectedOnCamera;
    private boolean autochangerHoldingFilter;
    
    private final Lock lock = new ReentrantLock();
    private final Condition stateUpdated = lock.newCondition();
    private volatile boolean updatingState = false;

   
    public LoaderModule(String moduleName, int aTickMillis,
            LoaderCarrierModule carrier, 
            LoaderClampModule clamp,
            String plutoGatewayName,
            NumericSensor filterPresenceSensor, 
            NumericSensor loaderOnCameraPresenceSensor) {
        super(moduleName, aTickMillis);
        this.carrier = carrier;
        this.clamp = clamp;
        this.plutoGatewayName = plutoGatewayName;
        this.empty = false;
        this.connectedOnCamera = false;
        this.autochangerHoldingFilter = false;
    }
    
    @Override
    public void initModule() {
        this.autochanger = (AutoChangerModule) this.getModule("autochanger");
        this.plutoGateway = (PlutoGatewayModule) this.getModule(plutoGatewayName);
        this.filterPresenceSensor = (NumericSensor) this.getModule("filterPresenceSensor");
        this.loaderOnCameraPresenceSensor = (NumericSensor) this.getModule("loaderOnCameraPresenceSensor");
        this.autochangerHoldingFilterSensor = (NumericSensor) this.getModule("autochangerHoldingFilterSensor");
    }
    
    public boolean isHardwareReady() {
        BridgeToHardware bridgeToLoader = (BridgeToHardware) this.getModule("bridge");
        return bridgeToLoader.isHardwareReady();
    }
    
     /**
     * Returns the boolean field empty. 
     * If the empty boolean is being updated and waits 
     * for a response from a sensor, this methods waits until empty is updated.
     * If the field empty is not being updated, it returns immediatly the field empty.
     * 
     * @return empty
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if there is no filter in the loader. This command doesn't read again the sensors.")
    public boolean isEmpty() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return empty;
            
        } finally {
            lock.unlock();
        }
    }
    
             /**
     * Returns the boolean field atHandoff. 
     * If the atHandoff boolean is being updated and waits 
     * for a response from a sensor, this methods waits until atHandoff is updated.
     * If the field atHandoff is not being updated, it returns immediatly the field atHandoff.
     * 
     * @return atHandoff
     * 
     */
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the loader is connected on the camera. This command doesn't read again the sensors.")
    public boolean isConnectedOnCamera() {
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return connectedOnCamera;
            
        } finally {
            lock.unlock();
        }
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
                description="Return true if the autochanger is holding the filter. This command doesn't read again the sensors.")
    public boolean isAutochangerHoldingFilter() {
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    fcslog.error(name + ": has been interrupted while waiting for end of update.");
                }

            }
            return autochangerHoldingFilter;
            
        } finally {
            lock.unlock();
        }
    }   
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, description="Return true if a filter is there and it is held by the clamp.")
    public boolean isHoldingAFilter() throws SDORequestException, BadCommandException, FcsHardwareException {
        this.updateStateWithSensors();
        return !this.isEmpty() && this.clamp.isAtClampedPosition();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, description="Return true if no filter is in the loader.")
    public boolean isMoving() {
        return this.clamp.isMoving();
    }
    

    public LoaderClampModule getClamp() {
        return this.clamp;
    }

    public LoaderCarrierModule getCarrier() {
        return carrier;
    }
    
    public boolean isCarrierAtStoragePosition() {
        return carrier.isAtStoragePosition();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, 
            description="Returns true if loader is connected to the camera.")
    public boolean isOnCamera() throws FcsHardwareException {
        this.carrier.updateStateWithSensors();
        return carrier.isConnectedOnCamera();
    }
    
    
    

    

    public void initializeHardware() throws FcsHardwareException, ErrorInCommandExecutionException, BadCommandException {
        this.updateStateWithSensors();
        
        this.clamp.initializeHardware();
        
        this.carrier.initializeHardware();
            
        
    }
    
    @Override
    public void tick() {
        try {
            this.updateStateWithSensors();
//        try {
//            this.updateStateWithSensors();
//        } catch (ErrorInCommandExecutionException | HardwareException ex) {
//                fcslog.error(ex.getMessage());                
//                this.getSubsystem().raiseAlarm(ex.toString());
//        }
//        sendToStatus(getStatusData());
        } catch (FcsHardwareException ex) {
                fcslog.error(ex.getMessage());                
                this.getSubsystem().raiseAlarm(ex.toString());
        }
    }
    

    
    @Command ( level=Command.ENGINEERING1, description="Return true if the carrier can move.", type=Command.CommandType.QUERY)
    public boolean checkPreConditionsForCarrierMotion() throws SDORequestException, BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        fcslog.info(name + " checking pre-conditions for carrier motion");
        updateStateWithSensors();
        return carrier.isEmpty() || this.isHoldingAFilter();
    }
    
    @Command ( level=Command.ENGINEERING1, type=Command.CommandType.QUERY,
            description="Check if the clamp can be open.")
    public boolean checkPreConditionsForOpeningHooks() throws SDORequestException, BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        fcslog.info(name + " checking pre-conditions for opening hooks");
        updateStateWithSensors();
        return carrier.isEmpty() || 
                (this.isHoldingAFilter() && 
                carrier.isAtHandoffPosition() &&
                autochanger.isHoldingFilterAtHandoff());
    }
    
    /**
     * This method closes the clamps.
     * When this action is completed the filter is clamped softly on the loader and can't fall.
     * @return a message for the end user.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Close hooks.")
    public String closeHooks() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        return clamp.close();
    }
    
    /**
     * This method closes the clamps strongly.
     * When this action is completed, the filter is clamped softly on the loader and is tight fermely on the carrier.
     * 
     * @return a message for the end user.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Clamp hooks.")
    public String clampHooks() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        return clamp.clamp();
    }
    
    /**
     * This method opens the clamp.
     * @return a message for the end user.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command ( level=Command.ENGINEERING1, type=Command.CommandType.ACTION,
            description="Open the hooks")
    public String openHooks() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        checkPreConditionsForOpeningHooks();
        return clamp.open();
    }


    
     /**
     * Moves the loader carrier to the handoff position.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException
     */
    //TODO check the preconditions
    @Command ( level=Command.ENGINEERING1, description="Move the loader carrier to HANDOFF position", type=Command.CommandType.ACTION)
    public void moveToHandoff() throws BadCommandException, FcsHardwareException, SDORequestException, ErrorInCommandExecutionException {
        checkPreConditionsForCarrierMotion();
        carrier.goToHandOff();
    }

    /**
     * Moves the loader carrier to the storage position.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException
     */
    @Command ( type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Move the loader carrier to STORAGE position")
    public void moveToStorage() throws BadCommandException, FcsHardwareException, SDORequestException, ErrorInCommandExecutionException {
        checkPreConditionsForCarrierMotion();
        carrier.goToStorage();
    }
    
     /**
     * This methods updates the carrier and clamp state in reading all the sensors.
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Update clamp state in reading sensors.")
    public void updateStateWithSensors() throws FcsHardwareException {
        lock.lock();
         
        
        try {
                updatingState = true; 
                this.plutoGateway.updateValues();
                String[] readHexaValues = this.plutoGateway.getHexaValues();
                fcslog.finest("readHexaValues[0]=" + readHexaValues[0]);
                fcslog.finest("readHexaValues[1]=" + readHexaValues[1]);
                this.filterPresenceSensor.updateValue(readHexaValues);
                this.loaderOnCameraPresenceSensor.updateValue(readHexaValues);
                this.autochangerHoldingFilterSensor.updateValue(readHexaValues);
                this.clamp.updateStateWithSensors(readHexaValues);
                //this.storagePositionSensor.updateValue(readHexaValues);
                //this.handoffPositionSensor.updateValue(readHexaValues);
                this.empty = filterPresenceSensor.digitalValue == 0;
//                this.atStorage = storagePositionSensor.getDigitalValue() == 1;
//                this.atHandoff = handoffPositionSensor.getDigitalValue() == 1;
                this.connectedOnCamera = loaderOnCameraPresenceSensor.getDigitalValue() == 1;
                this.autochangerHoldingFilter = autochangerHoldingFilterSensor.getDigitalValue() == 1;
        
        } finally {

            updatingState = false;
            stateUpdated.signal(); 
            lock.unlock();
        } 
        //TODO implement this
        //this.publishData();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.NORMAL, description="List and display info on the hooks.")
    public String listHooks() {
        return this.clamp.listHooks();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display hooks sensors values.")
    public String listSensorsValues() {
        return this.clamp.listSensorsValues();
    }
    
    //for the simulation 
    public void test() {       
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        fcslog.debug("Command running:" + currentCmdContext.getCommandName() );
        fcslog.info("coming from:" + currentCmdContext.getCommandOriginator());
        this.getSubsystem().interruptActionThread();
    }
    

    
}
