
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.concurrent.locks.Condition;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.bus.KVList;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module.ValueUpdate;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutoChangerTrucksLocation;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceInLatchStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.AutoChanger;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterLocation;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenException;
import org.lsst.ccs.subsystems.fcs.errors.ClampsOrLatchesDisagreeException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * An abstract changer module without a fliprail, without online clamp, 
 * and without a motor.
 * This Module is in charge of the 2 latches which hold a filter in the autochanger.
 * This basic autochanger is used in single filter test and will be used with the final 
 * hardware too.
 * It extends MobileItemModule for the actions which can be applied to the 2 latches : OPENLATCHES and CLOSELATCHES.
 * @author virieux
 */
public abstract class BasicAutoChangerModule extends MobileItemModule implements AutoChanger {
    
    private boolean trucksEmpty;
    Filter filterOnTrucks;
    Filter filterToGrab;
    protected CarouselModule carousel;
    
    
    /**
    * Position of the trucks of the autochanger. It is monitored by the 4 rail sensors.
    * If the sensors detects a value different for the 2 sides, the trucksLocation is ERROR.
    */
    AutoChangerTrucksLocation trucksLocation;

    /**
     * A latch which hold a filter on the trucks at position X-
     */
    private final FilterLatchModule latchXminus;

    /**
     * A latch which hold a filter on the trucks at position X+
     */
    private final FilterLatchModule latchXplus;
    
    /**
     * The truck on side X-
     */
    protected TruckModule truckXminus;
    
    /**
     * The truck on side X+
     */
    protected TruckModule truckXplus;
    
    CompactIOModule railsSensorsDIO;
    CompactIOModule filterSensorsDIO;
     /**
     * This is the name of the DIO on which are plugged the filter sensors.
     * Used in the groovy file and in initModule.
     * 
     */
    String railsSensorsDIOName;
         /**
     * This is the name of the DIO on which are plugged the rails sensors.
     * Used in the groovy file and in initModule.
     * 
     */
    String filterSensorsDIOName;
    
     

    public final static String publishedByAutoChangerOutputName = "publishedByAutoChanger";
    protected volatile ModuleState state = ModuleState.HALTED;
    private volatile FilterPresenceInLatchStatus presenceStatus;
    protected volatile LockStatus latchesState;
    
    private boolean initialized = false;
    

    

    
    /*******************************************************************************************/
    /********************** Tools for the synchronization **************************************/
    /*******************************************************************************************/
    protected volatile boolean updatingLatches = false;
    protected volatile boolean closingLatches = false;
    protected volatile boolean openingLatches = false;
    
    private volatile boolean updatingTrucksLocation = false;
    //When we update the 2 latches, we want to take a lock on the autochanger until
    //the end of the update, so any other thread has to wait for this update.
    final Condition stateUpdated = lock.newCondition();
    
    //closeCompleted and openCompleted are used by methods whith ThreadSleep
    final Condition closeCompleted = lock.newCondition();
    final Condition openCompleted = lock.newCondition();
    
    //When we update the 2 trucks position reading the sensors, 
    //we want to take a lock on the autochanger until
    //the end of the update, so any other thread has to wait for this update.
    final Condition trucksLocationUpdated = lock.newCondition();
    
    protected long timeoutForOpeningLatches;
    protected long timeoutForClosingLatches;
    
    /**
     * When we close the Latches, after the numeric sensors have detected that the latches are closed, we want to wait
     * some time to be sure that the latches are really closed.
     */
    protected long timeToWaitForCompleteClose;
    
     /**
     * When we open the Latches, after the numeric sensors have detected that the latches are opened, we want to wait
     * some time to be sure that the latches are really opened.
     */
    protected long timeToWaitForCompleteOpen;

    public BasicAutoChangerModule(String moduleName, int aTickMillis, 
            FilterLatchModule latchXminus, 
            FilterLatchModule latchXplus, 
            TruckModule truckXminus, 
            TruckModule truckXplus, 
            String railsSensorsDIOName, 
            String filterSensorsDIOName) {
        super(moduleName, aTickMillis);
        this.latchXminus = latchXminus;
        this.latchXplus = latchXplus;
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
        this.railsSensorsDIOName = railsSensorsDIOName;
        this.filterSensorsDIOName = filterSensorsDIOName;
    }
    
    /*******************************************************************************************/
    /********************** End of Tools for the synchronization********************************/
    /*******************************************************************************************/

    

    
    
    
    
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/

    public CompactIOModule getRailsSensorsDIO() {
        return railsSensorsDIO;
    }

    public CompactIOModule getFilterSensorsDIO() {
        return filterSensorsDIO;
    }
    
    @Override
    public Filter getFilterOnTrucks() {
        return filterOnTrucks;
    }
    
        /**
     * @return the state
     */
    public ModuleState getState() {
        return state;
    }

    /**
     * @param state the state to set
     */
    public void setState(ModuleState state) {
        this.state = state;
    }

    public FilterLatchModule getLatchXminus() {
        return latchXminus;
    }

    public FilterLatchModule getLatchXplus() {
        return latchXplus;
    }


    public TruckModule getTruckXminus() {
        return truckXminus;
    }

    public TruckModule getTruckXplus() {
        return truckXplus;
    }
    
    public boolean isTrucksEmpty() {
        return trucksEmpty;
    }
    
    public void setTrucksEmpty(boolean isEmpty) {
        this.trucksEmpty = isEmpty;
    }

    public void setFilterOnTrucks(Filter filterOnTrucks) {
        this.filterOnTrucks = filterOnTrucks;
    }
    
    public String getFilterOnTrucksName() {
        if (this.filterOnTrucks == null) {
            return "none";
        } else {
            return this.filterOnTrucks.getName();
        }
    }

    public String getFilterSensorsDIOName() {
        return filterSensorsDIOName;
    }

    public void setFilterSensorsDIOName(String filterSensorsDIOName) {
        this.filterSensorsDIOName = filterSensorsDIOName;
    }

    public String getRailsSensorsDIOName() {
        return railsSensorsDIOName;
    }

    public void setRailsSensorsDIOName(String railsSensorsDIOName) {
        this.railsSensorsDIOName = railsSensorsDIOName;
    }
    
    @ConfigChanger
    public void setTimeToWaitForCompleteClose(long timeToWaitForCompleteClose) {
        this.timeToWaitForCompleteClose = timeToWaitForCompleteClose;
    }
    
    @ConfigChanger
    public void setTimeToWaitForCompleteOpen(long timeToWaitForCompleteOpen) {
        this.timeToWaitForCompleteOpen = timeToWaitForCompleteOpen;
    }

    @ConfigChanger
    public void setTimeoutForClosingLatches(long timeoutForClosingLatches) {
        this.timeoutForClosingLatches = timeoutForClosingLatches;
    }
    
    @ConfigChanger
    public void setTimeoutForOpeningLatches(long timeoutForOpeningLatches) {
        this.timeoutForOpeningLatches = timeoutForOpeningLatches;
    }
    
    
    

    public abstract boolean isMovingToStandback();


    public abstract boolean isMovingToStandby(); 


 
    
    
    
    /*****************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  **************************************************/
    /*****************************************************************************************************/
    
    /*****************************************************************************************************/
    /********************** Overriden methods from Modules or generic methodes ***************************/
    /*****************************************************************************************************/
    
    
    @Override
    public void initModule() {
        log.info("[AutoChangerModule] Initializing the Auto Changer module ");
        trucksLocation = AutoChangerTrucksLocation.UNKNOWN;
        latchesState = LockStatus.UNKNOWN;
        carousel = (CarouselModule) getModule("carousel");
        this.filterSensorsDIO = (CompactIOModule) this.getModule(this.filterSensorsDIOName);
        this.railsSensorsDIO = (CompactIOModule) this.getModule(this.railsSensorsDIOName);
        this.timeoutForClosingLatches = java.lang.Math.max(this.latchXminus.getTimeoutForClosing(),this.latchXplus.getTimeoutForClosing());
        this.timeoutForOpeningLatches = java.lang.Math.max(this.latchXminus.getTimeoutForOpening(),this.latchXplus.getTimeoutForOpening());
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hardware is connected and ready.")
    @Override
    public boolean isHardwareReady() {
        return ((MainModule) this.getModule("main")).isHardwareReady();
    }
    
    @Override
    public void tick() {
        try {
            this.updateLatchesStateWithSensors();
        } catch (FcsHardwareException | ErrorInCommandExecutionException ex) {
            this.getSubsystem().raiseAlarm(ex.toString());
        }
        this.publishData();
    }

    
    
     /*
     * what to do when the Modules we observe change their values.
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        log.debug(getName() + "===> PROCESS_UPDATE from " + source.toString());
        this.publishDataAndNotifyObservers();
    }

    
    public StatusDataPublishedByBasicAutoChanger getStatusData() {
        return FcsUtils.createStatusDataPublishedByBasicAutoChanger(this);
    }

    public void publishDataAndNotifyObservers() {
        StatusDataPublishedByBasicAutoChanger status = getStatusData();
        //sendToStatus(status);
        //publishAutoChangerData();
        this.publishData();
        setChanged();
        notifyObservers(new ValueUpdate(publishedByAutoChangerOutputName, status));
        //notifyObservers(new ValueUpdate(publishedByAutoChangerOutputName, this));
    }
    
    /**
     * Publish on the status bus a KVList object.
     * Used by the GUIs and the Trending data base persister.
     */
    @Deprecated //because GUI implements DataStatusListener
    public void publishAutoChangerData() {
        KVList kvlist = new KVList(10);
        long timestamp = System.currentTimeMillis();
        kvlist.add("dataType", "autochanger");
        kvlist.add(getName() + "/isTrucksEmpty", this.trucksEmpty);
        kvlist.add(getName() + "/railSensorStandbackXminusValue", this.truckXminus.railSensorStandback.digitalValue);
        kvlist.add(getName() + "/railSensorStandbackXplusValue",this.truckXplus.railSensorStandback.digitalValue);
        kvlist.add(getName() + "/railSensorStandbyXminusValue",this.truckXminus.railSensorStandby.digitalValue);
        kvlist.add(getName() + "/railSensorStandbyXplusValue", this.truckXplus.railSensorStandby.digitalValue);
        kvlist.add(getName() + "/isMovingToStandback", this.isMovingToStandback());
        kvlist.add(getName() + "/isMovingToStandby", this.isMovingToStandby());
        publish(timestamp,kvlist);
    }
    
     /**
     * Publish Data on status bus for trending data base and GUIs.
     * 
     */
     public void publishData() {
        //log.debug(getName() + " Publish Data");
        StatusDataPublishedByBasicAutoChanger status = this.getStatusData();
        //log.debug(getName() + " status data=" + status.toString());
        this.publish("autochanger", status);
     }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n");
        sb.append("Trucks position = ");
        sb.append("Filter on trucks:");
        if (this.filterOnTrucks == null) {
            sb.append(" NONE").append("\n");
        } else {
            sb.append(this.filterOnTrucks.getName()).append("\n");
        }
        return sb.toString();
    }
    
    /*****************************************************************************************************/
    /********************** END of Overriden methods from Modules  ***************************************/
    /*****************************************************************************************************/

    /*****************************************************************************************************/
    /********************** Methods for the motion of the trucks *****************************************/
    /*****************************************************************************************************/
    /**
     * 
     * @return true if the trucks are moving
     */
    public abstract boolean isMoving();

    public abstract String goToStandby() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException;
    
    public abstract String goToStandback() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException ;

    //public abstract String moveFilterToOnline(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException;

    @Override
    public abstract String moveFilterToStandby(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException;

    public abstract String moveFilterToStandback(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException;
    
    public abstract void updateStateWithSensors() throws FcsHardwareException, ErrorInCommandExecutionException;

    /**
     * This method checks if the preconditions are ok before a motion of the trucks can be started.
     * If the preconditions are not ok, it throws a BadCommandException.
     * A motion of the trucks can be started : 
     * if the latches are not in error and 
     * if the truks are empty or (the trucks are loaded and the latches are Locked).
     * @throws BadCommandException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException 
     */
    @Command ( level=Command.ENGINEERING1, description="checks if the preconditions are ok before a motion of the trucks can be started", type=Command.CommandType.QUERY)
    public void checkPreConditionsForTrucksMotion() throws BadCommandException, FcsHardwareException, ErrorInCommandExecutionException {
        log.info("Checking pre-conditions for trucks motion");
        updateLatchesStateWithSensors();
        log.debug(getName() + " LATCHES STATUS= " + this.latchesState.toString());
        if (latchesState == LockStatus.ERROR)  
            throw new BadCommandException(getName() + 
                    "Trucks motion forbidden when latches status is ERROR.");
        if (latchesState == LockStatus.UNKNOWN)
            throw new BadCommandException(getName() + 
                    "Trucks motion forbidden when latches status is UNKNOWN.");
        if (!trucksEmpty && latchesState == LockStatus.UNLOCKED) {          
            throw new BadCommandException(getName() + 
                    "Trucks motion forbidden when trucks are loaded and latches are UNLOCKED."); 
        }
        
        if (this.isHoldingFilterAtStandby() && carousel.isHoldingFilterAtStandby())
            throw new BadCommandException(getName() + 
                    "Trucks motion forbidden when both carousel and autochanger are holding filter.");
        

        //TODO Trucks motion forbidden when both carousel and autochanger are holding filter.
        //TODO Trucks motion forbidden when both loader and autochanger are holding filter.
        log.info("===== Preconditions for trucks motion are checked.");
    }
    

    @Override
    @Command ( level=Command.ENGINEERING1, description="Go to STANDBY to grab filter and stay at STANDBY", type=Command.CommandType.ACTION)
    public String grabFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException, CanOpenException {
        if (this.filterOnTrucks != null) {
            throw new BadCommandException(getName() + ": can't grab a filter " + "when a filter is already loaded in trucks ");
        }
        log.info(getName() + ": grabbing " + filter.getName() + " at standby position.");
        goToStandby();
        closeLatchesAtStandby(); 
        //now the  filter is on the autochanger
        synchronized (this) {
                this.filterOnTrucks = filter;
                setTrucksEmpty(false);
                this.filterOnTrucks.setFilterLocation(FilterLocation.ONAUTOCHANGER);
        }
        
        String ack = getName() + ": " + filter.getName() + " is grabbed on autochanger";
        log.info(ack);
        return ack;
    }
        
    /**
     * Open the latches at standby position and go back empty to STANBACK position.
     * @return
     * @throws ErrorInCommandExecutionException
     * @throws BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws CanOpenException 
     */
    @Override
    @Command ( level=Command.ENGINEERING1, description="Ungrab filter at STANDBY and go back to HANDOFF", type=Command.CommandType.ACTION)
    public String ungrabFilterAtStandby() throws ErrorInCommandExecutionException, BadCommandException, FcsHardwareException, CanOpenException {
        //TODO to check : carousel clamp must be locked
        //
        log.info(getName() + ": ungrabbing filter at standby position.");
        openLatchesAtStandby();
        //eject filter from autochanger
        synchronized (this) {
            this.filterOnTrucks = null;
            setTrucksEmpty(true);
            this.publishDataAndNotifyObservers();
        }
        //when a filter is at standby position the trucks goes empty to SWAPOUT position
        log.info(getName() + " trucks going empty to STANDBACK position");
        //goToPosition(this.getTrucksPositionSwapout());
        goToStandback();
        
        String ack = getName() + ": trucks are empty at STANDBACK position ";
        return ack;
    }
    
    /*****************************************************************************************************/
    /********************** END of Methods for the motion of the trucks **********************************/
    /*****************************************************************************************************/

    /*****************************************************************************************************/
    /********************** Methods related to the latches ***********************************************/
    /*****************************************************************************************************/
    
     /**
     * This method returns the latchesState if it is not been updated otherwise it waits until the end of 
     * the update.
     * @return the trucksLocation
     */
    public LockStatus getLatchesState() {
        lock.lock();
        try {
            while (this.updatingLatches) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    log.error(ex);
                }         
            }
            return this.latchesState;
        } finally {
            lock.unlock();
        }
    }
    
    /**
     * 
     * @return true if the trucks are at standby position and hold a filter.
     */
    @Command ( level=Command.ENGINEERING1, description="Return true if the trucks are at STANDBY position and hold a filter", type=Command.CommandType.QUERY)
    public boolean isHoldingFilterAtStandby() {
        if (isMoving()) return false;
        if (!isAtStandby()) return false;
        if (isTrucksEmpty()) return false;
        if (this.getLatchesState().equals(LockStatus.LOCKED)) return true;
        return false;
    }
    
     /**
     * 
     * @return true if the trucks are at standback position and hold the filter.
     */
    @Command ( type=Command.CommandType.QUERY,level=Command.ENGINEERING1, description="Return true if the trucks are at STANDBACK position and hold a filter")
    public boolean isHoldingFilterAtStandback() {
        if (isMoving()) return false;
        if (!isAtStandback()) return false;
        if (isTrucksEmpty()) return false;
        if (this.getLatchesState().equals(LockStatus.LOCKED)) return true;
        return false;
        
        
    }
    
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.OPENLATCHES)) {
            return this.getLatchesState().equals(LockStatus.UNLOCKED);
        } else if (action.equals(MobileItemAction.CLOSELATCHES)) {
            return this.getLatchesState().equals(LockStatus.LOCKED);
        } else {
            throw new IllegalArgumentException("Action on latches must be OPENLATCHES or CLOSELATCHES");
        }
    }
    
    @Override
    public void postAction(MobileItemAction action) {        
    }
    
    
    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        if (action.equals(MobileItemAction.OPENLATCHES)) {
            this.latchXminus.latchActuator.open();
            this.latchXplus.latchActuator.open();
        } else if (action.equals(MobileItemAction.CLOSELATCHES)) {
            this.latchXminus.latchActuator.close();
            this.latchXplus.latchActuator.close();           
        } else throw new IllegalArgumentException(getName() + "Action on latches must be OPENLATCHES or CLOSELATCHES");
        
    }
    
    @Override
    public void stopAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        //TODO complete this
        log.info(name + " stopAction : nothing to be done.");
    }
    
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws FcsHardwareException, ErrorInCommandExecutionException {
        try {
            this.updateLatchesStateWithSensors();
        } catch (ClampsOrLatchesDisagreeException ex) {
            log.debug(getName() + ": a little delai between the update of the 2 latches : have to wait ...");
        }

    }
    

    /**
     * Open the 2 latches when the autochanger trucks are at standby position.
     * @return a message for the end user
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Override
    @Command ( level=Command.ENGINEERING1, description="Open the 2 latches when the autochanger trucks are at standby position", type=Command.CommandType.ACTION, alias ="openLatches")
    public String openLatchesAtStandby() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        
        this.updateStateWithSensors();
        
        if (!this.isAtStandby()) throw new BadCommandException(getName() + "CANNOT OPEN LATCHES at STANDBY : AUTOCHANGER is NOT AT STANDBY");
        
        if (!carousel.isHoldingFilterAtStandby()) throw new BadCommandException(getName() + "CANNOT OPEN LATCHES at STANDBY : CAROUSEL is NOT HOLDING THE FILTER");
               
        this.executeAction(MobileItemAction.OPENLATCHES, this.timeoutForOpeningLatches);

        
        //We want to wait to be sure that the latches are opened.
        //Here we can't avoid having Thread.sleep
//
        
        this.latchXminus.latchActuator.powerOff();
        this.latchXplus.latchActuator.powerOff();
    
        publishDataAndNotifyObservers();
        return "Autochanger latches are unlocked";
    }
    
     /**
     * Close the 2 latches when the autochanger trucks are at standby position.
     * @return a message for the end user
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Override
    @Command ( level=Command.ENGINEERING1, description="Close the 2 latches when the autochanger trucks are at standby position", type=Command.CommandType.ACTION, alias ="closeLatches")
    public String closeLatchesAtStandby() throws BadCommandException, ErrorInCommandExecutionException, FcsHardwareException {
        
        this.updateStateWithSensors();
        
        if (!this.isAtStandby()) throw new BadCommandException(getName() + "CANNOT CLOSE LATCHES at STANDBY : AUTOCHANGER is NOT AT STANDBY");
                      
        this.executeAction(MobileItemAction.CLOSELATCHES, this.timeoutForClosingLatches);
       
        //We want to wait to be sure that the latches are opened.
        //Here we can't avoid having Thread.sleep
//        try {
//            log.info(getName() + ": Waiting for the latches to close at standby");
//            Thread.sleep(timeToWaitForCompleteClose);
//        } catch (InterruptedException ex) {
//            log.error(ex);
//        }
        
        this.latchXminus.latchActuator.powerOff();
        this.latchXplus.latchActuator.powerOff();
    
        publishDataAndNotifyObservers();
        return "Autochanger latches are locked";
    }
    

    /**
     * This methods updates the latches status in reading the sensors.
     */
    public void updateLatchesStateWithSensors() throws FcsHardwareException, ErrorInCommandExecutionException {
        lock.lock();
        this.updatingLatches = true;
        
        try {
            
            this.filterSensorsDIO.updateValue();
            
            String filterSensorsHexaValue = this.filterSensorsDIO.getHexaValue();
            this.latchXminus.updateState(filterSensorsHexaValue);
            this.latchXplus.updateState(filterSensorsHexaValue);
            
            if (latchXminus.getLockStatus().equals(latchXplus.getLockStatus()))  {
                this.latchesState = latchXminus.getLockStatus();
            } else {
            throw new ClampsOrLatchesDisagreeException("Error in latches lock status : the latches don't agree.",getName());
            }

            if (latchXminus.getPresenceStatus().equals(latchXplus.getPresenceStatus())) {
                this.presenceStatus = latchXminus.getPresenceStatus();
            } else {
            throw new ClampsOrLatchesDisagreeException("Error in latches filter presence status : the latches don't agree.", getName());
            }
            
            if (this.latchesState.equals(FcsEnumerations.LockStatus.LOCKED) && this.presenceStatus.equals(FcsEnumerations.FilterPresenceInLatchStatus.ENGAGED)) {
                this.trucksEmpty = false;
            } else {
                this.trucksEmpty = true;
            }
            log.info(getName() + ": latches are updated");

        } finally {
            this.updatingLatches = false;
            this.stateUpdated.signal();
            lock.unlock();
        }
        publishDataAndNotifyObservers();
    }

    
    /*****************************************************************************************************/
    /********************** END of Methods related to the latches ****************************************/
    /*****************************************************************************************************/
    
    /*****************************************************************************************************/
    /********************** Methods to retrieve the trucks location **************************************/
    /*****************************************************************************************************/
    
     /**
     * This method returns the truckLocation if it is not been updated otherwise it waits until the end of 
     * the update.
     * @return the trucksLocation
     */
    public AutoChangerTrucksLocation getTrucksLocation() {
        lock.lock();
        try {
            while (updatingTrucksLocation) {
                try {
                    trucksLocationUpdated.await();
                } catch (InterruptedException ex) {
                    log.error(ex);
                }         
            }
            return trucksLocation;
        } finally {
            lock.unlock();
        }
    }
    
    
     /**
     * 
     * @return true if the trucks are at standby position, otherwise false
     */
    @Override
    @Command ( level=Command.ENGINEERING1, description="Return true if the trucks are at STANDBY position", type=Command.CommandType.QUERY)
    public boolean isAtStandby() {
        return getTrucksLocation() == AutoChangerTrucksLocation.STANDBY;
    }
    
     /**
     * For the single-filter-test only ?
     * @return true if the trucks are at standback position, otherwise false
     */
    @Command ( level=Command.ENGINEERING1, description="Return true if the trucks are at STANDBACK position", type=Command.CommandType.QUERY)
    public boolean isAtStandback() {
        return getTrucksLocation() == AutoChangerTrucksLocation.STANDBACK;
    }
    
     /**
     * 
     * @return true if the trucks are at standback position, otherwise false
     */
    @Command ( level=Command.ENGINEERING1, description="Return true if the trucks are at HANDOFF position", type=Command.CommandType.QUERY)
    public boolean isAtHandoff() {
        return getTrucksLocation() == AutoChangerTrucksLocation.HANDOFF;
    }
    
         /**
     * 
     * @return true if the trucks are at standback position, otherwise false
     */
    @Command ( level=Command.ENGINEERING1, description="Return true if the trucks are at ONLINE position", type=Command.CommandType.QUERY)
    public boolean isOnline() {
        return getTrucksLocation() == AutoChangerTrucksLocation.ONLINE;
    }
    

        
     /**
     * This methods updates the trucks location in reading the sensors.
     */
    public void updateTrucksLocationWithSensors() throws ErrorInCommandExecutionException {
        lock.lock();
        updatingTrucksLocation = false;
        try {
            this.railsSensorsDIO.updateValue();

            String dioHexaValue = this.railsSensorsDIO.getHexaValue();
            log.debug(this.railsSensorsDIO.getName() + " HEXA VALUE=" + dioHexaValue);
            this.truckXminus.updateLocation(dioHexaValue);
            this.truckXplus.updateLocation(dioHexaValue);

            if (truckXminus.getTruckLocation() == truckXplus.getTruckLocation()) {
                this.trucksLocation = truckXminus.getTruckLocation();

            } else {
                String msg = getName() + "Trucks are not at the same location";
                log.error(msg);
                throw new ErrorInCommandExecutionException(msg);
            }
            
       } finally {
           this.updatingTrucksLocation = false;
           this.trucksLocationUpdated.signal();
                           
           lock.unlock();
       } 
       publishDataAndNotifyObservers();

    }

    public void grabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    public void ungrabFilterAtHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    @Command ( level=Command.ENGINEERING1, description="Open the 2 latches when the autochanger trucks are at HANDOFF position", type=Command.CommandType.ACTION, alias ="openLatchesHO")
    public void openLatchesAtHandoff() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    @Command ( level=Command.ENGINEERING1, description="Close the 2 latches when the autochanger trucks are at HANDOFF position", type=Command.CommandType.ACTION, alias ="closeLatchesHO")
    public void closeLatchesAtHandoff() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }


    
    /*****************************************************************************************************/
    /********************** END of Methods to retrieve the trucks location *******************************/
    /*****************************************************************************************************/



    
  


    
}
