package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.common.EngineState;
import org.lsst.ccs.subsystems.fcs.common.FilterLatch;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.common.Motor;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class will be used with the final hardware in which we have 
 * a clamp online to hold the filter at online position.
 * It's not used for Single FIlter Test.
 * @author virieux
 */
public class AutoChangerModule extends BasicAutoChangerModule {
	
    /**
     * 
     */
    private static final long serialVersionUID = -3737919230282962231L;

    private FilterLatch onlineClamp;
    private double trucksPositionOnline;
    private double trucksPositionAtStandby;
    private double trucksPositionSwapout;
    Motor motor;

    public AutoChangerModule(FilterLatch onlineClamp, double trucksPositionOnline, double trucksPositionAtStandby, double trucksPositionSwapout, Motor motor, String moduleName, int aTickMillis, FilterLatchModule latchXminus, FilterLatchModule latchXplus, TruckModule truckXminus, TruckModule truckXplus, String railsSensorsDIOName, String filterSensorsDIOName) {
        super(moduleName, aTickMillis, latchXminus, latchXplus, truckXminus, truckXplus, railsSensorsDIOName, filterSensorsDIOName);
        this.onlineClamp = onlineClamp;
        this.trucksPositionOnline = trucksPositionOnline;
        this.trucksPositionAtStandby = trucksPositionAtStandby;
        this.trucksPositionSwapout = trucksPositionSwapout;
        this.motor = motor;
    }

    

    @Override
    public void initModule() {
        log.info("[AutoChangerModule] Initializing the Auto Changer module ");
        setState(ModuleState.HALTED);

        this.motor.setMinimalPosition(getTrucksPositionAtStandby());
        this.motor.setMaximalPosition(getTrucksPositionOnline());
    }
    
    

    @Override
    public StatusDataPublishedByAutoChanger getStatusData() {
        return FcsUtils.createStatusDataPublishedByAutoChanger(this);
    }


    //TODO in real life check if it's enough
    private boolean isAbleToMove() {
            return !this.onlineClamp.isLocked();
    }

    /**
     * when a filter is at online position it is locked by a clamp.
     * when the clamp is closed, the filter is hold at online position.
     * @return 
     * @throws ErrorInCommandExecutionException 
     */
    public String closeOnlineClamp() throws ErrorInCommandExecutionException {
            //this.onlineClamp.lock();
            if (!onlineClamp.isLocked())
                    throw new ErrorInCommandExecutionException(getName() 
                                    + "online clamp couldn't be closed.");
            String ack = getName() + ": online clamp closed";
            log.info(ack);
            return ack;
    }


    /**
     * when a filter is at online position it is locked by a clamp.
     * when the clamp is open, the filter is not hold at online position,
     * so the autochanger can move the filter back to standby position.
     * @throws ErrorInCommandExecutionException 
     */
    public String openOnlineClamp() throws ErrorInCommandExecutionException {
            //this.onlineClamp.unlock();
            if (onlineClamp.isLocked())
                    throw new ErrorInCommandExecutionException(getName() 
                                    + "online clamp couldn't be open.");
            String ack = getName() + ": online clamp open";
            log.info(ack);
            return ack;
    }


    public FilterLatch getOnlineClamp() {
            return onlineClamp;
    }


    public void setOnlineClamp(FilterLatch onlineClamp) {
            this.onlineClamp = onlineClamp;
    }

//    @Override
//    public void publishDataAndNotifyObservers() {
//            StatusDataPublishedByAutoChanger status = getStatusData();
//            sendToStatus(status);
//            setChanged();
//            notifyObservers(new ValueUpdate(publishedByAutoChangerOutputName, status));
//    }
    
    


    /**
     * This method moves the autochanger trucks to a position given as a parameter.
     * @param requiredPosition the position the trucks will have at the end of the
     * execution this method if every thing works fine.
     * It implies the autochanger motor.
     *
     * @return
     * @throws IllegalArgumentException
     * @throws BadCommandException
     */
    @Override
    public String goToPosition(double requiredPosition) throws IllegalArgumentException, BadCommandException {
        String message = null;
        //availables values for a position : [Standby,Online]
        //availables values for a position : [Standby,Online]
        if (requiredPosition > this.getTrucksPositionOnline() || requiredPosition < this.getTrucksPositionAtStandby()) {
            throw new IllegalArgumentException("Please enter a position between " + this.getTrucksPositionAtStandby() + " and " + this.getTrucksPositionOnline());
        }
        if (isMoving()) {
            // Thread-safe because of encapsulation for carousel.state
            //if (getState() == ModuleState.RUNNING) {
            // Thread-safe because of encapsulation for carousel.state
            //if (getState() == ModuleState.RUNNING) {
            message = "Auto-changer is moving, stop it before sending a new motion command.";
            log.error(message);
            throw new BadCommandException(message);
        }
        if (!this.isAbleToMove()) {
            message = "Auto-changer is unable to move because " + "fliprail is open or online clamp is locked.";
            log.error(message);
            throw new BadCommandException(message);
        }
        setState(ModuleState.RUNNING);
        this.motor.setRequiredPosition(requiredPosition);
        log.info(getName() + " is moving to position: " + requiredPosition);
        log.info("Please wait during auto-changer motion...");
        message = "Moving to required position: " + requiredPosition;
        double displacement = requiredPosition - getTrucksPosition();
        motor.move(displacement);
        log.info("AutoChanger moving to required position: " + requiredPosition + ", please wait...");
        while (isMoving()) {
            try {
                log.info("...auto-changer moving, please wait.....");
                log.info(getName() + " WAITING TO BE AT REQUIRED POSITION=" + requiredPosition);
                log.info(state.toString());
                Thread.sleep(tickMillis);
            } catch (InterruptedException e) {
                return "Sleeping interrupted";
            }
        }

        //auto changer is not moving any more.
        if (!isMoving()) {
            log.info("THREAD=" + Thread.currentThread().getName());
            this.state=ModuleState.HALTED;

            publishDataAndNotifyObservers();
            log.info("auto-changer is now halted.");
            log.info("filter on truck: " + this.getFilterOnTrucksName());
            if (getTrucksPosition() == requiredPosition) {
                message = "goToPosition command completed";
                log.info(message);
            } else {
                message = "goToPosition command non completed: auto-changer is not stopped at the required position";
                log.info(message);
            }
        }
        log.error(message);
        return message;
    }

    @Deprecated
    public String stopAutochanger() {
        String message;
        if (isMoving()) {
            message = "Stopping auto-changer";
            motor.stop();
            this.state=ModuleState.HALTED;
        } else {
            message = "Auto-changer is already stopped";
        }
        return message;
    }

    @Override
    public String moveFilterToOnline(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException {
        if (isTrucksEmpty()) {
            throw new BadCommandException("Autochanger is empty");
        }
        if (getTrucksPosition() == this.getTrucksPositionOnline()) {
            throw new BadCommandException("Filter " + aFilter.getName() + " is already at online position.");
        }
        goToPosition(this.getTrucksPositionOnline());
        if (getTrucksPosition() == this.getTrucksPositionOnline()) {
            log.info(getName() + ": " + aFilter.getName() + " is at online position");
            return closeOnlineClamp();
        }
        log.error("");
        return null;
    }

    @Override
    public String moveFilterToStandby(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException {
        if (isTrucksEmpty()) {
            throw new BadCommandException("Autochanger is empty");
        }
        if (getTrucksPosition() == this.getTrucksPositionAtStandby()) {
            throw new BadCommandException("Filter " + this.filterOnTrucks.getName() + " is already at standby position.");
        }
    //        if (!(getTrucksPosition() == this.trucksPositionOnline)) {
    //            throw new BadCommandException("Wrong position for autochanger trucks. Change to engineering mode.");
    //        }
        openOnlineClamp();
        goToPosition(this.getTrucksPositionAtStandby());
        if (getTrucksPosition() == this.getTrucksPositionAtStandby()) {
            log.info(getName() + ": " + this.filterOnTrucks.getName() + " is at standby position.");
            return getName() + ": " + this.filterOnTrucks.getName() + " is at standby position.";
        } else {
            String message = "the autochanger truck is not at the required position.";
            log.error(message);
            throw new ErrorInCommandExecutionException(message);
        }
    }


    public Filter getFilterOnline() {
        if ((getTrucksPosition() == this.getTrucksPositionOnline()) && (!this.isMoving())) {
            return filterOnTrucks;
        } else {
            return null;
        }
    }

    public String getFilterOnlineName() {
        if (getFilterOnline() == null) {
            return "none";
        } else {
            return getFilterOnline().getName();
        }
    }

    @Override
    public double getTrucksPosition() {
        return this.motor.getPosition();
    }

    public double getTrucksPositionAtStandby() {
        return trucksPositionAtStandby;
    }

    public double getTrucksPositionOnline() {
        return trucksPositionOnline;
    }

    public double getTrucksPositionSwapout() {
        return trucksPositionSwapout;
    }


    /**
     * This method has to be executed at the FCS startup.
     * It checks the hardware trucks and set the trucks position.
     */
    @Override
    public synchronized void locateTrucks() {
        //TODO IN REAL LIFE : check if the trucks are empty or not
        this.filterOnTrucks = null;
        setTrucksEmpty(true);
    }

    public void setTrucksPositionAtStandby(double trucksPositionAtStandby) {
        this.trucksPositionAtStandby = trucksPositionAtStandby;
    }

    public void setTrucksPositionOnline(double trucksPositionOnline) {
        this.trucksPositionOnline = trucksPositionOnline;
    }

    public void setTrucksPositionSwapout(double trucksPositionSwapout) {
        this.trucksPositionSwapout = trucksPositionSwapout;
    }

    public Motor getMotor() {
        return motor;
    }

    public void setMotor(Motor motor) {
        this.motor = motor;
    }

    public boolean isMoving() {
        return this.motor.getEngineState().equals(EngineState.RUNNING);
    }

    public boolean isAtStandby() {
        return this.getTrucksPosition() == this.trucksPositionOnline;
    }

    @Override
    public String goToStandback() throws BadCommandException, ErrorInCommandExecutionException {
        return this.goToPosition(this.trucksPositionSwapout);
    }
    
    
    @Override
    public String goToStandby() throws BadCommandException {
        return this.goToPosition(this.trucksPositionAtStandby);
    }

    @Override
    public String moveFilterToStandback(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException {
        return this.moveFilterToOnline(aFilter);
    }

    @Override
    public boolean isMovingToStandback() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public boolean isMovingToStandby() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    @Override
    public void updateStateWithSensors() throws FcsHardwareException, ErrorInCommandExecutionException {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    void goToHandOff() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    /**
     * 
     * @return true if the trucks are at standby position and hold a filter.
     */
    @Command ( level=Command.ENGINEERING1, description="Return true if the trucks are at HANDOFF position and hold a filter", type=Command.CommandType.QUERY)
    public boolean isHoldingFilterAtHandoff() {
        if (isMoving()) return false;
        if (!isAtHandoff()) return false;
        if (isTrucksEmpty()) return false;
        return this.getLatchesState().equals(FcsEnumerations.LockStatus.LOCKED);
    }

    public void closeLatchesAtHandoff() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    void moveFilterToHandoff(Filter aFilter) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }

    boolean isAtStandoff() {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }





}
