/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.UnsupportedEncodingException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.errors.CWrapperNotConnected;
import org.lsst.ccs.utilities.beanutils.WrappedException;

/**
 * This Modules : 
 * - start a tcpip server on a portNumber, 
 * - waits for the connection of the client (which is supposed to be a Can OPen C wrapper), 
 * - send commands received by the call method to the client and 
 * - send back the client response. 
 * - it handles the asynchronous messages coming from the client and for that it
 * starts a thread which listen to the client. 
 * @author virieux
 */
abstract class FcsTcpProxy extends Module {

    private final int portNumber;
    private ServerSocket serverSock;
    private Thread readerThread;
    private volatile boolean stopped = false;
    protected volatile boolean stopping = false;
    protected volatile boolean tcpServerStarted = false;
    protected ClientContext clientContext;
    
    
    
    private CommandDispenser commandDispenser;
    



    
    final Lock lock = new ReentrantLock();
    final Condition aClientIsConnected = lock.newCondition();

    protected static class ClientContext {
        String clientName;
        BufferedReader reader;
        BufferedWriter writer;
        Socket socket;

        ClientContext(String name, Socket socket, BufferedReader reader, OutputStream os) {
            try {
                this.clientName = name;
                this.reader = reader;
                this.socket = socket;
                writer = new BufferedWriter(new OutputStreamWriter(os, "ISO-8859-1"), 256);
            } catch (UnsupportedEncodingException e) {
                /*. SHOULD NOT HAPPEN */
                log.error(name + ":context not started", e);
                throw new Error(e);
            }
        }
    }
    


    
    public FcsTcpProxy(String aName, int aTickMillis, int portNumber) {
        super(aName, aTickMillis);
        this.portNumber = portNumber;
    }

    public boolean isTcpServerStarted() {
        return tcpServerStarted;
    }


    
    
    @Override
    public void initModule() {
        stopped=false;
        stopping=false;
        tcpServerStarted= false;
        commandDispenser = new CommandDispenser();
    }
    
    @Override
    public void start() {
        startServer();
        startThreadReader();
    }
    
    /**
     * Starts the server tcp on the port portNumber.
     * And waits for a client connection.
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Starts the tcp server.")
    public void startServer() {
        try {
            serverSock = new ServerSocket(portNumber);
            log.info(name + ":SERVER STARTED ON PORT:" + portNumber);
        } catch (IOException e) {
            log.error("server not started",  e);
            throw new WrappedException(e) ;
        }
        
        try {
            log.info(name + "WAITING FOR C-WRAPPER CLIENT...");
            log.debug("DEBUGGING");
            Socket sock = serverSock.accept();
            log.info("socket accept on " + portNumber);
            BufferedReader reader = new BufferedReader(
                    new InputStreamReader(sock.getInputStream(), "ISO-8859-1"), 256);
            String nameAndProtocol = reader.readLine();
            if (nameAndProtocol == null) {
                            log.error(name + " : nameAndProtocol is null : that should not happen.");
            } else {
                String[] words = nameAndProtocol.split(" ") ;
                String cname = words[0] ;

                clientContext = new ClientContext(cname, sock, reader, sock.getOutputStream());
                log.info("REGISTERED : " + cname);
                tcpServerStarted = true;
            }           
                        
        } catch (IOException e) {
            // LOG
            log.warn(" unexpected ", e);
        }
    }

    
    
     /**
     * Starts a thread which read on the tcp socket, waiting for messages coming 
     * from tcp proxy.
     * If the message has a known token (already registred in commandDispenser)
     * it is a response to a can open command that we had sent to the proxy.
     * If the token of the message is null, that means that is's an unsynchronous 
     * message coming from the can open stack and transmitted directly from the 
     * tcp proxy.
     * That can be :
     * - a boot message : "boot,nodeID"
     * - an emergency message : "emcy,80+NodeID"
     * 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Starts to listen to the tcp client.")
    public void startThreadReader() {
        try {
            if (!isTcpServerStarted()) throw new BadCommandException(name + " has to be started first.");
            /*We start thread to read from tcpProxy*/
            Runnable readingFromTcpProxy = new Runnable() {
                @Override
                public void run() {
                    while (!stopped) {
                        try {
                            //log.info(Thread.currentThread().toString());
                            String readline = clientContext.reader.readLine() ;
                            if (readline == null) {
                                if (stopping) {
                                    log.info(name + " : CWrapper is stopped.");
                                } else {
                                    log.error(name + " :nameAndProtocol is null : that should not happen.");
                                }
                            } else {
                                
                                //String readline = getClientContext().reader.readLine();
                                //log.debug(name + " :message read from tcpProxy=" + readline);
                                
                                if (readline.startsWith("emcy")) {
                                    processEmcyMessage(readline);
                                    //                        } else if (readline.startsWith("sync")){
                                    //                            processSyncMessage(readline);
                                } else if (readline.startsWith("Unknown")){
                                    processUnknownCommand(readline);
                                } else {
                                    String[] words = readline.split(",");
                                    String commandWord= words[0];
                                    String nodeID = words[1];
                                    if (commandWord.equals("boot")) {
                                        log.info("boot command received for node: " + nodeID );
                                        processBootMessage(nodeID);
                                    } else {
                                        String token = commandWord+nodeID;
                                        if (readline.startsWith("sync")){
                                            token = "sync";
                                        } else {
                                            token = commandWord+nodeID;
                                        }
                                        //log.debug("Corresponding token=" + token);
                                        if (commandDispenser.isTokenUsed(token)) {
                                            //log.debug("Response to a registred command:" + readline);
                                            try {
                                                commandDispenser.registerResponse(token, readline);
                                            } catch (Exception ex) {
                                                log.error("Error in registering response to command :" + readline,ex.getCause());
                                            }
                                        } else {
                                            log.info("Unsynchronous message read from tcpProxy=" + readline);
                                            
                                        }
                                    }
                                }
                            }
                            
                            
                        } catch (IOException ex) {
                            log.error(name + ": Error in Thread reading from the tcp client.");
                        }
                    }
                }
            };
            log.info("STARTING Thread reader");
            readerThread = new Thread(readingFromTcpProxy);
            readerThread.start();
        } catch (BadCommandException ex) {
            log.error(name + " tcp server has to be started before startThreadReader.",ex);
            this.getSubsystem().raiseAlarm(ex.toString());
        }
    }
    
     
//   @Override
//   public void shutdownNow() {
//        super.shutdownNow();
//        this.stopServer();
//    }
    
   /**
    * Stops the tcp server.
    */
    public void stopServer() {
        log.info(name + ": ABOUT TO STOP TCP server.....");
        //log.info("clientName=" + clientContext.clientName);
        stopping = true;
        try {
            call(clientContext.clientName, "quit");

        } catch (TimeoutException ex) {
            log.error(name + ": could not stop Cwrapper.");
        } catch (CWrapperNotConnected ex) {
            log.error(name + ": could not stop because CWrapper is not connected.");
        }
        
        stopped = true ;
        readerThread.interrupt();
        
        try {
            serverSock.close();
            this.tcpServerStarted = false;
            log.info(name + ": Subsystem " + this.getSubsystem().getName() + " TCPIP SERVER STOPPED:");
        } catch (IOException ex) {
            log.error(name + " : stop method does not work properly when closing socket");
            
        }

        try {
            clientContext.reader.close();
            clientContext.writer.close();
            clientContext.socket.close();

        } catch (IOException exc) {
           // FORGET IT
            log.error(name + " : stop method does not work properly");
        }
      
    }
    

    
    public synchronized boolean isReady(String clientName) {
        if (clientName == null) throw new IllegalArgumentException("Client name can't be null.");
        if (clientContext == null || clientContext.clientName == null) return false;
        if (stopped) return false;
        if (!tcpServerStarted) return false;
        log.debug(name + "client name:= " + clientContext.clientName + "#");
        return  clientContext.clientName.equals(clientName) ;
    }
    
    /**
     * This methods send a command to the tcp client.
     * The command is a String and can be understood by the client.
     * Example for a Can Open command : rsdo,1,1018,0
     * The command is stored in the command dispenser with a token and command
     * corelation. When the response comes back from the proxy we can retrieve it
     * from the dispenser with the call token.
     * @param clientName
     * @param command
     * @return result of the call or null if the call failed
     */
    public Object call(String clientName, String command) throws TimeoutException, CWrapperNotConnected   {
        
        //log.debug(" CALL :" + clientName + " " +command);
        if(clientContext == null) {
            throw new CWrapperNotConnected(this.portNumber,clientName);
        }
        if(stopped) {
            throw new IllegalStateException("textTCP module stopped") ;
        }
        
 
        try {

            String sentCommand = command +"\r\0\n" ;
            //log.info("Sent command=" + command);
            clientContext.writer.write(sentCommand,0,sentCommand.length());
            clientContext.writer.flush() ;

             //if we want to wait for the response we have to store this command.
            if (command.startsWith("rsdo") 
                    || command.startsWith("wsdo") 
                    || command.startsWith("info")
                    || command.startsWith("sync")
                    || command.startsWith("srtr")) {
                String commandToken = commandDispenser.register(command);
                try {
                    //TODO put the following timeout in configuration file
                    long timeout = 5 * this.tickMillis;
                    return commandDispenser.getCommandResponse(commandToken, timeout);

                } catch (TimeoutException e) {
                    log.error(" ERROR in tcpProxy call method for command :" + command );
                    throw new TimeoutException("TIMEOUT expired in tcpProxy call method for command :" + command);
                } finally {
                    commandDispenser.remove(commandToken);
                }
            } else {
                return "asynchronous command sent:" + command;
            }


        } catch (IOException e) {
            log.error(" ERROR in tcpProxy call method" );
            throw new WrappedException( " ERROR in tcpProxy call method " + clientName, e);
        }
        
    }
    
    /**
     * Process a boot message coming from the can open stack.
     * 
     * @param nodeID 
     */
    abstract void processBootMessage(String nodeID);
    
     /**
     * Process an emergency message coming from the can open stack.
     * 
     * @param nodeID 
     */
    abstract void processEmcyMessage(String message);
    

    
    
     /**
     * Process a response to unknown command.
     * 
     * @param nodeID 
     */
    abstract void processUnknownCommand(String message);
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/port=");sb.append(this.portNumber);
        return sb.toString();       
    }
      
    
}
