/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.beans.ConstructorProperties;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import java.math.BigInteger;
import java.util.Map;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * A general model for a controller EPOS.
 * @author virieux
 */
public class CanOpenEPOS extends CanOpenDevice implements Actuator, EPOSController {
    
    public boolean on;
    public int readValue;
    public int sentValue;
    
     /** The value of current that has to be sent to the motor to make it move.
     * UNIT= mA given in decimal mode
     */
    public int currentValue;
    
     /** position we want to reach
     * UNIT= mA given in decimal mode
     */
    public int targetPosition;







    
    /**
     * Here are the parameters that we can set in an Epos controller.
     * cf EPOS Application Notes Collection
     */
    public enum Parameter {
        MaxFollowingError("6065","00", "2",false),
        HomeOffset("607C","00","4",true),
        MaxProfileVelocity("607F","00", "2",false),       
        SpeedForSwitchSearch("6099","01", "2",false),
        SpeedForZeroSearch("6099","02", "2",false),
        HomingAcceleration("609A","00", "2",false),
        CurrentThresholdHomingMode("2080","00", "1",false),
        HomePosition("2081","00", "4",false),
        HomingMethod("6098","00","1",true),
        MinPositionLimit("607D","01", "4",true),
        MaxPositionLimit("607D","02", "4",true),
        PositionActualValue("6064","00","4",true),
        ProfileVelocity("6081","00", "4",false),
        ProfileAcceleration("6083","00", "4",false),
        ProfileDeceleration("6084","00", "4",false),
        QuickStopDeceleration("6085","00", "4",false),
        MotionProfileType("6086","00", "2",false),
        MotorType("6402","00","2",false),
        ContinuousCurrentLimit("6410","01", "2",false),
        OutputCurrentLimit("6410","02","2",false),
        PolePairNumber("6410","03","1",false),
        MaxSpeedInCurrentMode("6410","04", "4",false),
        ThermalTimeConstantWinding("6410","05", "2",false),
        PositionSensorType("2210", "02", "2", false),
        ;
        
        String index;
        String subindex;
        /* Size of the parameter : numbers of bytes*/
        String size; 
        boolean signed;


        private Parameter(String index, String subindex, String size, boolean signed) {
            this.index = index;
            this.subindex = subindex;
            this.size = size;
            this.signed = signed;

        }
        
        public String display() {
            StringBuilder sb = new StringBuilder(this.toString());
            sb.append("/index=").append(this.index);
            sb.append("/subindex=").append(this.subindex);
            sb.append("/size=").append(this.size).append(" bytes");
            if (signed) sb.append("/SIGNED"); else sb.append("/UNSIGNED");
            return sb.toString();
        }
        
        
    }
    
    /**
     * Here are all the modes that can be configurated for an EPOS controller.
     * For each mode, some parameters have to be set : field parameters
     */
    public enum EposMode {
        HOMING("6", Parameter.MaxFollowingError, Parameter.HomeOffset, Parameter.MaxProfileVelocity,
                    Parameter.QuickStopDeceleration, Parameter.SpeedForSwitchSearch, 
                    Parameter.SpeedForZeroSearch, Parameter.HomingAcceleration,
                    Parameter.CurrentThresholdHomingMode, Parameter.HomePosition),
        PROFILE_POSITION("1", Parameter.MaxFollowingError, Parameter.MinPositionLimit,
                    Parameter.MaxPositionLimit, Parameter.MaxProfileVelocity,
                    Parameter.ProfileVelocity, Parameter.ProfileAcceleration, 
                    Parameter.ProfileDeceleration, Parameter.QuickStopDeceleration,
                    Parameter.MotionProfileType),
        PROFILE_VELOCITY("3"),
        POSITION("FF"),
	VELOCITY("FE"), 
        CURRENT("FD", Parameter.ContinuousCurrentLimit, Parameter.OutputCurrentLimit, Parameter.MaxSpeedInCurrentMode, 
                        //the following parameters are specific to the motor that we control with this controller.
                        Parameter.PolePairNumber, Parameter.ThermalTimeConstantWinding, Parameter.MotorType, Parameter.PositionSensorType); 
      
        private final String modeInHexa;
        private final Parameter[] parameters;

        private EposMode(String modeToHexa, Parameter... params) {
            this.modeInHexa = modeToHexa;
            this.parameters = params;
        }
        
        String getModeInHexa() {
            return this.modeInHexa;
        } 
        
        /**
         * Retrieve and returns a mode which hexa code is given as an argument. 
         * @param hexa mode code in hexa
         * @return mode
         */
        public static EposMode getMode(String hexa) {
            EposMode[] modes = EposMode.values();
            boolean modeFound = false;
            for (EposMode mode : modes) {
                 if (mode.getModeInHexa().equals(hexa.toUpperCase())) {
                     modeFound = true;
                     return mode;                    
                 }
            }
            if (!modeFound) throw new IllegalArgumentException(hexa + ": this mode doesn't exist in CanOpenEpos.EposMode ENUM");
            return null;
            
        }
    }
    
    /* Mode of Operation */
    protected EposMode mode;
    
    /* Parameters */
    Map<String,Integer> paramsForHoming;   
    Map<String,Integer> paramsForProfilePosition;
    Map<String,Integer> paramsForProfileVelocity;
    Map<String,Integer> paramsForPosition;
    Map<String,Integer> paramsForVelocity;
    Map<String,Integer> paramsForCurrent;
    
    //Map<String,Integer>[] paramsMaps;
    
    
    /*the motor which is driven by this actuator*/
//    final protected Motor motor;
    
    //Used because we have to wait during the time needed to enable the actuator (very short but not null !)
    //of the clamp
    final Lock lock = new ReentrantLock();
    final Condition enablingCompleted = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean enabling = false;

//    @ConstructorProperties({"name","tickMillis","nodeID","serialNB",
//            "paramsForCurrent","paramsForProfilePosition","paramsForHoming",
//            "mode","currentValue"})
    public CanOpenEPOS(String aName, int aTickMillis, 
            String nodeID, String serialNB,
            Map<String, Integer> paramsForCurrent, 
            Map<String, Integer> paramsForProfilePosition, 
            Map<String, Integer> paramsForHoming,
            CanOpenEPOS.EposMode aMode,
            int currentValue) {
        super(aName, aTickMillis, nodeID, serialNB);
        this.mode = aMode;   
        this.paramsForCurrent = paramsForCurrent;
        this.paramsForProfilePosition = paramsForProfilePosition;
        this.paramsForHoming = paramsForHoming;
        this.currentValue = currentValue;

    }
    
        @Override
    public boolean isOn() {
        return on;
    }

    @Override
    public boolean isOff() {
        return !on;
    }
     
    
    @Override
    public int getReadValue() {
        return readValue;
    }

    @Override
    public int getSentValue() {
        return sentValue;
    }

    @Override
    //for the GUI 
    //TODO return a real value
    public int getSentCurrentMinValue() {
        return 0;
    }

    @Override
    //for the GUI
    //TODO return a real value
    public int getSentCurrentMaxValue() {
        return 3000;
    }
    
    
    public int getMotorParameterValue(String parameterName) throws EPOSConfigurationException {
        if (this.paramsForCurrent.containsKey(parameterName)) {
            return this.paramsForCurrent.get(parameterName);
        } else throw new EPOSConfigurationException(parameterName + " :undefined",getName());
    }
    
    /**
     * Print the configured parameters for this EPOS for a mode name given in argument.
     * This methods doesnt't read the CPU of the EPOS.
     * So if no writeParameters(modeInString) command has been done before,
     * the values in the EPOS CPU can be different.
     * @param modeInString
     * @return 
     */
    public String printParameters(String modeInString){
        EposMode aMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        Map<String,Integer> paramsMap = null;
        switch(aMode) {
           case HOMING : 
               paramsMap = this.paramsForHoming;
               break;
           case PROFILE_POSITION : 
               paramsMap = this.paramsForProfilePosition;
               break;
           case PROFILE_VELOCITY : 
               paramsMap = this.paramsForProfileVelocity;
               break;
           case POSITION : 
               paramsMap = this.paramsForPosition;
               break;
           case VELOCITY : 
               paramsMap = this.paramsForVelocity;
               break;
           case CURRENT : 
               paramsMap = this.paramsForCurrent;
               break;
        }
        return printParameters(paramsMap);
    }
    
    public String printParameters(Map<String,Integer> paramMap) {
        StringBuilder sb = new StringBuilder(name);
        sb.append("Parameters for CURRENT mode in configuration file:\n");
        for (Map.Entry<String,Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue().intValue();  
            sb.append("key=");sb.append(paramName);
            sb.append("/value=");sb.append(value);sb.append("\n");
        }
        return sb.toString();
    }
    
  
    
    // All the getters following are for the GUI needs
    @Deprecated
    public int getMotor_continuousCurrentLimit() throws SDORequestException {
        return this.readParameter(Parameter.ContinuousCurrentLimit);
        //return motor.getContinuousCurrentLimit();
    }

    @Deprecated
    public int getMotor_maxSpeedInCurrentMode() throws SDORequestException {
        return this.readParameter(Parameter.MaxSpeedInCurrentMode);
        //return motor.getMaxSpeedInCurrentMode();
    }


    @Deprecated
    public int getMotor_outputCurrentLimit() throws SDORequestException {
        return this.readParameter(Parameter.OutputCurrentLimit);
        //return motor.getOutputCurrentLimit();
    }

    @Deprecated
    public int getMotor_polePairNumber() throws SDORequestException {
        return this.readParameter(Parameter.PolePairNumber);
        //return motor.getPolePairNumber();
    }

    @Deprecated
    public int getMotor_thermalTimeConstantWinding() throws SDORequestException {
        return this.readParameter(Parameter.ThermalTimeConstantWinding);
        //return motor.getThermalTimeConstantWinding();
    }

    @Deprecated
    public int getMotor_type() throws SDORequestException {
        return this.readParameter(Parameter.MotorType);
        //return motor.getMotorType();
    }

    
    /**
     * Reads the EPOS mode in the CPU of the EPOS device, updates the field mode and returns
     * the mode as a String.
     * @return
     * @throws SDORequestException 
     */
    public String readMode() throws SDORequestException {
        String modeInHexa = this.readSDO("6060", "0").trim().toUpperCase();
        
        EposMode readMode = EposMode.getMode(modeInHexa);

        this.mode = readMode;
        return readMode.toString();
    }
    
    /**
     * Changes the mode to the new mode given as a String argument.
     * @param modeInString
     * @return
     * @throws SDORequestException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Change EPOS mode .")
    public String changeMode(String modeInString) throws SDORequestException {
        EposMode newMode = EposMode.valueOf(modeInString.trim().toUpperCase());
        return changeMode(newMode);
    }
    
    /**
     * This methods changes the mode to the new mode given as an argument.
     * It writes the new mode in the CPU of the EPOS controller and updates the field mode.
     * @param newMode
     * @return
     * @throws SDORequestException 
     */
    @Override
    public String changeMode(EposMode newMode) throws SDORequestException {
        this.writeSDO("6060", "0", "1", newMode.modeInHexa);
        this.mode = newMode;
        return getName() + " mode has been changed to: " + newMode.toString() + ". Please check parameters associated with this mode.";
    }
    

    /**
     * Displays the list of parameters that we have to define for a given mode.
     * @param modeInString
     * @return 
     */
    public static String displayListParameters(String modeInString) {
        StringBuilder sb = new StringBuilder("List of parameters for mode: "); sb.append(modeInString);sb.append("\n");
        Parameter[] params = EposMode.valueOf(modeInString).parameters;
        for (Parameter param : params) {
            sb.append(param.display());sb.append("\n");
        }
        return sb.toString(); 
    }
    
     /**
     * Displays the list of parameters.
     * @return 
     */
    public static String displayListParameters() {
        StringBuilder sb = new StringBuilder("List of parameters : "); sb.append("\n");
        Parameter[] params = Parameter.values();
        for (Parameter param : params) {
            sb.append(param.display());sb.append("\n");
        }
        return sb.toString(); 
    }
    
    /**
     * This methods reads in the CPU of the EPOS the values of the parameters for a given mode.
     * @param modeInString
     * @return
     * @throws SDORequestException 
     */
    public String readParameters(String modeInString) throws SDORequestException {
        StringBuilder sb = new StringBuilder("Value set for the parameters for the mode : ");sb.append(modeInString);sb.append("\n");
        Parameter[] params = EposMode.valueOf(modeInString).parameters;
        for (Parameter param : params) {
            
            String valueInHexa = readSDO(param.index,param.subindex);
            int valueInt = Integer.parseInt(valueInHexa, 16);
            sb.append(param.toString());
            sb.append("=");sb.append(valueInt);sb.append("\n");
        }
        return sb.toString();
    }
    
    public String readParameters() throws SDORequestException {
        String modeInString = readMode();
        return readParameters(modeInString);
    }
    
    public void writeParameters(String modeInString) throws SDORequestException, ErrorInCommandExecutionException {
        EposMode aMode = EposMode.valueOf(modeInString);
        writeParameters(aMode);
    }
    
    /**
     * This methods writes in the CPU of the EPOS devis the values of the parameters set for the mode.
     * Usually the values of the parameters are given by the configuration system.
     * @param mode
     * @throws SDORequestException 
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException 
     */
    @Override
    public void writeParameters(EposMode mode) throws SDORequestException, ErrorInCommandExecutionException {
        String modeInString = mode.toString();
        Map<String,Integer> paramsMap = null;
        switch(mode) {
           case HOMING : 
               paramsMap = this.paramsForHoming;
               break;
           case PROFILE_POSITION : 
               paramsMap = this.paramsForProfilePosition;
               break;
           case PROFILE_VELOCITY : 
               paramsMap = this.paramsForProfileVelocity;
               break;
           case POSITION : 
               paramsMap = this.paramsForPosition;
               break;
           case VELOCITY : 
               paramsMap = this.paramsForVelocity;
               break;
           case CURRENT : 
               paramsMap = this.paramsForCurrent;
               break;
        }
        if (paramsMap == null) throw new ErrorInCommandExecutionException(getName() + "parameters for mode :" + modeInString + "are NULL");
        if (paramsMap.isEmpty()) throw new ErrorInCommandExecutionException(getName() + "parameters for mode :" + modeInString + "are not defined.");
//        for (Map.Entry<String,Integer> entry : paramsMap.entrySet()) {
//            String paramInString = entry.getKey();
//            int value = entry.getValue().intValue();
//            String valueInHexa = Integer.toHexString(value);
//            Parameter param = Parameter.valueOf(paramInString);
//            writeSDO(param.index,param.subindex,param.size,valueInHexa);           
//        }
        writeParameters(paramsMap);
        
    }
    
    /**
     * Write to the CPU of the EPOS a map of parameters.
     * @param paramMap
     * @throws SDORequestException 
     */
    @Override
    public void writeParameters(Map<String,Integer> paramMap) throws SDORequestException {
        for (Map.Entry<String,Integer> entry : paramMap.entrySet()) {
            String paramName = entry.getKey();
            int value = entry.getValue().intValue();
//            String valueInHexa = Integer.toHexString(value);
//            Parameter param = Parameter.valueOf(paramInString);
//            writeSDO(param.index,param.subindex,param.size,valueInHexa);  
            writeParameter(paramName,value);
        }
    }
    
    
    /**
     * This method writes a parameter in the CPU of the EPOS.
     * @param parameterName the name of the parameter
     * @param value in decimal
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
     @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Set EPOS parameter with the value given as argument.")
    public String writeParameter(String parameterName, int value) throws SDORequestException {
        Parameter param = Parameter.valueOf(parameterName);
        return this.writeParameter(param, value);
    }
    
    public String writeParameter(Parameter param, int value) throws SDORequestException {
         writeSDO(param.index,param.subindex,param.size,Integer.toHexString(value));
        return String.format("%s has been written to the CanEpos CPU, value=%d", param.toString(),value);
    }
    
    /**
     * Reads in the EPOS CPU the value of the Parameter which parameter name is given as argument.
     * @param parameterName
     * @return value of the parameter in decimal format.
     * @throws SDORequestException 
     */
    public int readParameter(String parameterName) throws SDORequestException {
        Parameter param = Parameter.valueOf(parameterName);
        return readParameter(param);
    }
    
    /**
     * Reads in the EPOS CPU the value of the Parameter.
     * @param param
     * @return value of the parameter in decimal format.
     * @throws SDORequestException 
     */
    public int readParameter(Parameter param) throws SDORequestException {
        String valueInHexa = readSDO(param.index,param.subindex);
        //return Integer.parseInt(valueInHexa, 16);
        return new BigInteger(valueInHexa,16).intValue();
    }
    

    @Override
    public boolean isTargetPositionReached(int position) throws SDORequestException {
        int myPosition = 0;

        myPosition = this.readPosition();
        
        return myPosition == position;
        
    }
    
     /**
     * Reads the current value or the velocity value stored in the EPOS and updates the readValue field.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException     */
    public void updateReadValue() throws BadCommandException, SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            this.readValue = readVelocity();
        } else if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
            this.readValue =  readCurrent();
        }
        //} else throw new BadCommandException(getName() + "is not in VELOCITY or CURRENT mode");
    } 
    
       @Override
    public void initModule() {
        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        //mode = CanOpenEPOS.EposMode.CURRENT;
    }
    
    @Override
    public String initializeHardware() throws Exception {
        this.initialized = true;
        return getName() + ": no hardware initialization for this device to be done.";
    }

    
    public boolean isEnable() throws SDORequestException {
        lock.lock();
        
        try {
            while(this.enabling) {
                try {
                    this.enablingCompleted.await();
                } catch (InterruptedException ex) {
                    log.debug(getName() + " INTERRUPTED during waiting for the actuator beeing enable");
                }

            }
        String controlWordInHexa = readControlWord();
        int controlWord = Integer.parseInt(controlWordInHexa, 16);
        return (controlWord == 15); /*F in hexa*/
        } finally {
            lock.unlock();
        }
    }
    
    public void faultReset() throws SDORequestException {
        writeControlWord("80");
    }
  
    /**
     * This methods enable the actuator : i.e. this makes the actuator able to receive commands.
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
    @Override
    public String enable() throws SDORequestException {
        lock.lock();
              
        try {
            this.enabling = true;
            shutdown();
            switchOnEnableOperation();
            return getName() + " ENABLE";
        } finally {

            this.enabling = false;
            this.enablingCompleted.signal(); 
            lock.unlock();
        }
    }
    
    
    public String shutdown() throws SDORequestException{      
        writeControlWord("6");
        //TODO test the return code for the command writeSDO
        return getName() + " DISABLE";
    }
    
    public void switchOnEnableOperation() throws SDORequestException {
        writeControlWord("F");
    }
    
    /**
     * This method save the parameters in the actuator memory.
     * @return 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
    public Object saveParameters() throws SDORequestException {
        return this.writeSDO("1010", "1", "4", "65766173");
    }
    
    public Object restoreParameters() throws SDORequestException {
        //TODO if is Power Disable
        return this.writeSDO("1011", "1", "4", "64616F6C");
    }
    
    /**
     * Write a value in hexa in the control world (index=6040, subindex=0,size=2)
     * @param value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public void writeControlWord(String value) throws SDORequestException {
        this.writeSDO("6040", "0", "2", value);
    }
    
     /**
     * Read the control world (index=6040, subindex=0,size=2)
     * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    public String readControlWord() throws SDORequestException {
        return this.readSDO("6040", "0");
    }
    
    /**
    * Read the status world (index=6041, subindex=0,size=2)
    * @return value in hexa
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
    */
    public String readStatusWord() throws SDORequestException {
        return this.readSDO("6041", "0");
    }
    

    

    
    
    public void quickStop() throws SDORequestException {
//        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
//            writeControlWord("B");
//        } else if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
//            writeControlWord("2");
//        } //TODO else throw an exception ?
        switch(this.mode) {
           case HOMING : 
                writeControlWord("B");
                break;
           case PROFILE_POSITION :
                writeControlWord("B");
                break;
           case PROFILE_VELOCITY : 
                break;
           case POSITION : 
                break;
           case VELOCITY :
                writeControlWord("B");
                break;
           case CURRENT :
                writeControlWord("2");
                break;
        }
    }
    
     /**
     * This methods enables the actuator and puts the motor on in sending the appropriate current value 
     * to the actuator. 
     * 
     * @return a message 
     */
    public String onOld() throws BadCommandException, EPOSConfigurationException, ErrorInCommandExecutionException {
        try {
            
            this.enable(); 
            
            if (!isEnable()) throw new BadCommandException("Actuator has to be enable prior ON command.");
            

            if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
                    setVelocity(); 
            } else if (mode.equals(
                    CanOpenEPOS.EposMode.CURRENT)) { 
                    if ( (this.currentValue == 0)) {
                        throw new BadCommandException("A current value has to be set");
                    }
                    writeCurrent(this.currentValue);
               
            }
            this.on = true;
            this.sentValue = this.currentValue;
            this.updateReadValue();
            //this.sendToStatus(this.getStatusData());
            return getName() + " ON";
        } catch (SDORequestException ex) {
            log.error(getName() + " Can Open ERROR in sending the command ON.");
            throw new ErrorInCommandExecutionException(getName() + ": Error in command ON" + ex.toString());
        }

    }
    
    @Override
    public String on() throws SDORequestException, BadCommandException, ErrorInCommandExecutionException {
        
        this.enable(); 
            
        if (!isEnable()) throw new BadCommandException(getName() + " Controller couln'd be enabled in ON command.");
        
        switch(this.mode) {
           case HOMING : 
                this.writeControlWord("1F");
                break;
           case PROFILE_POSITION : 
               //writeControlWord 1F : absolute pos.
               //writeControlWord 3F : absolute pos., start immediately
               //writeControlWord 7F : relative pos., start immediately
               //writeControlWord 5F : relative positioning
                this.writeControlWord("3F");
                break;
           case PROFILE_VELOCITY : 
                break;
           case POSITION : 
                break;
           case VELOCITY :
                    setVelocity();
                break;
           case CURRENT :
                if ( (this.currentValue == 0)) {
                    throw new BadCommandException("A current value has to be set");
                }
                writeCurrent(this.currentValue);
                this.sentValue = this.currentValue;
                break;
                }
        this.on = true;       
        this.updateReadValue();
        //this.sendToStatus(this.getStatusData());
        return getName() + " ON";
    }
    

    
        /**
     * This method set to zero the current value and stops the motor motion.
     * Then it disables the actuator.
     * @return 
     * @throws org.lsst.ccs.bus.BadCommandException 
     * @throws org.lsst.ccs.bus.ErrorInCommandExecutionException 
     */
    public String offOld () throws BadCommandException, ErrorInCommandExecutionException {
        try {
            if (mode.equals(EposMode.VELOCITY)) {
               
                    stopVelocity();
                    shutdown();
                
            } else if (mode.equals(EposMode.CURRENT)) {

                    stopCurrent();
                    shutdown();
                
            }
            this.on = false;
            this.sentValue = 0;
            this.updateReadValue();
            //this.sendToStatus(this.getStatusData());
            return getName() + " OFF";
        } catch (SDORequestException ex) {
            log.error(ex.getMessage());
            throw new ErrorInCommandExecutionException(ex + "Error in reading SDO request");
        }
    }
    
    
    @Override
    public String off() throws BadCommandException, SDORequestException {
       switch(this.mode) {
           case HOMING : 
                this.writeControlWord("11F");
                break;
           case PROFILE_POSITION : 
                this.writeControlWord("10F");
                break;
           case PROFILE_VELOCITY :                
                break;
           case POSITION : 
                break;
           case VELOCITY :
                    stopVelocity();
                break;
           case CURRENT :
                stopCurrent();
                break;
        }
        shutdown();
        this.on = false;
        this.sentValue = 0;
        this.updateReadValue();
        log.info(name + " is TURN OFF");
        return name + " OFF";
    }
    


    
    /*********************************/
    /*Methods connected to the motors*/
    /*********************************/
    
    /**
     * Write the values of the motor fields in the actuator memory.
     * @throws SDORequestError 
     */
//    //the motor parameters are now (since June 2014) included in ParamsForCurrent
//    public void writeMotorParameters() throws SDORequestException {
//        shutdown();    
//        writeMotorType(this.getMotorType());
//        
//        // mandatory in Current mode
//        writeMotorData("01","4", this.motor.getContinuousCurrentLimit());
//        writeMotorData("04","4", this.motor.getMaxSpeedInCurrentMode());
//        writeMotorData("05","2", this.motor.getThermalTimeConstantWinding());
//        //Guillaume wants to set this parameters too :
//        writeMotorData("02","2", this.motor.getOutputCurrentLimit());
//        writeMotorData("03","1", this.motor.getPolePairNumber());
//        writePositionSensorType(this.motor.getPositionSensorType());
//              
//        //TODO mandatory in velocity mode
//        
//    }
    
    /**
     * This command set the motor type on the actuator.
     */
    public void writeMotorType(int value) throws SDORequestException {
        String typeInHexa = Integer.toHexString(value);
        writeSDO("6402","0","2",typeInHexa);
    }
    
    /**
     * This command read the motor type on the actuator.
     * @return the motor type in decimal
     */
    @Deprecated
    public int readMotorType() throws SDORequestException {
//        String motorTypeInHexa = readSDO("6402","0");
//        return Integer.parseInt(motorTypeInHexa, 16);
        return this.readParameter(Parameter.MotorType);
    }
    
     /**
     * This command read the position sensor type on the actuator.
     * @return the position sensor type in decimal
     */
    @Deprecated
    public int readPositionSensorType() throws SDORequestException {
//        String positionSensorTypeInHexa = readSDO("2210","02");
//        return Integer.parseInt(positionSensorTypeInHexa, 16);
        return this.readParameter(Parameter.PositionSensorType);
    }
    
    @Deprecated
    public void writePositionSensorType(int value) throws SDORequestException {
//        this.motor.setPositionSensorType(value);
//        String valueInHexa = Integer.toHexString(value);
//        writeSDO("2210","2","2", valueInHexa);
        this.writeParameter(Parameter.PositionSensorType, value);
    }
    
    @Deprecated
    public void writeMaxSpeed(int value) throws SDORequestException {
//        this.motor.setMaxSpeedInCurrentMode(value);
//        writeMotorData("4", "4", value );
        this.writeParameter(Parameter.MaxSpeedInCurrentMode, value);
    }
    
    /**
     * This method writes the parameters for the motor on the actuator.
     * The parameters for the motor data are stored in the index 6410.
     * @param subindex
     * @param sizeInHexa size of the parameter in Can Open
     * @param value FORMAT=int the value of the parameter in decimal format
     */
    @Deprecated //the motor parameters are written with the parameters for Current
    protected void writeMotorData(String subindex, String sizeInHexa, int value) throws SDORequestException {
        String valueInHexa = Integer.toHexString(value);
        writeSDO("6410",subindex,sizeInHexa,valueInHexa);
    }
    
    /**
     * This methods reads the parameters of the motor stored in the actuator.
     * (index 6410)
     * @param subindex
     * @return value FORMAT=int the value of the parameter in decimal format
     */
    @Deprecated
    public int readMotorData(String subindex) throws SDORequestException {
        //TODO this should return a short or a long as it deviceErrorFile coded with 2 or 4 bytes.
        String valueInHexa = readSDO("6410",subindex);
        return Integer.parseInt(valueInHexa, 16);
    }
    
    /**
     * This methods read the parameters of the motor stored in the actuator 
     * (hardware configuration)and compare with the configuration stored in the
     * Configuration Data Base (software configuration).
     * @return true if the hardware and software configuration are the same,
     *         false otherwise. 
     * @throws org.lsst.ccs.subsystems.fcs.errors.EPOSConfigurationException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     */
    @Override
    public boolean checkMotorParameters() throws EPOSConfigurationException, SDORequestException {
        boolean ok = true;
              
        //motor type
        int readMotorType = readParameter(Parameter.MotorType);
        if (!(readMotorType == this.getMotorParameterValue("MotorType"))) ok = false;
        
        //motor type
        int readPositionSensorType = readParameter(Parameter.PositionSensorType);
        if (!(readPositionSensorType == this.getMotorParameterValue("PositionSensorType"))) ok = false;
        
        //motor data
//        int readContinuousCurrentLimit = readMotorData("01");
        int readContinuousCurrentLimit = readParameter(Parameter.ContinuousCurrentLimit);
        if (!(readContinuousCurrentLimit == this.getMotorParameterValue("ContinuousCurrentLimit"))) ok = false;
        
//        int readMaxSpeedInCurrentMode = readMotorData("04");
        int readMaxSpeedInCurrentMode = readParameter(Parameter.MaxSpeedInCurrentMode);
        if (!(readMaxSpeedInCurrentMode == this.getMotorParameterValue("MaxSpeedInCurrentMode"))) ok = false;
        
//        int readThermalTimeConstantWinding = readMotorData("05");
        int readThermalTimeConstantWinding = readParameter(Parameter.ThermalTimeConstantWinding);
        if (!(readThermalTimeConstantWinding == this.getMotorParameterValue("ThermalTimeConstantWinding"))) ok = false;
        
//        int readOutputCurrentLimit = readMotorData("02");
        int readOutputCurrentLimit = readParameter(Parameter.OutputCurrentLimit);
        if (!(readOutputCurrentLimit == this.getMotorParameterValue("OutputCurrentLimit"))) ok = false;
        
//        int readPolePairNumber = readMotorData("03");
        int readPolePairNumber = readParameter(Parameter.PolePairNumber);
        if (!(readPolePairNumber == this.getMotorParameterValue("PolePairNumber"))) ok = false;
        
        if (!ok) {
            throw new EPOSConfigurationException("ERROR in hardware configuration of the motor", this.name,
                    readMotorType,readContinuousCurrentLimit,readMaxSpeedInCurrentMode,
                    readThermalTimeConstantWinding,readOutputCurrentLimit,readPolePairNumber);           
        }
        return ok;
    }
    
    /**
     * Read the values of the motor parameters and displays them for the end user.
     * @return
     * @throws SDORequestException 
     */
    public String displayMotorParameters() throws SDORequestException {
        StringBuilder sb = new StringBuilder("Read decimal values for motor parameters are :");
              
        //motor type
        int readMotorType = readParameter(Parameter.MotorType);
        
        //position sensor type
        int readPositionSensorType = readParameter(Parameter.PositionSensorType);
        
        //motor data
        int readContinuousCurrentLimit = readParameter(Parameter.ContinuousCurrentLimit);     
        int readMaxSpeedInCurrentMode = readParameter(Parameter.MaxSpeedInCurrentMode);
        int readThermalTimeConstantWinding = readParameter(Parameter.ThermalTimeConstantWinding);      
        int readOutputCurrentLimit = readParameter(Parameter.OutputCurrentLimit);
        int readPolePairNumber = readParameter(Parameter.PolePairNumber);

        sb.append(" motor type ="); sb.append(readMotorType);
        sb.append(" position sensor type ="); sb.append(readPositionSensorType);
        sb.append(" continuousCurrentLimit ="); sb.append(readContinuousCurrentLimit);
        sb.append(" maxSpeedInCurrentMode ="); sb.append(readMaxSpeedInCurrentMode);
        sb.append(" thermalTimeConstantWinding ="); sb.append(readThermalTimeConstantWinding);
        sb.append(" outputCurrentLimit ="); sb.append(readOutputCurrentLimit);
        sb.append(" polePairNumber ="); sb.append(readPolePairNumber);
        return sb.toString();
    }
    
    /*Methods available in CURRENT mode*/ 
    /**
     * In current mode this methods send a current to the motor.
     * This make run the motor.
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws BadCommandException 
     * 
     */
    @Override
     @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Set the current value in the EPOS CPU.")
    public void writeCurrent(int aValue) throws BadCommandException, SDORequestException {
        if (!isEnable()) 
            throw new BadCommandException(getName() + "is not ENABLE");
        if (!mode.equals(CanOpenEPOS.EposMode.CURRENT)) 
            throw new BadCommandException(getName() + "is not in CURRENT mode");       
        try {
            //        if (aValue > motor.getContinuousCurrentLimit())
            if (aValue > this.getMotorParameterValue("ContinuousCurrentLimit"))
                throw new IllegalArgumentException(aValue + " is greater than Continuous Current Limit");
        } catch (EPOSConfigurationException ex) {
            log.error(getName() + " ERROR in command writeCurrent:" + ex.toString());
            throw new BadCommandException(getName() + ": ContinuousCurrentLimit is not defined");
        }
        
//        motor.setCurrentValue(aValue);
        this.currentValue = aValue;
        String currentValueInHexa = Integer.toHexString(aValue);
        writeSDO("2030","0","2",currentValueInHexa);
        //TODO check the return code of writeSDO
    }
    
    /**
     * In Current Mode this methods returns the current actualy received by the motor.
     * @return the current UNIT=mA / FORMAT=decimal 
     * @throws BadCommandException 
     */
    public int readCurrent() throws BadCommandException, SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
            String currentInHexa = readSDO("6078","0");
            return Integer.parseInt(currentInHexa, 16);
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
    }
    
    /**
     * In current mode this methods set to zero the value of the current sent to
     * the motor.
     * This stops the motor.
     * @throws BadCommandException 
     */
    public void stopCurrent() throws BadCommandException, SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
            writeSDO("2030","00","2","0");
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
        
    }
    
//        /*Methods available in VELOCITY mode*/ 
//    @Deprecated
//    public void setMaxiVelocity(int velocity) throws SDORequestException {
//        if (velocity > this.getMotorParameterValue(Parameter.MaxProfileVelocity)) 
//            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
//        motor.maxProfileVelocity = velocity;
//        String velocityInHexa = Integer.toHexString(velocity);
//        writeSDO("607F","00","2",velocityInHexa);
//    }
//    
//    @Deprecated
//    public void setMaxiAcceleration(int acceleration) {
//        if (acceleration > this.motor.maxAcceleration ) 
//            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
//        motor.maxAcceleration = acceleration;
//        //writeMaxonMotor(0x60C5,00,acceleration);
//    }
//    
    public void setVelocity() throws BadCommandException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            //writeMaxonMotor(0x206B,00,this.velocityValue);
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");

    }
    
    public int readVelocity()throws BadCommandException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            //String velocityInHexa = readSDO(0x2028,00);
            return 0;
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");
        
    } 
    
    /**
     * Stop the motion in the mode VELOCITY.
     * Not used in single-filter-test, neither in testbenchLPSC.
     * @throws BadCommandException 
     */
    public void stopVelocity() throws BadCommandException, SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            //writeMaxonMotor(0x206B,00,0);
            writeSDO("206B","00","2","0");
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");
        
    }
    
    /**
     * To make end user's life more easy : reads in the CPU the value of the parameter PositionActualValue 
     * and returns it in a decimal format.
     * @return value in decimal format 
     * @throws SDORequestException 
     */
    public int readPositionActualValue() throws SDORequestException {
        return this.readParameter(Parameter.PositionActualValue);
    }
    
    /**
     * Writes value 35 in hexa to set the Homing method as Actual (See EPOS documentation)
     * @return
     * @throws SDORequestException
     * @throws BadCommandException if the EPOS is not enable.
     */
    public String setHomingMethodActual() throws SDORequestException, BadCommandException {
        if (!mode.equals(CanOpenEPOS.EposMode.HOMING))
            throw new BadCommandException(getName() + "is not in HOMING mode");
//        if (!isEnable()) 
//            this.enable();
 
        return this.writeParameter(Parameter.HomingMethod, 35);
       
    }
    
    /**
     * Starts homing : (See EPOS documentation)
     * For engineering mode.
     * @throws SDORequestException
     * @throws BadCommandException if the EPOS 
     */
    public void startHoming() throws SDORequestException, BadCommandException, ErrorInCommandExecutionException {
        if (!mode.equals(CanOpenEPOS.EposMode.HOMING))
            throw new BadCommandException(getName() + "is not in HOMING mode");
//        if (!isEnable()) 
//            throw new BadCommandException(getName() + "is not ENABLE");
//        this.writeControlWord("1F");
        this.on();

    }
    
    /**
     * Set the Home Position with the value given as argument in decimal format.
     * @param position in decimal format
     * @return
     * @throws SDORequestException
     * @throws BadCommandException 
     */
    public String setHomePosition(int position) throws SDORequestException, BadCommandException {
        if (!mode.equals(CanOpenEPOS.EposMode.HOMING))
            throw new BadCommandException(getName() + "is not in HOMING mode");

        return this.writeParameter(Parameter.HomePosition, position);

    }
    
    /**
     * Defines the actual position as the absolute position which value is given as an argument.
     * @throws SDORequestException
     * @throws ErrorInCommandExecutionException
     * @throws BadCommandException 
     */
    @Override
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING3, description="Define the actual position as position given as argument.")
    public void defineAbsolutePosition(int position) throws SDORequestException, ErrorInCommandExecutionException, BadCommandException {
            log.debug(name, "Defining Absolute Position:" + position);
            this.changeMode(CanOpenEPOS.EposMode.HOMING);
            this.writeParameters(CanOpenEPOS.EposMode.HOMING);
            this.setHomePosition(position);
            this.setHomingMethodActual();
            this.on();
    }
     
    /**
     * In PROFILE_POSITION mode this methods set the target position.
     * This make run the motor.
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws BadCommandException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException 
     * 
     */
    @Override
    public void writeTargetPosition(int aValue) throws BadCommandException, SDORequestException {
        if (!mode.equals(CanOpenEPOS.EposMode.PROFILE_POSITION)) 
            throw new BadCommandException(getName() + "is not in PROFILE_POSITION mode");
        if (!isEnable()) 
            throw new BadCommandException(getName() + "is not ENABLE");
               
        String targetPositionInHexa = Integer.toHexString(aValue);
        this.targetPosition = aValue;
        writeSDO("607A","0","4",targetPositionInHexa);
        //TODO check the return code of writeSDO
    }
    
    @Override
    public int readPosition() throws SDORequestException {
        return this.readPositionActualValue();
    }
    
    public void activateCurrentMode() throws SDORequestException {
        this.changeMode(EposMode.CURRENT); 
        this.writeParameters(this.paramsForCurrent);
    }
    
    public void activateHomingMode() throws SDORequestException {
        this.changeMode(EposMode.HOMING); 
        this.writeParameters(this.paramsForHoming);
    }
    
    public void activateProfilePositionMode() throws SDORequestException {
        this.changeMode(EposMode.PROFILE_POSITION); 
        this.writeParameters(this.paramsForProfilePosition);
    }
    

    
        /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        System.out.println(displayListParameters("HOMING")); 
        System.out.println(displayListParameters("CURRENT"));
        System.out.println(displayListParameters("PROFILE_POSITION"));
        System.out.println(displayListParameters());
        


    }
    
    
    
}
