/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.Map;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.DeploymentException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;

/**
 * This class gathers the description of the real hardware and the tools (methods)
 * to give access to this hardware.
 * The concrete classes which extends this abstract class will have to add the
 * fields for the pieces of hardware and to give the list of hardware in the method listHardware.
 * @author virieux
 */
public abstract class BridgeToCanOpenHardware extends Module implements BridgeToHardware {
    
    protected CanOpenProxy tcpProxy;

    protected boolean configurationChecked;
    protected boolean hardwareInitialized;
    
    protected Map<String,PieceOfHardware> hardwareMap;


    public BridgeToCanOpenHardware(String aName, int aTickMillis, CanOpenProxy tcpProxy) {
        super(aName, aTickMillis);
        this.configurationChecked = false;
        this.hardwareInitialized = false;
        this.tcpProxy = tcpProxy;
    }

    
    /**
     * This method returns the list of hardware that this bridge will manage.
     * @return a list of PieceOfHardware
     * exemple :
     * public PieceOfHardware listHardware() {
     *      PieceOfHardware[] hardwareList = {clampActuatorXminus, clampActuatorXplus,  ai814, ao412, dio8n1, dio8n2};
     *      return hardwareList;
     * }
     * 
     */
    public abstract PieceOfHardware[] listHardware();

    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="print my children; for tests")
    public void printChildren() {
        Map<String,PieceOfHardware> children = this.getChildren(PieceOfHardware.class);
        for(Map.Entry<String,PieceOfHardware> entry : children.entrySet()) {
            String key = entry.getKey();
            log.debug("key=" + key);
            PieceOfHardware device = entry.getValue();
            log.debug("hardware=" + device.getName() + "#" + device.toString());
            
        }
    }

    @Override
    public CanOpenProxy getTcpProxy() {
        return tcpProxy;
    }
    
    @Override
    public void startServer() {
        tcpProxy.startServer();
    }
    
    @Override
    public void startThreadReader() {
        tcpProxy.startThreadReader();
    }
    
    
    
     /**
     * This method returns true if the C-Wrapper is connected to the tcp proxy.
     * The C-Wrapper has to send the tcpProxy clientName as a keyword before
     * any communication can begin. So if the C-Wrapper doesn't send the good
     * keyword or if it doesn't connect to the tcp proxy, this method returns 
     * false.
     * @return 
     */
    @Override
    public boolean isCWrapperConnected() {
        return tcpProxy.isReady(tcpProxy.getMyClientName());
    }
    
    
    /**
     * returns true if the hardware is booted, identified and initialized.
     * @return 
     */
    @Override
    public boolean isHardwareReady() {
        return tcpProxy.isHardwareReady();
    }
    
    @Override
    public boolean isRealHardware() {
        return true;
    }
    
    
    
    @Override
    public void initModule() {
        
        log.info("INIT MODULE BRIDGE TO REAL HARDWARE: " + name); 
        hardwareMap = this.getChildren(PieceOfHardware.class);
        PieceOfHardware[] hardwareList = listHardware();
        tcpProxy.hardwareList = hardwareList;
        int numberOfNodes = hardwareList.length;
        tcpProxy.setExpectedNodesNB(numberOfNodes);
        log.info("Number of can open devices expected =" + numberOfNodes);
        log.info(this.toString());
        
        
        //we have to read the errors table for the maxon motor and can open devices.
        try {
            CanOpenErrorsTable.loadDeviceErrorTable();
            CanOpenErrorsTable.loadErrorRegisterTable();
            CanOpenErrorsTable.loadCommunicationErrorTable();
        } catch (DeploymentException ex) {
            log.error(ex.getMessage() + ": Could not load CanOpen Error Tables");
            //this.getSubsystem().updateState(State.InError, "Could not load CanOpen Error Tables");
            this.getSubsystem().raiseAlarm("Could not load CanOpen Error Tables" + ex.toString());
        }
    }
    
    public void checkMatos() throws HardwareNotDetectedException, HardwareConfigurationException {
        int bootedHardwareNb= hardwareMap.entrySet().size();
        log.info(name, "number of pieces of hardware expected=" + bootedHardwareNb);
        Map<String,CanOpenNode> bootedNodesMap = this.tcpProxy.bootedNodesMap;
        log.info(name, "number of pieces of hardware booted=" + bootedNodesMap.entrySet().size());
        
        for(Map.Entry<String,PieceOfHardware> entry : hardwareMap.entrySet()) {
            String key = entry.getKey();
            PieceOfHardware device = entry.getValue();
            log.debug("Checking hardware=" + device.getName() + "#" + device.toString());
            String serial = device.getSerialNB();
            String nodeID = device.getNodeID();
            if (bootedNodesMap.containsKey(nodeID)) {
                CanOpenNode bootedNode = bootedNodesMap.get(nodeID);
                if (bootedNode.getSerialNB().equalsIgnoreCase(serial)) {
                    log.info("booted hardware=" + device.getName() + ":" + device.toString());
                } else {
                    log.error("hardware with nodeID" + nodeID + "is booted but with an unexpected serial number." );
                    log.error("hardware in error:" + device.toString());
                    log.error("serial number found for this nodeID:" + bootedNode.getSerialNB());
                    throw new HardwareConfigurationException("Unexpected serial number", device.getName());
                }
            } else throw new HardwareNotDetectedException("Hardware is not booted:",device.getName(),device.getNodeID(),device.getSerialNB());            
                    
        }
//        StringBuilder sb = new StringBuilder(name);       
//        return sb.toString();
    }
    
    
     /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper. 
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * @throws java.lang.InterruptedException
     * @throws java.util.concurrent.TimeoutException
     * @throws org.lsst.ccs.bus.BadCommandException
     */
    public Object sendCanOpen(String command) throws InterruptedException, TimeoutException, BadCommandException {
        return tcpProxy.sendCanOpen(command);
    }

    @Override
    public PDOStorage readPDOs() throws HardwareException, BadCommandException {
        log.debug(name + "/readPDOS");
        return this.tcpProxy.readPDOs();
      
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/hardware list={");
        for (PieceOfHardware p : this.listHardware() ) {
            sb.append(" ");
            sb.append(p.toString());sb.append(";");
        }
        sb.append("}");
        return sb.toString();
    }
}
