/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.common;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalLevel;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;

/**
 * This represents a Mechanical Item which can be moved by a motor.
 * Generaly we want to execute an action on this mechanical item :
 * - start the action,
 * - read sensors to know if the action is completed, 
 * - wait for the end of the action. 
 * This module provides a general way to start an action and wait until the action is completed.
 * It factorises some code that otherwise would have to be duplicated in each Module that needs such fonctionnalities.
 * @author virieux
 */
public abstract class MobileItemModule extends Module {

    protected volatile boolean itemMoving = false;
    protected volatile MobileItemAction currentAction;
    //Used because we have to wait for the update from the sensors to know if the action is completed
    protected final Lock lock = new ReentrantLock();
    protected final Condition motionCompleted = lock.newCondition();

    ScheduledFuture<?> readSensorsHandle;
    final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);
    //prendre un Scheduler qui peut me permettre de donner un nom au Thread.
    
    protected AtomicBoolean haltRequired;

    public MobileItemModule(String moduleName, int aTickMillis) {
        super(moduleName, aTickMillis);
        haltRequired = new AtomicBoolean();
        haltRequired.set(false);
    }

    public boolean isItemMoving() {
        return itemMoving;
    }
    
    

    /**
     * This stops the reading of the sensors.
     */
    private void cancelReadingSensors() {
        lock.lock();
        try {
            log.debug(getName() + " => Stop reading sensors");
            motionCompleted.signal();
        } finally {
            lock.unlock();
        }
        this.readSensorsHandle.cancel(true);
        log.debug(getName() + " => readingSensors canceled");
    }

    /*
     * This method must returns true if the action is completed, false otherwise.
     * It's abstract because it'up to the sub-classes to know when the action is completed.
     */
    public abstract boolean isActionCompleted(MobileItemAction action);
    
    public abstract void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception;
    
    public abstract void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException;
    
    public abstract void stopAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, HardwareException;
    
    public abstract void postAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException;
    
    /**
     * This executes an action which moves the MobileItem and waits for the end od this action.
     * The MobileItemAction is an ENUM from the fcs.bus package.
     * @param action, timeout
     * @param timeoutForAction
     * @return 
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     * @throws org.lsst.ccs.subsystems.fcs.errors.HardwareException 
     */
    protected String executeAction(MobileItemAction action, long timeoutForAction) throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        
        lock.lock();
               
        try {
            this.itemMoving = true;
            
            this.currentAction = action;
            
            log.debug(name, "STARTING ACTION:" + action.toString());
            
            startAction(action);
            
            readSensorsUntilActionIsCompleted(action, System.currentTimeMillis(), timeoutForAction);
            
            waitForEndOfAction(action);
            
            if (haltRequired.get()) {
                stopAction(action,0);
                return name + "=> Received an ABORT command for action :" + action.toString();
            
            } else if (isActionCompleted(action)) {              
                postAction(action);
                return name + " " + action.doneString();
                
            } else {
                throw new ErrorInCommandExecutionException(name + "=>" +  action.getFailureMsg());
            }
            
        } finally {

            stopAction(action,0);
            this.itemMoving = false;
            this.motionCompleted.signal(); 
            haltRequired.set(false);
            lock.unlock();
        }
    }
    
    /**
     * Start reading the sensors at a fixed rate (scheduler.scheduleAtFixedRate) until the action is completed or the timeout for this action is past.
     * @param action
     * @param beginTime
     * 
     * @param timeout 
     */
    public void readSensorsUntilActionIsCompleted(final MobileItemAction action, final long beginTime, final long timeout) {
        final Runnable readSensors;
        readSensors = new Runnable() {
            
            @Override
            public void run() {

                try {
                    log.info(name + " : Reading sensors");
                    long duration = System.currentTimeMillis() - beginTime;
                    
                    updateStateWithSensorsToCheckIfActionIsCompleted();
                    boolean actionCompleted = isActionCompleted(action);

                    if (haltRequired.get()) {
                        itemMoving = false;
                        log.info(action.toString() + " ACTION ABORTED");
                        stopAction(action,0);
                        cancelReadingSensors();
                                                
                    } else if (actionCompleted) {
                        itemMoving = false;
                        log.info(action.toString() + " ACTION COMPLETED");
                        cancelReadingSensors();
                        
                    } else if (duration >= timeout) {
                        itemMoving = false;
                        log.info(action.toString() + " ACTION NOT COMPLETED during allocated time");
                        cancelReadingSensors();
                        throw new HardwareException(action.toString() + " exceeded timeout for this task: duration=" + duration + ",timeout=" + timeout);
                        
                        
                        
                    } else {
                        log.info(action.toString() + " not completed.....");
                        log.info(action.name() + "/duration=" + duration);
                        
                        
                    } 
                } catch (Exception ex) {
                    log.error("ERROR during action:" + ex.getMessage());
                    getSubsystem().raiseAlarm(ex.toString());
                    cancelReadingSensors();
                }
            }
        };
        this.readSensorsHandle = scheduler.scheduleAtFixedRate(readSensors, 500, 500, TimeUnit.MILLISECONDS);
    }

    /**
     * This method waits until the action is completed.
     * This methods is called by executeAction and has already acquired the lock.
     */
    private void waitForEndOfAction(MobileItemAction action) {
            while (itemMoving) {
                try {
                    log.info(name +  " Waiting for end of " + action.toString());
                    motionCompleted.await();
                } catch (InterruptedException ex) {
                    log.info(name + " " + ex.toString());
                }
            }
            log.info(name + " STOP WAITING FOR END OF ACTION");
    }
    
    private void halt(MobileItemAction action, long delay) {
        try {
            log.debug(name + " STOPPING ACTION: " + action.toString());
            stopAction(action,delay);
        } catch (BadCommandException | ErrorInCommandExecutionException | HardwareException ex) {
            log.error(name + " couldn't stop action " + action.toString());
        }        
    }
    
    /**
     * just to avoid taping "sendSignal HALT"
     * @param delay 
     * @throws org.lsst.ccs.bus.BadCommandException 
     */
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, description="Abort the running action.")
    public void abort(long delay) throws BadCommandException {
        if (this.itemMoving) {
            sendSignal(delay,"HALT");
        } else throw new BadCommandException(getName() + " is not moving; nothing to stop.");
    }
    
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, description="Abort the running action.")
    public void abort() throws BadCommandException {
        abort(0);
    }
    
    @Command(type=Command.CommandType.ABORT, level=Command.ENGINEERING1, description="Abort the running action.")
    public void stop() throws BadCommandException {
        abort(0);
    }
    
    /**
     * This shutdowns the scheduler.
     */
//    @Override
//    public void shutdownNow() {
//        super.shutdownNow();
//        scheduler.shutdown();
//    }
    
    @Override
    public TreeWalkerDiag signal(Signal signal) {
        SignalLevel sl = signal.getLevel();
        log.info(sl.toString());
        if (this.itemMoving) {
            switch(signal.getLevel()) {
                case HALT :
                    log.info(getName() + " HALT required");
                    this.haltRequired.set(true);
                    halt(currentAction,signal.getExpectedMaxDelay());
                    break;
                case STOP :
                    log.info(getName() + " STOP required");
                    this.haltRequired.set(true);
                    halt(currentAction,signal.getExpectedMaxDelay());
                    break;
                case INTERRUPT1 :
                    break;
                case INTERRUPT2 :
                    break;
                case RE_START :
                    this.haltRequired.set(false);
                    break;
                    
            }
        } else log.warning(getName() + " is not moving; nothing to stop.");
        return TreeWalkerDiag.HANDLING_CHILDREN;
    }
    
 
    





    
}
