/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutoChangerTrucksLocation;

/**
 * A truck is a mobile part in the autochanger. It's the mechanical item which holds
 * a filter through a latch and moves the filter from STANDBY position to STANDBACK and back.
 * There is 2 trucks in the autochanger, one on each side of the filter. A truck move along a rail.
 * The position of a truck is given by 2 sensors (hall effect sensor) which are located on the rails.
 * One is at STANDBY position, the other is at STANDBACK position.
 * The trucks are moved by a motor. 
 * They are moved together by a motor which synchronizes the motion of both trucks.
 * 
 * @author virieux
 */
public class TruckModule extends Module {
    
    private AutoChangerTrucksLocation truckLocation;
    
    /**
     * A effect Hall sensor which says if the truck is at standback position or not.
     */
    protected NumericSensor railSensorStandback;


    /**
     * A effect Hall sensor which says if the truck is at standby position or not.
     */
    protected NumericSensor railSensorStandby;
    
    /***************************************************************************************************/
    /************Tools for the synchronization of the truck position update reading the 2 sensors ******/
    /***************************************************************************************************/
         
    private volatile boolean updatingTruckLocation = false;
    //When we update the truck position reading the 2 sensors, 
    //we want to take a lock on the truck until
    //the end of the update, so any other thread has to wait for this update.
    final Lock lock = new ReentrantLock();
    final Condition truckLocationUpdated = lock.newCondition();

    public TruckModule(String aName, int aTickMillis, NumericSensor railSensorStandback, NumericSensor railSensorStandby) {
        super(aName, aTickMillis);
        this.railSensorStandback = railSensorStandback;
        this.railSensorStandby = railSensorStandby;
    }

    
    /******************************************************************************************************/
    /*********END of Tools for the synchronization of the truck position update reading the 2 sensors *****/
    /******************************************************************************************************/
    
    
    
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    //FOR SPRING

    public NumericSensor getRailSensorStandback() {
        return railSensorStandback;
    }

    public void setRailSensorStandback(NumericSensor railSensorStandback) {
        this.railSensorStandback = railSensorStandback;
    }

    public NumericSensor getRailSensorStandby() {
        return railSensorStandby;
    }

    public void setRailSensorStandby(NumericSensor railSensorStandby) {
        this.railSensorStandby = railSensorStandby;
    }

    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/
    
    @Override
    public void initModule() {
        this.truckLocation = AutoChangerTrucksLocation.UNKNOWN;
    }
    
    /**
     * This method returns the truck position (an ENUM).
     * If the truck position is being updated it waits until the end of the update.
     * @return truckLocation
     */
    public AutoChangerTrucksLocation getTruckLocation() {
        
        lock.lock();
        try {
            while(updatingTruckLocation) {
                try {
                    this.truckLocationUpdated.await();
                } catch (InterruptedException ex) {
                    log.error(ex);
                }

            }
            return truckLocation;
            
        } finally {
            lock.unlock();
        }
        
    }
    

    
    /**
     * This methods updates the field truckLocation in parsing the values of the 8 channels of the DIO device.
     * @param hexaValueOfInputs
     * @throws ErrorInCommandExecutionException 
     */
    public void updateLocation(String hexaValueOfInputs) throws ErrorInCommandExecutionException {
        
        lock.lock();
        
        
        try {
            updatingTruckLocation = true;
            log.debug(getName() + " is updating its location in parsing the values sent by rail sensors:" + hexaValueOfInputs);
        
            this.railSensorStandback.updateValue(hexaValueOfInputs);
            this.railSensorStandby.updateValue(hexaValueOfInputs);

            int standbackValue = railSensorStandback.getDigitalValue();
            int standbyValue = railSensorStandby.getDigitalValue();


            //log.debug("standbyValue=" + standbyValue);
            //log.debug("standbackValue=" + standbackValue);

            if (standbyValue == 1 && standbackValue == 0) {
                truckLocation = AutoChangerTrucksLocation.STANDBY;

            } else if (standbyValue == 0 && standbackValue == 1) {
                truckLocation = AutoChangerTrucksLocation.STANDBACK;
            } else if (standbyValue == 0 && standbackValue == 0) {
                truckLocation = AutoChangerTrucksLocation.UNKNOWN;
            } else if (standbyValue == 1 && standbackValue == 1) {
                truckLocation = AutoChangerTrucksLocation.ERROR;
                throw new ErrorInCommandExecutionException(getName() + 
                        " ERROR IN READING RAIL SENSORS : a truck can't be at standby AND at standback ");
            }
            //publish on the status bus ?
            log.debug("truckLocation=" + truckLocation.toString());
        
        } finally {
            
            updatingTruckLocation = false;
            truckLocationUpdated.signal();
            lock.unlock();
        }
    }
    
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/rail sensor at STANDBACK=");
            if (this.railSensorStandback == null) {
                sb.append("NULL");
            } else {
                sb.append(this.railSensorStandback.getName());
            }
        sb.append("/rail sensor at STANDBY=");
            if (this.railSensorStandby == null) {
                sb.append("NULL");
            } else {
                sb.append(this.railSensorStandby.getName());
            }
        return sb.toString();
    }
}
