/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.Map;
import java.util.concurrent.locks.Condition;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.CurrentCommandContext;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.bus.KVList;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenEPOS;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is model for the clamps mechanism in the loader.
 * 4 hooks are used for the clamping of the filter in the loader.
 * This hooks are moved all together by the hooksController.
 * 
 * @author virieux
 */
public class LoaderClampModule extends MobileItemModule {
    
    private EPOSController hooksController;
    final LoaderHookModule hook1;
    final LoaderHookModule hook2;
    final LoaderHookModule hook3;
    final LoaderHookModule hook4;
    private  PlutoGatewayModule plutoGateway;
    private final  String plutoGatewayName;
    
    
    FcsEnumerations.LockStatus lockStatus;
   
    
    long timeoutForClosingHooks;
    long timeoutForClampingHooks;
    long timeoutForOpeningHooks;
    long timeoutForGoingToHomePosition;
    
    private volatile boolean initialized;
    
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    int position;
    int minPosition;
    int maxPosition;
    int targetPositionToOpen;
    int targetPositionToClose;
    int targetPositionToClamp;
    int currentToClamp;
    int currentToGoHome;
    Map<String,Integer> paramsForCurrentToGoHome;
    Map<String,Integer> paramsForCurrentToClose;

    public LoaderClampModule(String moduleName, int aTickMillis, 
            LoaderHookModule hook1, LoaderHookModule hook2, LoaderHookModule hook3, LoaderHookModule hook4,
            String plutoGWName,
            long timeoutForClosingHooks, 
            long timeoutForClosingHooksStrongly, 
            long timeoutForOpeningHooks,
            long timeoutForGoingToHomePosition,
            int targetPositionToOpen,
            int targetPositionToClose, 
            int targetPositionToCloseStrongly, 
            int currentToClose,
            int currentToGoHome, 
            Map<String, Integer> paramsForCurrentToClose,
            Map<String, Integer> paramsForCurrentToGoHomePosition,
            int minPosition,
            int maxPosition) {
        super(moduleName, aTickMillis);
        this.hook1 = hook1;
        this.hook2 = hook2;
        this.hook3 = hook3;
        this.hook4 = hook4;
        this.plutoGatewayName = plutoGWName;
        this.initialized = false;
        this.timeoutForClosingHooks = timeoutForClosingHooks;
        this.timeoutForClampingHooks = timeoutForClosingHooksStrongly;
        this.timeoutForOpeningHooks = timeoutForOpeningHooks;
        this.timeoutForGoingToHomePosition = timeoutForGoingToHomePosition;
        this.targetPositionToOpen = targetPositionToOpen;
        this.targetPositionToClose = targetPositionToClose;
        this.targetPositionToClamp = targetPositionToCloseStrongly;
        this.currentToClamp = currentToClose;
        this.currentToGoHome = currentToGoHome;
        this.paramsForCurrentToClose = paramsForCurrentToClose;
        this.paramsForCurrentToGoHome = paramsForCurrentToGoHomePosition;
        this.minPosition = minPosition;
        this.maxPosition = maxPosition;

    }

    public LoaderHookModule getHook1() {
        return hook1;
    }

    public LoaderHookModule getHook2() {
        return hook2;
    }

    public LoaderHookModule getHook3() {
        return hook3;
    }

    public LoaderHookModule getHook4() {
        return hook4;
    }

    public int getPosition() {
        return position;
    }

    //for the GUI
    public int getMinPosition() {
        return minPosition;
    }
    
    //for the GUI
    public int getMaxPosition() {
        return maxPosition;
    }
    
    
    

    //for the simulator
    public int getTargetPositionToOpen() {
        return targetPositionToOpen;
    }
    //for the simulator
    public int getTargetPositionToClose() {
        return targetPositionToClose;
    }
    //for the simulator
    public int getTargetPositionToClamp() {
        return targetPositionToClamp;
    }
    //for the simulator
    public int getCurrentToClose() {
        return currentToClamp;
    }
    //for the simulator
    public int getCurrentToGoHome() {
        return currentToGoHome;
    }

    public long getTimeoutForClosingHooks() {
        return timeoutForClosingHooks;
    }

    public long getTimeoutForClampingHooks() {
        return timeoutForClampingHooks;
    }

    public long getTimeoutForOpeningHooks() {
        return timeoutForOpeningHooks;
    }

    public long getTimeoutForGoingToHomePosition() {
        return timeoutForGoingToHomePosition;
    }
    
    
    
    

    public FcsEnumerations.LockStatus getLockStatus() {       
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return lockStatus;
            
        } finally {
            lock.unlock();
        }        
    }
    
     @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if LockStatus=LOCKED")
    public boolean isLocked() {
        return this.getLockStatus().equals(LockStatus.LOCKED);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if LockStatus=UNLOCKED")
    public boolean isUnlocked() {
        return this.getLockStatus().equals(LockStatus.UNLOCKED);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if loader is empty.")
    public boolean isEmpty() {
        return ((LoaderModule) this.getModule("loader")).isEmpty();
    }
    
   
    @Override
    public void initModule() {
        this.hooksController = (EPOSController) this.getModule("hooksController");
        this.plutoGateway = (PlutoGatewayModule) this.getModule(plutoGatewayName);
        this.lockStatus = LockStatus.UNKNOWN;
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hook is open and at home position")
    public boolean isAtHomePosition() throws SDORequestException {
        return hooksController.isTargetPositionReached(0);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hook is closed and at clamped position")
    public boolean isAtClampedPosition() throws SDORequestException {
        return hooksController.isTargetPositionReached(this.targetPositionToClamp);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hook is at closed position.")
    public boolean isAtClosedPosition() throws SDORequestException {
        return hooksController.isTargetPositionReached(this.targetPositionToClose);
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Returns true if hook is at open position.")
    public boolean isAtOpenPosition() throws SDORequestException {
        return hooksController.isTargetPositionReached(this.targetPositionToOpen);
    }
    
    /**
     * If loader is empty, go to home position otherwise go to clamped position.
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws HardwareException 
     */
    public void initializeHardware() throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        this.updateStateWithSensors();
        if (this.isEmpty()) {
            this.goToHomePosition();
            this.hooksController.defineAbsolutePosition(0);
            this.initialized = true;           
        } else {
            this.goToClampedPosition();
            this.hooksController.defineAbsolutePosition(this.targetPositionToClamp);
            this.initialized = true;
        }
    }
    
    @Override
    public void tick() {
        if (!initialized) {
            //initializeHardware();
            initialized = true;
            log.debug(getName() + " is initialized");
        }
        updateStateWithSensors();
    }
    
    /**
     * In the initialisation phase, this close the clamp to be able to define the absolute position of the encoder.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws HardwareException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Go to clamped position to intialize hardware.")
    public String goToClampedPosition() throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        if (isEmpty()) throw new BadCommandException(getName() + " can't go to close position if  empty.");
        return this.executeAction(FcsEnumerations.MobileItemAction.GOTOCLOSEPOSITION, timeoutForClampingHooks);
    }
    
    /**
     * In the initialisation phase, this open the clamp to be able to define the absolute position of the encoder.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     * @throws HardwareException 
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Go to home position to intialize hardware.")
    public String goToHomePosition() throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        if (!isEmpty()) throw new BadCommandException(getName() + " can't go to home position if not empty.");
        return this.executeAction(FcsEnumerations.MobileItemAction.GOTOHOMEPOSITION, timeoutForGoingToHomePosition);
    }
    

    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Close clamp.")
    public String close() throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        if (!initialized) throw new BadCommandException(getName() + " is not intialized. PLease initialize hardware first.");
        return this.executeAction(FcsEnumerations.MobileItemAction.CLOSELOADERHOOKS, timeoutForClosingHooks);
    }
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Clamp clamp.")
    public String clamp() throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        if (!initialized) throw new BadCommandException(getName() + " is not intialized. PLease initialize hardware first.");
        if (!isLocked()) throw new BadCommandException(getName() + " has to be LOCKED first. close hooks prior closeStrongly");
        return this.executeAction(FcsEnumerations.MobileItemAction.CLAMPLOADERHOOKS, timeoutForClampingHooks);
    }
    
    
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Open clamp.")
    public String open() throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        if (!initialized) throw new BadCommandException(getName() + " is not intialized. PLease initialize hardware first.");
        //TODO check if autochanger is holding the filter.
        return this.executeAction(FcsEnumerations.MobileItemAction.OPENLOADERHOOKS, timeoutForOpeningHooks);
    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        switch(action) {
           case GOTOCLOSEPOSITION :
               //return (this.lockStatus.equals(FcsEnumerations.LockStatus.CLAMPED));
               return (this.lockStatus.equals(FcsEnumerations.LockStatus.LOCKED));
               
           case GOTOHOMEPOSITION :
               return (this.lockStatus.equals(FcsEnumerations.LockStatus.UNLOCKED));
                           
           case CLOSELOADERHOOKS : 
               return (this.lockStatus.equals(FcsEnumerations.LockStatus.LOCKED));
               //TODO isAtClosedPosition where position <> .targetPositionToClose
               //return this.hooksController.isTargetPositionReached(this.targetPositionToClose);

           case CLAMPLOADERHOOKS : 
               //return (this.lockStatus.equals(FcsEnumerations.LockStatus.CLAMPED));
               return (this.lockStatus.equals(FcsEnumerations.LockStatus.LOCKED));
               
           case OPENLOADERHOOKS :
               return (this.lockStatus.equals(FcsEnumerations.LockStatus.UNLOCKED));
        }
        return false;
        
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        this.updateStateWithSensors();
        if ((currentAction.equals(MobileItemAction.CLAMPLOADERHOOKS)||
                (currentAction.equals(MobileItemAction.CLOSELOADERHOOKS))||
                (currentAction.equals(MobileItemAction.OPENLOADERHOOKS)))) {
            position = this.hooksController.readPosition();
        }
    }

    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        switch(action) {  
            case GOTOCLOSEPOSITION :
                hooksController.enable();
                hooksController.changeMode(CanOpenEPOS.EposMode.CURRENT);
                hooksController.writeParameters(paramsForCurrentToClose);
                hooksController.writeCurrent(this.currentToClamp);
                hooksController.on();
                break;
                
           case GOTOHOMEPOSITION :
               hooksController.enable();
                hooksController.changeMode(CanOpenEPOS.EposMode.CURRENT);
                hooksController.writeParameters(paramsForCurrentToGoHome);
                hooksController.writeCurrent(this.currentToGoHome);
                hooksController.on();
                break;
               
           case CLOSELOADERHOOKS :
               hooksController.enable();
                hooksController.changeMode(CanOpenEPOS.EposMode.PROFILE_POSITION);
                hooksController.writeParameters(CanOpenEPOS.EposMode.PROFILE_POSITION);
                hooksController.writeTargetPosition(this.targetPositionToClose);
                hooksController.on();
               break;
               
           case CLAMPLOADERHOOKS : 
               if (!(this.lockStatus.equals(LockStatus.LOCKED))) {
                   throw new BadCommandException(getName() + ": Hooks have to be LOCKED before being LOCKED STRONGLY");
               }
               hooksController.enable();
               hooksController.writeTargetPosition(this.targetPositionToClamp);
               hooksController.on();
               break;
               
           case OPENLOADERHOOKS :
               hooksController.enable();
               hooksController.changeMode(CanOpenEPOS.EposMode.PROFILE_POSITION);
               hooksController.writeParameters(CanOpenEPOS.EposMode.PROFILE_POSITION);
               hooksController.writeTargetPosition(this.targetPositionToOpen);
               hooksController.on();
               break;
        }
    }
    
    @Override
    public void stopAction(MobileItemAction action, long delay) throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        log.debug(getName() + " is STOPPING action " + action.toString() 
                + " within delay " + delay);
        CurrentCommandContext currentCmdContext = Subsystem.LOCAL_EXECUTION_INFO.get();
        log.debug("Command running:" + currentCmdContext.getCommandName() );
        log.info("coming from:" + currentCmdContext.getCommandOriginator());
        
        this.hooksController.off();
        
    }

    @Override
    public void postAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException {

        switch(action) {
            case GOTOCLOSEPOSITION :
                if (this.isLocked() && this.isAtClampedPosition()) {
                    this.hooksController.defineAbsolutePosition(this.targetPositionToClamp);
                    this.lockStatus = FcsEnumerations.LockStatus.CLAMPED;
                    this.publishData();
                }
                break;
                
            case GOTOHOMEPOSITION :
                this.hooksController.off();
                this.hooksController.defineAbsolutePosition(0);
                this.hooksController.off();
                
                break;
                
            case CLOSELOADERHOOKS : 
                this.hooksController.off();
                break;
                
            case CLAMPLOADERHOOKS :
                if (this.isLocked() && this.isAtClampedPosition()) {
                    this.lockStatus = FcsEnumerations.LockStatus.CLAMPED;
                    this.publishData();
                }
                break;
            case OPENLOADERHOOKS :
                break;
        }
    }
    

    /**
     * This methods updates the lockStatus in reading all the hooks sensors.
     */
    @Command(type=Command.CommandType.ACTION, level=Command.ENGINEERING1, description="Update clamp state in reading sensors.")
    public void updateStateWithSensors() {
        lock.lock();
         
        
        try {
                updatingState = true; 
                this.plutoGateway.updateValues();
                String[] readHexaValues = this.plutoGateway.getHexaValues();
//                log.debug("readHexaValues[0]=" + readHexaValues[0]);
//                log.debug("readHexaValues[1]=" + readHexaValues[1]);
                this.hook1.updateStateWithSensors(readHexaValues);
                this.hook2.updateStateWithSensors(readHexaValues);
                this.hook3.updateStateWithSensors(readHexaValues);
                this.hook4.updateStateWithSensors(readHexaValues);

                boolean inError = (this.hook1.getLockStatus() == LockStatus.ERROR) || 
                                        (this.hook2.getLockStatus() == LockStatus.ERROR) ||
                                        (this.hook3.getLockStatus() == LockStatus.ERROR) ||
                                        (this.hook4.getLockStatus() == LockStatus.ERROR);

                boolean locked = (this.hook1.getLockStatus() == LockStatus.LOCKED) && 
                                        (this.hook2.getLockStatus() == LockStatus.LOCKED) &&
                                        (this.hook3.getLockStatus() == LockStatus.LOCKED) &&
                                        (this.hook4.getLockStatus() == LockStatus.LOCKED);

                boolean unlocked = (this.hook1.getLockStatus() == LockStatus.UNLOCKED) && 
                                        (this.hook2.getLockStatus() == LockStatus.UNLOCKED) &&
                                        (this.hook3.getLockStatus() == LockStatus.UNLOCKED) &&
                                        (this.hook4.getLockStatus() == LockStatus.UNLOCKED);

                if (locked && unlocked) inError = true;
                if (inError) {
                    this.lockStatus = LockStatus.ERROR;
                } else if (locked) {                   
                    if (this.position == this.targetPositionToClose) {
                        this.lockStatus = LockStatus.CLAMPED;
                    } else {
                        this.lockStatus = LockStatus.LOCKED;
                    }
                } else if (unlocked) {
                    this.lockStatus = LockStatus.UNLOCKED;
                } else {
                    this.lockStatus = LockStatus.UNKNOWN;
                }
        
        } finally {

            updatingState = false;
            stateUpdated.signal(); 
            lock.unlock();
        } 
        this.publishData();
    }
    
    /**
     * Updates the position of the clamp in reading the CPU of the controller.
     * @throws org.lsst.ccs.bus.BadCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="Update clamp position in reading controller.")
    public void updatePosition() throws BadCommandException, SDORequestException {
        this.position = hooksController.readPosition();
        this.publishData();       
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp info.")
    public String listHooks() {
        StringBuilder sb = new StringBuilder("Hooks status:");sb.append("\n");
        sb.append(this.hook1.toString());sb.append("/");sb.append(this.hook1.getLockStatus());sb.append("\n");
        sb.append(this.hook2.toString());sb.append("/");sb.append(this.hook2.getLockStatus());sb.append("\n");
        sb.append(this.hook3.toString());sb.append("/");sb.append(this.hook3.getLockStatus());sb.append("\n");
        sb.append(this.hook4.toString());sb.append("/");sb.append(this.hook4.getLockStatus());sb.append("\n");
        return sb.toString();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp sensors values.")
    public String listSensorsValues() {
        StringBuilder sb = new StringBuilder(getName() + "Sensors values=");sb.append("/");
        sb.append(this.hook1.listSensorsValues());sb.append("/");
        sb.append(this.hook2.listSensorsValues());sb.append("/");
        sb.append(this.hook3.listSensorsValues());sb.append("/");
        sb.append(this.hook4.listSensorsValues());sb.append("/");
        return sb.toString();
    }
    
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp sensors values.")
    public String listStatus() {
        StringBuilder sb = new StringBuilder("Clamp status:");sb.append(this.lockStatus);sb.append("\n");
        sb.append(this.hook1.getName());sb.append("/");sb.append(this.hook1.getLockStatus());sb.append("\n");
        sb.append(this.hook2.getName());sb.append("/");sb.append(this.hook2.getLockStatus());sb.append("\n");
        sb.append(this.hook3.getName());sb.append("/");sb.append(this.hook3.getLockStatus());sb.append("\n");
        sb.append(this.hook4.getName());sb.append("/");sb.append(this.hook4.getLockStatus());sb.append("\n");
        return sb.toString();
    }
    

    
    @Override
    @Command(type=Command.CommandType.QUERY, level=Command.ENGINEERING1, description="List and display clamp info.")
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("/timeoutForClosingHooks=");sb.append(this.timeoutForClosingHooks);
        sb.append("/timeoutForClosingHooksStrongly=");sb.append(this.timeoutForClampingHooks);
        sb.append("/timeoutForOpeningHooks=");sb.append(this.timeoutForOpeningHooks);
        return sb.toString();
    }

     public StatusDataPublishedByLoaderClamp getStatusData() {
        StatusDataPublishedByLoaderClamp status = FcsUtils.createStatusDataPublishedByLoaderClamp(this);
        return status;
    }
     
     /**
     * Publish Data on status bus for trending data base and GUIs.
     */
     public void publishData() {
        StatusDataPublishedByLoaderClamp status = this.getStatusData();
        //this.sendToStatus(status); 
        this.publish("loaderClamp", status);
     }
     
     /**
     * Publish Data on status bus for trending data base and GUIs.
     * If a key is changed here, it has to be changed also in the GUIs (FilterExchangeMainPanel)
     */
    //perhaps better to use publishData
    @Deprecated //since the Loader GUI implements DataStatusListener
    public void publishClampData() {
        KVList kvlist = new KVList(7);
        long timestamp = System.currentTimeMillis();
        kvlist.add("dataType", "loaderClamp");
        kvlist.add(getName() + "/clampState", this.lockStatus.toString());
        kvlist.add("hook1Name",this.hook1.getName());
        kvlist.add("hook2Name",this.hook2.getName());
        kvlist.add("hook3Name",this.hook3.getName());
        kvlist.add("hook4Name",this.hook4.getName());
        
        LoaderHookModule[] listHooks = new LoaderHookModule[] {hook1,hook2,hook3,hook4};
        for (LoaderHookModule hook : listHooks ) {
            kvlist.add(hook.getName() + "/lockStatus", hook.lockStatus.toString());
            kvlist.add(hook.getName() + "/lockSensorValue", hook.getLockSensor().digitalValue);
            kvlist.add(hook.getName() + "/unlockSensorValue", hook.getUnlockSensor().digitalValue);
            kvlist.add(hook.getName() + "/inError", hook.isInError());
        }
        
        publish(timestamp,kvlist);
        
    }



   


    
}
