/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.singlefiltertest;

import java.util.Observable;
import org.lsst.ccs.bus.Alarm;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module.ValueUpdate;
import org.lsst.ccs.subsystems.fcs.AutoChangerModule;
import org.lsst.ccs.subsystems.fcs.CarouselModule;
import org.lsst.ccs.subsystems.fcs.FcsAlarm;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.Filter;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.singlefiltertest.config.SingleFilterTestConfig;

/**
 *
 * @author virieux
 */
public class SftMainModule extends MainModule {

    private Filter dummyFilter;
    private SftCarouselModule carousel;
    private SftAutoChangerModule autochanger;
    private boolean filterLocated;

    public SftMainModule(String aName, int aTickMillis, BridgeToHardware bridge, Filter dummyFilter) {
        super(aName, aTickMillis, bridge);
        this.dummyFilter = dummyFilter;
    }
    
    
    
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
     /**

     * @return the dummyFilter
     */
    public Filter getDummyFilter() {
        return dummyFilter;
    }
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    @Override
    public void initModule() {
        
        super.initModule();
        carousel = (SftCarouselModule) this.getModule("carousel");
        autochanger = (SftAutoChangerModule) this.getModule("autochanger");
        hardwareReady = false;
        filterLocated=false;
    }
    
    @Override
    public void tick() {
        if (this.isHardwareReady() && !filterLocated) {
            try {
                this.locateFilter();
                filterLocated = true;
                log.info("Filter has been located. Hardware is ready to receive commands.");
            } catch (Exception ex) {
                filterLocated = false;
                log.error(ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                //to execute in "normal mode"
                //this.getSubsystem().updateState(State.InError, "UNABLE TO LOCATE THE FILTER :" + ex.getMessage());
            }
        }
    }
    

    public void locateFilter() throws HardwareException, ErrorInCommandExecutionException, BadCommandException {
        //TODO that is OK for the simulator not for real hardware
//        carousel.getSockets()[0].putFilterOnSocket(dummyFilter);
//        carousel.engageClampsContact();
        log.debug("MODE DEBUG");
        carousel.updateClampsStateWithSensors();
        if (carousel.getClampXminus().getClampState().equals(FilterClampState.CLAMPEDONFILTER)
                || carousel.getClampXminus().getClampState().equals(FilterClampState.UNCLAMPEDONFILTER)) {
            carousel.getSockets()[0].putFilterOnSocket(dummyFilter);
        }
        autochanger.updateLatchesStateWithSensors();

 
        if (!autochanger.isTrucksEmpty()) {
            autochanger.setFilterOnTrucks(dummyFilter);
        }
        
        autochanger.updateTrucksLocationWithSensors();
        
    }
    
    public void checkPreConditionsForUnlockClamps() throws HardwareException, BadCommandException, ErrorInCommandExecutionException {
        this.updateStateWithSensors();
        if (carousel.isHoldingFilterAtStandby() && (!autochanger.isHoldingFilterAtStandby())) {
            throw new ErrorInCommandExecutionException("Can't unlock clamps if filter is not held by autochanger.");
        }
    }

    public String displayData() {
        StringBuilder sb = new StringBuilder("Data published by ");
        sb.append(getContext().getSubsystem().getName()).append("\n  ");
        sb.append(CarouselModule.publishedByCarouselOutputName);
        log.info("Carousel status=" + carousel.getStatusData().toString());
        sb.append(carousel.getStatusData().toString());sb.append("\n  ");
        sb.append("clampXminus status=");sb.append(carousel.getClampXminus().getStatusData().toString());
        sb.append("\n  ");
        sb.append("clampXplus status=");sb.append(carousel.getClampXplus().getStatusData().toString());
        sb.append("\n  ");
        sb.append(AutoChangerModule.publishedByAutoChangerOutputName);
        log.info("autochanger status=" + autochanger.getStatusData().toString());
        sb.append(autochanger.getStatusData().toString());
        sb.append("\n  ");
        sb.append("latchXminus status=");sb.append(autochanger.getLatchXminus().getStatusData().toString());
        sb.append("\n  ");
        sb.append("latchXplus status=");sb.append(autochanger.getLatchXplus().getStatusData().toString());
        sb.append("\n  ");
        return sb.toString();
    }

    /**
     * Initial state for sequence1 :
     * autochanger is in highposition and trucks are empty
     * carousel holds a filter and its clamps are in CLAMPEDONFILTER state.
     * @return
     */
    public String goToInitialState() throws HardwareException, ErrorInCommandExecutionException, BadCommandException {
        this.updateStateWithSensors();
        String ack = "Hardware is at initial state and ready for a sequence1";
        if (carousel.isHoldingFilterAtStandby()) {
                if (autochanger.isAtStandback()) {
                    return ack;
                } else if (autochanger.isAtStandby()) {
                    log.info("=======> Autochanger moving back empty to online position.");
                    autochanger.ungrabFilterAtStandby();      
                    return ack;
                } else {
                    return "Autochanger trucks are in the midle, I don't know what to do.";
                }
   
        } else if (autochanger.isHoldingFilterAtStandback()) {
                if (!carousel.isReadyToGrabAFilterAtStandby()) {
                    carousel.releaseClamps();
                }
                log.info("=======> Autochanger trucks moving back the filter to standby position.");
                autochanger.moveFilterToStandby(dummyFilter);
                
                // carousel checks if the filter is clamped at standby position
                log.info("=======> Carousel is grabbing filter at standby position.");
                carousel.grabFilterAtStandby(getDummyFilter());

                //then the autochanger can unclamp the filter and go to swapout position
                log.info("=======> Autochanger moving back empty to online position.");
                autochanger.ungrabFilterAtStandby();
                return ack;
                               
        } else if (autochanger.isHoldingFilterAtStandby()) {
                //in this case the carousel doesn't hold the filter
                //autochanger trucks hold the filter and move it to Online Position
                log.info("=======> Autochanger trucks about to move loaded with the filter to standback position.");
                autochanger.moveFilterToStandback(dummyFilter);

                log.info("=======> Carousel about to release Clamps.");
                carousel.releaseClamps();

                //then we go back to the initial position :
                //autochanger move back the filter in standby position
                log.info("=======> Autochanger trucks moving back the filter to standby position.");
                autochanger.moveFilterToStandby(dummyFilter);


                // carousel checks if the filter is clamped at standby position
                log.info("=======> Carousel is grabbing filter at standby position.");
                carousel.grabFilterAtStandby(getDummyFilter());

                //then the autochanger can unclamp the filter and go to swapout position
                log.info("=======> Autochanger moving back empty to online position.");
                autochanger.ungrabFilterAtStandby();
                return ack;
            
        } else if (!autochanger.isAtStandback() && !autochanger.isAtStandby()) {
                return "Autochanger trucks are in the midle, I don't know what to do.";
        } else {
                return "Nobody holds the filter : I don't know how to go to initial state.";
        }
    }

    /**
     * This method is the main sequence of commands that we want to test on the Single Filter Test.
     * It executes this sequence of commands once.
     * From the initial position (autochanger trucks empty in the higher position) :
     * <UL> <LI> the autochanger trucks go down along the rails to standby position, 
     *      <LI> the autochanger grabs the filter at standby position, 
     *      <LI> the carousel ungrabs the filter at standby position,
     *      <LI> the autochanger moves the filter to the higher position,
     *      <LI> the carousel releases the clamps to be ready to clamp a filter again,
     *      <LI> the autochanger moves back the filter at standby position,
     *      <LI> the carousel checks if the filter is clamped at standby position,
     *      <LI> the autochanger unlocks the filter latches,
     *      <LI> the autochanger trucks go empty to the higher position.
     * </UL>
     * @return a message for the operator
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */
    public String executeSequence1() throws BadCommandException, ErrorInCommandExecutionException, HardwareException, CanOpenException {

        //autochanger trucks go to standby position to grab the filter held by carousel
        log.info("=======> Autochanger trucks about to move empty to standby position.");
        autochanger.grabFilterAtStandby(dummyFilter);

        // carousel clamps are opened and go in the READYTOCLAMP state
        // when autochanger moves up the filter
        log.info("=======> Carousel is ungrabbing filter at standby position.");
        carousel.ungrabFilterAtStandby(dummyFilter);

        //autochanger trucks hold the filter are move it to Online Position
        log.info("=======> Autochanger trucks about to move loaded with the filter to standback position.");
        autochanger.moveFilterToStandback(dummyFilter);

        log.info("=======> Carousel about to release Clamps.");
        carousel.releaseClamps();

        //then we go back to the initial position :
        //autochanger move back the filter in standby position
        log.info("=======> Autochanger trucks moving back the filter to standby position.");
        autochanger.moveFilterToStandby(dummyFilter);


        // carousel checks if the filter is clamped at standby position
        log.info("=======> Carousel is grabbing filter at standby position.");
        carousel.grabFilterAtStandby(getDummyFilter());

        //then the autochanger can unclamp the filter and go to swapout position
        log.info("=======> Autochanger moving back empty to online position.");
        autochanger.ungrabFilterAtStandby();

        log.info("=======> Execution of Sequence1 completed.");
        return getName() + " Execution of Sequence1 completed";
    }

    /**
     * This method will execute nb times the Sequence1 : the parameter nb
     * is given by the user at the console.
     * @param nb : number of times we want to execute the Sequence1
     * @return 
     */
    public String executeSequence1(int nb) throws BadCommandException, ErrorInCommandExecutionException, HardwareException, CanOpenException {


        if ((nb < 1) || (nb > 20000)) {
            throw new IllegalArgumentException("nb has to be comprise between 1 and 20000");
        }

        for (int i = 1; i < nb+1; i++) {

            log.info("===========================BEGIN execution Sequence1 nb=" + i + " ================");
            executeSequence1();
            log.info("===========================END execution Sequence1 nb=" + i + " ================");
        }
        return getName() + ":" + nb + " executions of Sequence1 completed";
    }

    boolean autochangerLatchesAreLocked() {
        throw new UnsupportedOperationException("Not yet implemented");
    }



    public Filter getFilterAtStandby() {
        return carousel.getFilterAtStandby();
    }

    @Override
    public void processUpdate(Observable source, ValueUpdate v) {

        //Unuseful : this is already done in BasicAutoChangerModule and in CarouselModule
        //carousel.sendToStatus(carousel.getStatusData());       
        //autochanger.sendToStatus(autochanger.getStatusData());
    }
    
    public String printFilters() {
        StringBuilder sb = new StringBuilder(getName() + " printFilters \n ");
        sb.append(dummyFilter.getName());
        sb.append(" position: ").append(this.carousel.getFilterPosition(dummyFilter));
        sb.append(" socket number: ").append(this.carousel.getSocketNumber(dummyFilter));
        sb.append(" standby position: ").append(this.carousel.getStandbyPositionForFilter(dummyFilter));
        log.info(sb.toString());
        return sb.toString();
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n").append(carousel.toString());
        sb.append("\n").append(autochanger.toString());
        return sb.toString();
    }

    /**
     * This command updates the state of the carousel and of the autochanger in reading the sensors.
     * It has to be used in Engineering mode at the console when we send commands step by step.
     * @throws HardwareError
     * @throws ErrorInCommandExecutionException
     * @throws BadCommandException
     */
    public void updateStateWithSensors() throws HardwareException, ErrorInCommandExecutionException, BadCommandException {
        carousel.updateStateWithSensors();
        autochanger.updateStateWithSensors();
    }
    
    

    public SftCarouselModule getCarouselModule() {
        return carousel;
    }

    public SftAutoChangerModule getAutoChangerModule() {
        return autochanger;
    }
    
    public SingleFilterTestConfig getFullState() {
        return FcsUtils.createSingleFilterTestConfig(this);
    }
    
}
