/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.HashMap;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.bus.Alarm;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.state.State;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.CWrapperNotConnected;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenCommunicationException;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenDeviceException;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenException;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenGenericException;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenMotionException;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenTemperatureException;
import org.lsst.ccs.subsystems.fcs.errors.CanOpenVoltageException;
import org.lsst.ccs.subsystems.fcs.errors.NodeIDMismatchException;
import org.lsst.ccs.subsystems.fcs.errors.ErrorInBootingHardwareProcessException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareNotDetectedException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.errors.UnexpectedBootMessageReceivedException;
import org.lsst.ccs.subsystems.fcs.FcsAlarm;
import org.lsst.ccs.subsystems.fcs.errors.HardwareConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;

/**
 * This Module starts a tcpip server, waits for the connection of a client whose 
 * name is the value of the field myClientName.
 * This client is supposed to be a CanOpen C-wrapper.
 * When the client starts it scans the can open network and send to the server
 * some information on the can open nodes living on the network.
 * 
 * @author virieux
 */
public class CanOpenProxy extends FcsTcpProxy {
    
    
    
    //private ApcTcpProxy tcpProxy;
    /*This is the key word that has to be sent by the client when it connects to the tcp proxy. */
    private String myClientName;
    
    /*number of can open nodes expected on the network */
    private int expectedNodesNB;
    
    /*number of booted nodes*/
    private int bootedNodesNB;
    
   
    
    /*An array of booted hardware*/
    private CanOpenNode[] nodes = new CanOpenNode[getExpectedNodesNB()];

    
    
    private boolean hardwareBootProcessEnded = false;
    private boolean hardwareIdentified = false;
    private boolean hardwareInitialized = false;
    
    /**
     * A timeout for the booting process : 
     * we don't want to wait longer that this elapse of time.
     * This has to be tuned during the test bench.
     * UNIT = milliseconds
     */
    private long hardwareBootTimeout;
    
    private long hardwareBootProcessBeginTime;
    
    /*An array of hardware controled by this main Module*/
    //PieceOfHardware[] hardwareList = {actuatorA, actuatorB, actuatorC};
    public PieceOfHardware[] hardwareList;
    
    private boolean canOpenNodeNumbersOK = false;
    private boolean canOpenNodesChecked = false;

    public CanOpenProxy(String aName, int aTickMillis, int portNumber, String aClientName, int expectedNodesNB, long hardwareBootTimeout) {
        super(aName, aTickMillis, portNumber);
        this.myClientName = aClientName;
        this.expectedNodesNB = expectedNodesNB;
        this.hardwareBootTimeout = hardwareBootTimeout;
    }

    /**
     * This methods stops the CWrapper client, and the TCP server.
     * Then it starts a new TCP server for the next connection of the hardware.
     */
    public void disconnectHardware() throws InterruptedException {
        this.stop();
        Thread.sleep(tickMillis);
        initModule();
    }

    
    
    /**
     * To store the values returned by the PDO.
     */
    public static class PDOStorage {
        
        String sep = "_";
        String varSuffixe = "ADC";
        /**
         * a hash map for the storage of the values sent from the PDO.
         * key = varSuffixe + nodeID + sep + input numero  Example : ADC23_5 
         * value = value in hexa
         */
        //private HashMap<String, String> mapSensorsValues = new HashMap<String, String>();    
        private HashMap<String, String> mapSensorsValues = new HashMap<>(); 
        
        /**
         * update the PDO storage in reading a PDO line.
         * 
         * @param pdoLine something like : "ADC1_1=10,ADC1_2=F53B, ...."
         */
        void updatePDOs(String pdoLine) {
            String[] words = pdoLine.split(",");
            for (int ix=1; ix<words.length; ix++) {
                //example toStore= ADC1_1=10
                String[] toStore = words[ix].split("=");
                //example varName= ADC1_1
                String varName = toStore[0];
                String varValue = toStore[1];
                String[] mots = varName.split(sep);
                String nodeID = mots[0].substring(3);
                //System.out.println(String.format("%nwords(%s):%s nodeID=%s", ix, words[ix], nodeID));
                String key = nodeID + sep + mots[1];
                mapSensorsValues.put(varName,varValue);
            }

        }

        public String getVarValue(String nodeID, String inputNB) {
            String token = varSuffixe + nodeID + sep + inputNB;
            return mapSensorsValues.get(token);
        }

        public String toString() {
            StringBuilder sb = new StringBuilder("PDO STORAGE:");
            sb.append("(SIZE=").append(mapSensorsValues.size()).append(")");
            sb.append(mapSensorsValues.toString());
            return sb.toString();
        }  
    }
    
    private PDOStorage pdoStorage;

        
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    

    /**
     * @return the expectedNodesNB
     */
    public int getExpectedNodesNB() {
        return expectedNodesNB;
    }

    /**
     * @param expectedNodesNB the expectedNodesNB to set
     */
    public void setExpectedNodesNB(int nodeNB) {
        this.expectedNodesNB = nodeNB;
    }

    public long getHardwareBootTimeout() {
        return hardwareBootTimeout;
    }

    @ConfigChanger
    public void setHardwareBootTimeout(long hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }
    
    @ConfigChanger
    public void setHardwareBootTimeout(int hardwareBootTimeout) {
        this.hardwareBootTimeout = hardwareBootTimeout;
    }
    
    
    
     /**
     * @return the bootedNodesNB
     */
    public int getBootedNodesNB() {
        return bootedNodesNB;
    }


   

    /**
     * @return the myClientName
     */
    public String getMyClientName() {
        return myClientName;
    }

    public PDOStorage getPdoStorage() {
        return pdoStorage;
    }
    
    public boolean isHardwareIdentified() {
        return hardwareIdentified;
    }
    
    public boolean isHardwareInitialized() {
        return hardwareInitialized;
    }
    
    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/
    
    @Override
    public void initModule() {
       super.initModule();
       log.info("Expected Nodes NB="+expectedNodesNB);
       //ATTENTION getExpectedNodesNB()=0 if expectedNodesNB is not set in the description file !
       nodes = new CanOpenNode[expectedNodesNB];
       hardwareBootProcessBeginTime = 0;
       bootedNodesNB = 0;
       hardwareBootProcessEnded = false;
       hardwareIdentified = false;
       hardwareInitialized = false;
       canOpenNodeNumbersOK = false;
       canOpenNodesChecked = false;
       this.pdoStorage = new PDOStorage();
    }
    

    
    
    
    
    @Override
    public void tick() {
        //INITIALISATION  
        //TODO : All the initialisations have to go in the startTicking method
        
        // we can start the hardware booting process when the tcp client is ready. 
        if ((hardwareBootProcessBeginTime == 0) && isReady(myClientName)) {
            hardwareBootProcessBeginTime = System.currentTimeMillis();
        }
        
        //we want to check if the hardware is booted.
        if ((hardwareBootProcessBeginTime > 0) && !hardwareBootProcessEnded) { 
            long hardwareBootDuration = System.currentTimeMillis() - hardwareBootProcessBeginTime;
            if (this.bootedNodesNB == this.getExpectedNodesNB()) {
                hardwareBootProcessEnded = true;
            } else if (hardwareBootDuration > hardwareBootTimeout) {
                hardwareBootProcessEnded = true;
                log.error("Timeout expired during hardware booting process ");
                this.getSubsystem().updateState(State.InError, 
                        "Timeout expired during hardware booting process : number of booted is not was expected"); 
            }
                
        }
        
        // and retrieve some information in the hardware
        if ((hardwareBootProcessEnded) && !hardwareIdentified) {          
            try {
                identifieHardware();
                
            } catch (TimeoutException ex) {
                log.error("Timeout expired during hardware identifie process ");
                this.getSubsystem().updateState(State.InError,"Timeout expired during hardware identifie process." + ex.getMessage());
            }
            hardwareIdentified = true;
                       
        }
        
        if (this.hardwareIdentified && !canOpenNodesChecked) {
            try {
                canOpenNodesChecked = checkCanOpenNodes();               
                log.info("Hardware configuration is checked");
            } catch (HardwareConfigurationException ex) {
                canOpenNodesChecked = false;
                log.error("ERROR in configuration for a Can Open node: " + ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                this.getSubsystem().updateState(State.InError, "ERROR in configuration for a nodeID" + ex.toString());
            } catch (HardwareNotDetectedException ex) {
                canOpenNodesChecked = false;
                log.error("HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                this.getSubsystem().updateState(State.InError, "HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
            } catch (ErrorInBootingHardwareProcessException ex) {
                canOpenNodesChecked = false;
                log.error("HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                this.getSubsystem().updateState(State.InError, "HARDWARE NOT DETECTED : POWER FAILURE ?" + ex.getMessage());
            } catch (BadCommandException ex) {
                log.error(ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
            }
        }
        
        if (this.canOpenNodesChecked && !this.hardwareInitialized) {
            try {
                this.initializeHardware();
                this.hardwareInitialized = true;
                log.info("Hardware is initialized");
            } catch (Exception ex) {
                this.hardwareInitialized = false;
                log.error("SOME HARDWARE COULDN'T BE INITIALIZED:" + ex.getMessage());
                Alarm alarm = new FcsAlarm(ex.toString());
                this.sendToStatus(alarm);
                //this.getSubsystem().updateState(State.InError, "HARDWARE COULDN'T BE INITIALIZED:" + ex.getMessage());
            }
        }
        

    }
    
    
    
    /*Check if a piece of hardware with nodeID aNodeID is already booted or not.
     * If it is booted it is on the nodes table.    
     * It means that this node ID is on the can open bus.
     */
    public boolean isBooted(String aNodeID) {
        
        if (bootedNodesNB == 0) return false;
        boolean nodeFound = false;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            if (!(nodes[i] == null) &&  (nodes[i].getNodeID().equals(aNodeID))) {
                nodeFound = true;
            }
        }
        return nodeFound;
    }
    
    /* Check if a piece of hardware is already booted or not.
     * If it is booted its node is on the nodes table. 
     * This means that this node ID is on the can bus.
     */
    public boolean isBooted(PieceOfHardware piece) {
        
        if (bootedNodesNB == 0) return false;
        boolean pieceFound = false;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            
            if (!(nodes[i] == null) &&  (nodes[i].getNodeID().equals(piece.getNodeID()))) {
                pieceFound = true;
            }
        }
        return pieceFound;
    }
    
     /**
     * Check if a piece of hardware is booted 
     * and if its can open node ID is correct.
     */
    public boolean isNodeIdBootedAndOK(PieceOfHardware piece) {
        
        if (bootedNodesNB == 0) return false;
        boolean pieceFound = false;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            
            if (!(nodes[i] == null) &&  (nodes[i].getNodeID().equals(piece.getNodeID()))
                    && (nodes[i].getSerialNB().equals(piece.getSerialNB()))) {
                pieceFound = true;
            }
        }
        return pieceFound;
    }

    /**
     * List the can open nodes which are in the nodes table.
     * @return the list of can open nodes and the information stored in this.nodes.
     */
    public String listNodes() {
        
        StringBuilder sb = new StringBuilder("Nodes LIST = (values are in HEXA)");
        for (int i=0; i < this.bootedNodesNB; i++) {
            sb.append("\n");
            if (nodes[i] == null) {
                sb.append("null node");
            } else sb.append(nodes[i].toString()) ;
        }
        return sb.toString();

    }
    
    
    /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper. 
     * Test if the command is valid before sending it to the can open stack.
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     * 
     */
    public String sendCanOpen(String command) throws TimeoutException, BadCommandException  {
        try {
          String[] words = command.split(",");
          String keyWord = words[0]; 
          if (keyWord.equals("rsdo")) {
              if (words.length != 4) throw new BadCommandException("rsdo command must have 3 parameters: nodeID, index, subindex");
          } else if (keyWord.equals("wsdo")) {
              if (words.length != 6) throw new BadCommandException("wsdo command must have 5 parameters: nodeID, index, subindex, size, data");
          } else if (keyWord.equals("info")) {
              if (words.length != 2) throw new BadCommandException("info command must have 1 parameter: nodeID");
          } else if (keyWord.equals("sync")) {
              if (words.length != 1) throw new BadCommandException("sync command takes no parameter");
          } else if (keyWord.equals("srtr")) {
              if (words.length < 2) throw new BadCommandException("srtr command takes at least one parameter (cobib)");
          } else if (keyWord.equals("quit")) {
              if (words.length != 1) throw new BadCommandException("quit command takes no parameter");
          } else throw new IllegalArgumentException(command);
          return (String) call(getMyClientName(), command);
          
        }   catch (CWrapperNotConnected ex) {
            log.error(ex.toString());
            Alarm alarm = new FcsAlarm(ex.toString());
            this.sendToStatus(alarm);
            this.getSubsystem().updateState(State.InError, "Can't communicate with harwdare because CWrapper is not connected." + ex.getMessage());
            return ex.toString();
        }  
    }
      
    
    /**
     * build a Can Open wsdo Command that can understand the C wrapper.
     * exemple : wsdo,2,6411,01,2,3000
     */
    private static String buildWsdoCommand(String nodeID, String index, String subindex, String size, String data) {
        char sep = ',';
        String command = String.format("wsdo%1$s%2$s%1$s%3$s%1$s%4$s%1$s%5$s%1$s%6$s",sep,nodeID,index,subindex,size,data);
        //System.out.println("Command built = " + command);
        return command;
    }
    
    /**
     * build a Can Open rsdo Command that can understand the C wrapper.
     * exemple : rsdo,1,1018,0
     */
    private static String buildRsdoCommand(String nodeID, String index, String subindex) {
        char sep = ',';
        String command = String.format("rsdo%1$s%2$s%1$s%3$s%1$s%4$s",sep,nodeID,index,subindex);
        return command;
    }

    /**
     * Write a SDO request and send it to the can open stack, then analyses the response
     * or throws an exception if the request failed.
     * The parameters are given in HEXA.
     * @param nodeID FORMAT=HEXA
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=HEXA
     * @param value FORMAT=HEXA
     * @return "OK" if OK
     */
    public String writeSDO(String nodeID, String index, String subindex, String size, String value) throws SDORequestException  {
        try {
            String sdoResponseLine = sendCanOpen(buildWsdoCommand(nodeID,index,subindex,size,value));
            String[] words = sdoResponseLine.split(",");
            String command = words[0];
            String receivedNodeID = words[1];
            String errorCode = words[2];

            if (errorCode.equals("0")) {
                return "OK";
            } else if  (errorCode.equals("-1")) {
                log.error("Wrong SDO command : " + sdoResponseLine);
                throw new SDORequestException("Wrong SDO command : " + sdoResponseLine);
            } else {
                log.error("SDO request was going wrong");
                String error = String.format("%08d", Integer.parseInt(errorCode));
                log.info("ERROR=" + error);
                String errorName = CanOpenErrorsTable.commErrorCodes.getProperty(error);
                throw new SDORequestException("SDO request was going wrong", errorCode, nodeID, index, subindex, errorName);
            }
        } catch (BadCommandException ex) {
            SDORequestException err = new SDORequestException("write SDO request bad command : ", nodeID,"bad command", index, subindex, ex.getMessage());
            log.error(String.format("%s %s for %s this should not happen.", "malformed SDO request : ",ex.getMessage(),nodeID));
            throw err;
        } catch (TimeoutException ex) {
            SDORequestException err = new SDORequestException("write SDO request timeout expired : ", nodeID,"timeout expired", index, subindex, ex.getMessage());
            log.error(String.format("%s %s is %s stopped ?", "SDO request timeout expired : ",ex.getMessage(),nodeID));
            throw err;
        }
       
    }
    
     /**
     * Command to be used by the end user at the console.
     * Write a SDO message and send it to the can open stack.
     * The parameters are given in the following format:
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=HEXA
     * @param subindex FORMAT=HEXA
     * @param size FORMAT=decimal
     * @param value FORMAT=decimal
     * @return 
     */
    public String writeSDO(int nodeID, String index, String subindex, int size, int value) throws SDORequestException, BadCommandException {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        if (size <0 || size > 4) 
            throw new IllegalArgumentException("size must be > 0 and < 4");
        return writeSDO(Integer.toHexString(nodeID),index,subindex,Integer.toHexString(size),Integer.toHexString(value));
    }
    
    
     /**
     * Read a SDO with the given index and subindex
     * RETURNS the value read in hexa (String) if no error occured
     * or returns the error code if a error was detected.
     *
     */
    public String readSDO(String nodeID, String index, String subindex) throws SDORequestException {
        try {
            String sdoLine = sendCanOpen(buildRsdoCommand(nodeID,index,subindex));
            String[] words = sdoLine.split(",");
            int responseLength = words.length;
            
            String command = words[0];
            String receivedNodeID = words[1];
            String errorCode = words[2];
           
            if (errorCode.equals("0") && (responseLength > 3)) {
                return words[3];
            } else if  (errorCode.equals("-1")) {
                log.error("Wrong SDO command : " + sdoLine);
                throw new SDORequestException("Wrong SDO command : " + sdoLine);
            } else {
                log.error("SDO request was going wrong, received response=" + sdoLine);
                String error = String.format("%08d", Integer.parseInt(errorCode));
                log.info("ERROR=" + error);
                String errorName = CanOpenErrorsTable.commErrorCodes.getProperty(error);
                throw new SDORequestException("SDO request was going wrong", nodeID, errorCode, index, subindex, errorName);
            }
        } catch (BadCommandException ex) {
            SDORequestException err = new SDORequestException("read SDO request: ", nodeID, "bad command",index, subindex, ex.getMessage());
            log.error(String.format("%s %s on node %s should not happen", "bad command : ",ex.getMessage(),nodeID));
            throw err;
        } catch (TimeoutException ex) {
            SDORequestException err = new SDORequestException("read SDO request timeout expired : ", nodeID, "timeout expired",index, subindex, ex.getMessage());
            log.error(String.format("%s %s is %s stopped ?", "SDO request timeout expired : ",ex.getMessage(),nodeID));
            throw err;
        }
       
    }
    
    /**
     * Command to be used by the end user at the console in engineering mode.
     * @param nodeID FORMAT=decimal
     * @param index FORMAT=hexa
     * @param subindex FORMAT=hexa
     * @return the value sent by the can open stack if the read SDO request succeded.
     * @throws SDORequestError 
     */
    public String readSDO(int nodeID, String index, String subindex) throws SDORequestException, BadCommandException  {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        return readSDO(Integer.toHexString(nodeID),index,subindex);
    }
    
    /**
     * This method sends a sync command to the can open stack and returns the reply.
     * example of sync reply we can receive :
     * sync,23_1=25a7,23_2=113,23_3=10,23_4=109,23_5=108,23_6=104,23_7=101,23_8=112
     * @return 
     */
    public PDOStorage readPDOs() throws HardwareException, BadCommandException {
        try {
            String replyToSync = sendCanOpen("sync");           
            //updatePdoStorage
            
            this.pdoStorage.updatePDOs(replyToSync);
            return this.pdoStorage;
        } catch (TimeoutException ex) {
        
            HardwareException err = new HardwareException("Timeout expired when waiting for the reply to a sync command in readPDOs : "
                    + ex.toString());
            throw err;
        }
    }

    /**
     * Identification of the hardware : we want to retrieve the information 
     * stored in the hardware of the booted nodes and update the array of nodes
     * with this information.
     * The Can Open node ID is a main information to retrieve.
     */
    public void identifieHardware() throws TimeoutException {
        log.info("Identification of the hardware");
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            if (!(nodes[i] == null)) {
                try {
                    log.debug("Sending to can open command : info," + nodes[i].getNodeID());
                    Object result = sendCanOpen("info," + nodes[i].getNodeID());
                    log.debug("Received on socket command = " + result);
                    String infoLine = (String) result;
                    String[] words = infoLine.split(",");
                    String command = words[0];
                    String nodeID = words[1];            
                    if (command.equals("info") && nodeID.equals(nodes[i].getNodeID())) { 
                        log.debug("updating Node Info for node " + nodeID);
                        String type = words[2];
                        String vendor = words[3];
                        String productCode = words[4];
                        String revision = words[5];
                        String serialNB = words[6];
                        nodes[i].setNodeInfo(type, vendor, productCode, revision, serialNB);
                    } else {
                        log.error("ERROR for command = " + command + "NodeID = " + nodeID);
                        //TODO throw an exception
                    }
                } catch (BadCommandException ex) {
                    //This will not happen because the can open command : "info, nodeID" is correct.
                }
            }
        }
        log.info("Hardware is identified");
        log.info(listNodes());
    }

    /**
     * This method returns true if : all the hardware items are booted and identified 
     * and the hardware have the node ID expected within the configuration
     * and the hardware is initialized.
     */
    public boolean isHardwareReady() {
        log.info("hardwareBootProcessEnded=" + hardwareBootProcessEnded);
        log.info("hardwareIdentified=" + hardwareIdentified);
        log.info("canOpenNodeNumbersOK=" + this.canOpenNodeNumbersOK);
        log.info("canOpenNodesChecked=" + this.canOpenNodesChecked);
        return this.hardwareBootProcessEnded && this.hardwareIdentified 
                && this.canOpenNodeNumbersOK && this.canOpenNodesChecked
                && this.hardwareInitialized;
    }

    /**
     * This method returns the can open detected on the can open network 
     * for a piece of hardware identified by its serial number.
     * It can return NULL if the Serial Number is not on the can open network.
     * @param serialNB
     * @return 
     */    
    public CanOpenNode getDetectedNodeForSerialNumber(String aSerialNB) {
        CanOpenNode detectedNode = null;
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            if (nodes[i].getSerialNB().equals(aSerialNB)) {
                detectedNode = nodes[i];
            }
        }
        return detectedNode;
    }

    @Override
    void processBootMessage(String nodeID)  {
        if (isBooted(nodeID)) {
            if (hardwareBootProcessEnded) {
                    /*The node is already in the table of nodes : something weird has just happened*/ 
                    String msg = "An asynchronus boot message is arrived for node ID = " + nodeID;
                    log.error(msg);
                    UnexpectedBootMessageReceivedException ex = new UnexpectedBootMessageReceivedException(msg,nodeID);
                    Alarm alarm = new FcsAlarm(ex.toString());
                    this.sendToStatus(alarm);
//                    this.getSubsystem().updateState(State.InError,
//                            " Possible power failure for node : " + nodeID);
              
            }
            
        } else {
            //TODO : if the number of nodes we found is greater than the expectedNodesNB, throw an exception.
            /*the node is not already in the table of nodes : we have to add it*/
            log.info("bootedNodesNB=" + bootedNodesNB);
            log.info("Expected Nodes NB="+getExpectedNodesNB());
            nodes[bootedNodesNB] = new CanOpenNode(nodeID);
            bootedNodesNB++;
            log.info("Node " + nodeID + " added");
            log.info("Hardware booted nb=" + bootedNodesNB);
        }
    }
    
    @Override
    void processEmcyMessage(String message) {
        //TODO put the corresponding node InERROR state ?
            String[] words = message.split(",");
            String nodeID = words[1];
            //TODO try to know why errCode is never used
            String errCode = words[2];
            String errReg = words[3];
            log.error("Received EMERGENCY message for nodeID: " + nodeID);
            log.error("message=" + message);
            String errorName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errReg);
            log.info("errorName=" + errorName);
            
        try {
                     
            if (errReg.equals("01")) throw new CanOpenGenericException(errorName,nodeID, errReg);
            else if (errReg.equals("04")) throw new CanOpenVoltageException(errorName,nodeID, errReg);
            else if (errReg.equals("08")) throw new CanOpenTemperatureException(errorName,nodeID, errReg);
            else if (errReg.equals("10")) throw new CanOpenCommunicationException(errorName,nodeID, errReg);
            else if (errReg.equals("20")) throw new CanOpenDeviceException(errorName,nodeID, errReg);
            else if (errReg.equals("80")) throw new CanOpenMotionException(errorName,nodeID, errReg);
            else throw new CanOpenException("Received EMERGENCY message (UNKNOWN Cnn Open Error) for nodeID: " + nodeID);
                
            

        } catch (CanOpenException ex) {
            
            log.error(ex.getMessage());
            Alarm alarm = new FcsAlarm(ex.toString());
            this.sendToStatus(alarm);
            //TODO in normal mode : put the subsystem in ERROR
            String err = String.format("NodeID %s (%02d in decimal) : %s ",nodeID,Integer.parseInt(nodeID, 16), ex.getMessage());
            this.getSubsystem().updateState(State.InError, err);
        }
    }
    

    
    @Override
    void processUnknownCommand(String message) {
        String[] words = message.split(",");
        String command = words[1];
        String errorMessage = "Unknown command:" + command;
        log.error(errorMessage);
        
    }
    
    public boolean checkCanOpenNodes() throws BadCommandException, ErrorInBootingHardwareProcessException, HardwareConfigurationException, HardwareNotDetectedException {
        if (!(isReady(this.myClientName) && isHardwareIdentified()))
                    throw new BadCommandException("Hardware not booted or not identified");
        
        log.info("CHECKING HARDWARE CONFIGURATION ");
        if (this.bootedNodesNB == 0) {
            
            ErrorInBootingHardwareProcessException ex1 = new ErrorInBootingHardwareProcessException("NO HARDWARE DETECTED : POWER FAILURE ?", this.hardwareBootTimeout, this.bootedNodesNB, this.expectedNodesNB);
            log.error(ex1.toString());
            throw  ex1;      
        }
        if (this.bootedNodesNB != this.expectedNodesNB) {

            ErrorInBootingHardwareProcessException ex2 = new ErrorInBootingHardwareProcessException("SOME HARDWARE IS MISSING : POWER FAILURE ?", this.hardwareBootTimeout,this.bootedNodesNB, this.expectedNodesNB);
            log.error(ex2.toString());
            throw ex2;       
        }
        this.canOpenNodeNumbersOK = true;
        boolean configNodesID = true;
        for (PieceOfHardware hardware : this.hardwareList)  {
            log.info("ABOUT TO CHECK: " + hardware.getName() + " NODE_ID=" + hardware.getNodeID());
            boolean configOK = checkCanOpenNodeConfiguration(hardware);
            configNodesID = configNodesID && configOK;
            log.info(hardware.getName() + " CONFIG OK=" + configOK);
        }
       
        return configNodesID;
        
    }
    

    
        
     /**
     * When the hardware is booted and we have retrieve the information, 
     * we want to check for each piece of hardware that the node ID stored 
     * in the Configuration data base is the same than the node ID which is stored 
     * on the hardware with a switch : so we compare the serial number found on the can open bus
     * for the piece of hardware node id with the serial number of the harwdare configuration.
     * @return 
     */
    public boolean checkCanOpenNodeConfiguration(PieceOfHardware pieceOfHardware) throws BadCommandException, HardwareConfigurationException, HardwareNotDetectedException  {
        
        if (!(this.hardwareBootProcessEnded && this.hardwareIdentified))
                    throw new BadCommandException("Hardware not booted or not identified");
        if (bootedNodesNB == 0) return false;
        
        log.info("Checking configuration for " + pieceOfHardware.getName());
        
        boolean serialNBFound = false;
        boolean nodeIDFound = false;
        String bootedNodeForSerial = null;
        String bootedSerialForNode = null;
        
        for (int i=0; i < this.bootedNodesNB; i++ ) {
            
            if (!(nodes[i] == null) && (nodes[i].getNodeID().equalsIgnoreCase(pieceOfHardware.getNodeID()))) {
                //the node id of this piece of hardware has been found in the nodes list.
                nodeIDFound = true;
                bootedSerialForNode = nodes[i].getSerialNB();                 
            }
            
            if (!(nodes[i] == null) &&  (nodes[i].getSerialNB().equalsIgnoreCase(pieceOfHardware.getSerialNB()))) {  
                //the serial number of this piece of hardware has been found in the nodes list.
                serialNBFound = true;
                bootedNodeForSerial = nodes[i].getNodeID();
            }
        }
        
        if (!nodeIDFound && !serialNBFound) {
            this.getSubsystem().updateState(State.InError,
                    " Possible power failure for node ID: " + pieceOfHardware.getNodeID());
            HardwareNotDetectedException ex = new HardwareNotDetectedException("HARDWARE NOT DETECTED", 
                pieceOfHardware.getName(),pieceOfHardware.getNodeID(),pieceOfHardware.getSerialNB());
            log.error(ex.toString());
            throw ex;
            
        } else if (nodeIDFound && !serialNBFound) {
            //the node id of this piece of hardware is booted but the serial number associated is not the good one.
            log.error("Serial number found for this can open node id =" +  bootedSerialForNode); 
            String msg = String.format(" Serial number %s associated to %s was not found on the Can Open Bus, "
                    + "Serial number found for node id %s = %s" ,pieceOfHardware.getSerialNB(),pieceOfHardware.getName(),pieceOfHardware.getNodeID(), bootedSerialForNode );
            throw new HardwareConfigurationException(msg, pieceOfHardware);
            
        } else if (!nodeIDFound && serialNBFound) {
            //the serial number of this piece of hardwre has been found but not the node id
            log.error(pieceOfHardware.toString() +
                        " has a wrong node ID on the Can Open Bus :" + bootedNodeForSerial);
            //throw new NodeIDMismatchException(" Serial number found with another node ID  ", pieceOfHardware, bootedNodeForSerial,pieceOfHardware.getSerialNB());
            String msg = String.format(" NodeID %s associated to %s was not found on the Can Open Bus, "
                    + "NodeID found for serialNB %s = %s" ,pieceOfHardware.getNodeID(),pieceOfHardware.getName(),pieceOfHardware.getSerialNB(),bootedNodeForSerial);
            throw new HardwareConfigurationException(msg, pieceOfHardware);
       
        } else if (nodeIDFound && serialNBFound) {
            if ((pieceOfHardware.getNodeID().equals(bootedNodeForSerial)) 
                    && (pieceOfHardware.getSerialNB().equals(bootedSerialForNode))) {
                //and it's the same node ID : every thing is OK
//                    
                log.info(pieceOfHardware.getName() + " is booted :" + pieceOfHardware.toString());
                return true;
            } else  {
                    log.error(pieceOfHardware.toString() +
                        " does not match hardware on the Can Open Bus");
                    throw new NodeIDMismatchException(" ERROR in HARDWARE CONFIGURATION", pieceOfHardware,  bootedNodeForSerial, bootedSerialForNode );
                
            } 
        } 
        return false;
    }
    
    
    public String listHardware() {
        StringBuilder sb = new StringBuilder();
        for (PieceOfHardware pieceOfHardware : this.hardwareList)  {
            sb.append(pieceOfHardware.toString());
            sb.append("**");
        }
        return sb.toString();      
    }
    
    public String initializeHardware() throws BadCommandException, Exception  {
        if (!(this.hardwareIdentified && this.canOpenNodesChecked))
                    throw new BadCommandException("Hardware not identified yet or can open nodes have not beeen checked.");
        boolean noError = true;
        String errMsg = null;
        StringBuilder err = new StringBuilder();
        for (PieceOfHardware pieceOfHardware : this.hardwareList)  {
            try {
                pieceOfHardware.initializeHardware();
            } catch (Exception ex) {
                noError = false;                
                err.append(ex.getMessage());
            }
        }
        if (noError) {
            this.hardwareInitialized = true;
            return "All pieces of hardware are initialized";
        } else throw new Exception(err.toString());
    }
    
    /**
     * Configure a node as a hearbeat producer.
     * Every heartbeat_time milliseconds this node will produce a message like:
     * 0x700+nodeID: 05 - 1340268404.980000 
     * @param nodeID node ID in hexa
     * @param heartbeat_time FORMAT=hexa UNIT=milliseconds
     */
    public String configAsHeartbeatProducer(String nodeID, String heartbeat_time) throws SDORequestException, TimeoutException, BadCommandException {
        return writeSDO(nodeID,"1017","0","2",heartbeat_time);
    }
    
    /**
     * Command to be used by the end users.
     * Configure a node as a heartbeat producer.
     * Parameters given in decimal format.
     * @param nodeID FORMAT=decimal
     * @param heartbeat_time FORMAT=decimal UNIT=milliseconds
     */
    public String configAsHeartbeatProducer(int nodeID, int heartbeat_time) throws SDORequestException, TimeoutException, BadCommandException {
        
        if (heartbeat_time < 0 || heartbeat_time > 65025)
            throw new IllegalArgumentException("heartbeat_time is coded on 2 bytes"
                    + " can't be > 65535");
        return configAsHeartbeatProducer(Integer.toHexString(nodeID), Integer.toHexString(heartbeat_time));
    }
    
     /**
     * Configure a node as a hearbeat consumer.
     * Every heartbeat_time milliseconds this node will produce a message like:
     * 0x700+nodeID: 05 - 1340268404.980000 
     * @param nodeID node ID in decimal
     * @param heartbeat_time FORMAT=decimal UNIT=milliseconds
     */
    public String configAsHeartbeatConsumer(int nodeID, int producerNodeID, int heartbeat_time) throws SDORequestException, TimeoutException, BadCommandException {
        if (nodeID <0 || nodeID > 127) 
            throw new IllegalArgumentException("nodeID must be > 0 and < 128");
        if (producerNodeID <0 || producerNodeID > 127) 
            throw new IllegalArgumentException("producerNodeID must be > 0 and < 128");
        if (heartbeat_time < 0 || heartbeat_time > 65025)
            throw new IllegalArgumentException("heartbeat_time is coded on 2 bytes"
                    + " can't be > 65535");
        String value = String.format("%04x",producerNodeID) + String.format("%04x",heartbeat_time);
        return writeSDO(Integer.toHexString(nodeID),"1016","1","4",value);
    }   
    
}
