/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.errors.ClampMotorConfigurationException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 * A model general for a controller EPOS
 * @author virieux
 */
public class CanOpenEPOS extends CanOpenDevice implements Actuator {
    
    public boolean on;
    public int readValue;
    public int sentValue;



    @Override
    public boolean isOn() {
        return on;
    }

    @Override
    public boolean isOff() {
        return !on;
    }

    @Override
    public int getReadValue() {
        return readValue;
    }

    @Override
    public int getSentValue() {
        return sentValue;
    }

    @Override
    //for the GUI 
    //TODO return a real value
    public int getSentCurrentMinValue() {
        return 0;
    }

    @Override
    //for the GUI
    //TODO return a real value
    public int getSentCurrentMaxValue() {
        return 3000;
    }



    
    public enum EposMode {
	VELOCITY, CURRENT, PROFILE_POSITION_MODE, POSITION_MODE
    }
    
    /* Mode of Operation */
    protected CanOpenEPOS.EposMode mode;
    
    /*the motor which is driven by this actuator*/
    final protected Motor motor;
    
        //Used because we have to wait during the time needed to enable the actuator (very short but not null !)
    //of the clamp
    final Lock lock = new ReentrantLock();
    final Condition enablingCompleted = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean enabling = false;

    public CanOpenEPOS(String aName, int aTickMillis, 
            String nodeID, String serialNB, 
            Motor aMotor, String aModeInString) {
        super(aName, aTickMillis, nodeID, serialNB);
        this.motor = aMotor;
        this.mode = EposMode.valueOf(aModeInString);
        
    }
    
    // All the getters following are for the GUI needs
    public int getMotor_continuousCurrentLimit() {
        return motor.getContinuousCurrentLimit();
    }


    public int getMotor_maxSpeedInCurrentMode() {
        return motor.getMaxSpeedInCurrentMode();
    }



    public int getMotor_outputCurrentLimit() {
        return motor.getOutputCurrentLimit();
    }


    public int getMotor_polePairNumber() {
        return motor.getPolePairNumber();
    }


    public int getMotor_thermalTimeConstantWinding() {
        return motor.getThermalTimeConstantWinding();
    }
    
    
    public int getMotor_type() {
        return motor.getMotorType();
    }


    
     /**
     * Reads the current value or the velocity value stored in the EPOS and updates the readValue field.
     */
    public void updateReadValue() throws BadCommandException, SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            this.readValue = readVelocity();
        } else if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
            this.readValue =  readCurrent();
        } else throw new BadCommandException(getName() + "is not in VELOCITY or CURRENT mode");
    } 
    
       @Override
    public void initModule() {
        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        mode = CanOpenEPOS.EposMode.CURRENT;
    }
    
    @Override
    public String initializeHardware() throws Exception {
        this.initialized = true;
        return getName() + ": no hardware initialization for this device to be done.";
    }

    
    public boolean isEnable() throws SDORequestException {
        lock.lock();
        
        try {
            while(this.enabling) {
                try {
                    this.enablingCompleted.await();
                } catch (InterruptedException ex) {
                    log.debug(getName() + " INTERRUPTED during waiting for the actuator beeing enable");
                }

            }
        String controlWordInHexa = readControlWord();
        int controlWord = Integer.parseInt(controlWordInHexa, 16);
        return (controlWord == 15); /*F in hexa*/
        } finally {
            lock.unlock();
        }
    }
    
    public void faultReset() throws SDORequestException {
        writeControlWord("80");
    }
  
    /**
     * This methods enable the actuator : i.e. this makes the actuator able to receive commands.
     * @return
     * @throws SDORequestError 
     */
    public String enable () throws SDORequestException {
        lock.lock();
        this.enabling = true;
        
        try {
            shutdown();
            switchOnEnableOperation();
            return getName() + " ENABLE";
        } finally {
            this.enabling = false;
            this.enablingCompleted.signal();
            lock.unlock();
        }
    }
    
    
    public String shutdown() throws SDORequestException{      
        writeControlWord("6");
        //TODO test the return code for the command writeSDO
        return getName() + " DISABLE";
    }
    
    public void switchOnEnableOperation() throws SDORequestException {
        writeControlWord("F");
    }
    
    /**
     * This method save the parameters in the actuator memory.
     * @return
     * @throws SDORequestError 
     */
    public Object saveParameters() throws SDORequestException {
        return this.writeSDO("1010", "1", "4", "65766173");
    }
    
    public Object restoreParameters() throws SDORequestException {
        //TODO if is Power Disable
        return this.writeSDO("1011", "1", "4", "64616F6C");
    }
    
    /**
     * Write a value in hexa in the control world (index=6040, subindex=0,size=2)
     * @param value in hexa
     */
    public void writeControlWord(String value) throws SDORequestException {
        this.writeSDO("6040", "0", "2", value);
    }
    
     /**
     * Read the control world (index=6040, subindex=0,size=2)
     * @return value in hexa
     */
    public String readControlWord() throws SDORequestException {
        return this.readSDO("6040", "0");
    }
    
    /**
    * Read the status world (index=6041, subindex=0,size=2)
    * @return value in hexa
    */
    public String readStatusWord() throws SDORequestException {
        return this.readSDO("6041", "0");
    }
    

    

    
    
    public void quickStop() throws SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            writeControlWord("B");
        } else if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
            writeControlWord("2");
        } //TODO else throw an exception ?
    }
    
     /**
     * This methods enables the actuator and puts the motor on in sending the appropriate current value 
     * to the actuator. 
     * 
     * @return a message 
     */
    public String on() throws BadCommandException, ErrorInCommandExecutionException {
        try {
            
            this.enable(); 
            
            if (!isEnable()) throw new BadCommandException("Actuator has to be enable prior ON command.");
            

            if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
                    setVelocity(); 
            } else if (mode.equals(
                    CanOpenEPOS.EposMode.CURRENT)) { 
                    if ( (this.motor.getCurrentValue() == 0)) {
                        throw new BadCommandException("A current value has to be set");
                    }
                    writeCurrent(this.motor.getCurrentValue());
               
            }
            this.on = true;
            this.sentValue = this.motor.getCurrentValue();
            this.updateReadValue();
            //this.sendToStatus(this.getStatusData());
            return getName() + " ON";
        } catch (SDORequestException ex) {
            log.error(getName() + " Can Open ERROR in sending the command ON.");
            throw new ErrorInCommandExecutionException(getName() + ": Error in command ON" + ex.toString());
        }

    }
    
        /**
     * This method set to zero the current value and stops the motor motion.
     * Then it disables the actuator.
     * @return 
     */
    public String off() throws BadCommandException, ErrorInCommandExecutionException {
        try {
            if (mode.equals(EposMode.VELOCITY)) {
               
                    stopVelocity();
                    shutdown();
                
            } else if (mode.equals(EposMode.CURRENT)) {

                    stopCurrent();
                    shutdown();
                
            }
            this.on = false;
            this.sentValue = 0;
            this.updateReadValue();
            //this.sendToStatus(this.getStatusData());
            return getName() + " OFF";
        } catch (SDORequestException ex) {
            log.error(ex.getMessage());
            throw new ErrorInCommandExecutionException(ex + "Error in reading SDO request");
        }
    }

    
    /*********************************/
    /*Methods connected to the motors*/
    /*********************************/
    
    /**
     * Write the values of the motor fields in the actuator memory.
     * @throws SDORequestError 
     */
    public void writeMotorParameters() throws SDORequestException {
        shutdown();    
        writeMotorType(this.motor.getMotorType());
        
        // mandatory in Current mode
        writeMotorData("01","2", this.motor.getContinuousCurrentLimit());
        writeMotorData("04","4", this.motor.getMaxSpeedInCurrentMode());
        writeMotorData("05","2", this.motor.getThermalTimeConstantWinding());
        //Guillaume wants to set this parameters too :
        writeMotorData("02","2", this.motor.getOutputCurrentLimit());
        writeMotorData("03","1", this.motor.getPolePairNumber());
        writePositionSensorType(this.motor.getPositionSensorType());
              
        //TODO mandatory in velocity mode
        
    }
    
    /**
     * This command set the motor type on the actuator.
     */
    public void writeMotorType(int value) throws SDORequestException {
        String typeInHexa = Integer.toHexString(value);
        writeSDO("6402","0","2",typeInHexa);
    }
    
    /**
     * This command read the motor type on the actuator.
     * @return the motor type in decimal
     */
    public int readMotorType() throws SDORequestException {
        String motorTypeInHexa = readSDO("6402","0");
        return Integer.parseInt(motorTypeInHexa, 16);
    }
    
     /**
     * This command read the position sensor type on the actuator.
     * @return the position sensor type in decimal
     */
    public int readPositionSensorType() throws SDORequestException {
        String positionSensorTypeInHexa = readSDO("2210","02");
        return Integer.parseInt(positionSensorTypeInHexa, 16);
    }
    
    public void writePositionSensorType(int value) throws SDORequestException {
        this.motor.setPositionSensorType(value);
        String valueInHexa = Integer.toHexString(value);
        writeSDO("2210","2","2", valueInHexa);
    }
    
    public void writeMaxSpeed(int value) throws SDORequestException {
        this.motor.setMaxSpeedInCurrentMode(value);
        writeMotorData("4", "4", value );
    }
    
    /**
     * This method writes the parameters for the motor on the actuator.
     * The parameters for the motor data are stored in the index 6410.
     * @param subindex
     * @param sizeInHexa size of the parameter in Can Open
     * @param value FORMAT=int the value of the parameter in decimal format
     */
    protected void writeMotorData(String subindex, String sizeInHexa, int value) throws SDORequestException {
        String valueInHexa = Integer.toHexString(value);
        writeSDO("6410",subindex,sizeInHexa,valueInHexa);
    }
    
    /**
     * This methods reads the parameters of the motor stored in the actuator.
     * (index 6410)
     * @param subindex
     * @return value FORMAT=int the value of the parameter in decimal format
     */
    public int readMotorData(String subindex) throws SDORequestException {
        //TODO this should return a short or a long as it deviceErrorFile coded with 2 or 4 bytes.
        String valueInHexa = readSDO("6410",subindex);
        return Integer.parseInt(valueInHexa, 16);
    }
    
    /**
     * This methods read the parameters of the motor stored in the actuator 
     * (hardware configuration)and compare with the configuration stored in the
     * Configuration Data Base (software configuration).
     * @return true if the hardware and software configuration are the same,
     *         false otherwise.
     * @throws MismatchMotorConfigurationError 
     */
    public boolean checkMotorParameters() throws ClampMotorConfigurationException, SDORequestException {
        boolean ok = true;
              
        //motor type
        int readMotorType = readMotorType();
        if (!(readMotorType == motor.getMotorType())) ok = false;
        
        //motor type
        int readPositionSensorType = readPositionSensorType();
        if (!(readPositionSensorType == motor.getPositionSensorType())) ok = false;
        
        //motor data
        int readContinuousCurrentLimit = readMotorData("01");
        if (!(readContinuousCurrentLimit == motor.getContinuousCurrentLimit())) ok = false;
        
        int readMaxSpeedInCurrentMode = readMotorData("04");
        if (!(readMaxSpeedInCurrentMode == motor.getMaxSpeedInCurrentMode())) ok = false;
        
        int readThermalTimeConstantWinding = readMotorData("05");
        if (!(readThermalTimeConstantWinding == motor.getThermalTimeConstantWinding())) ok = false;
        
        int readOutputCurrentLimit = readMotorData("02");
        if (!(readOutputCurrentLimit == motor.getOutputCurrentLimit())) ok = false;
        
        int readPolePairNumber = readMotorData("03");
        if (!(readPolePairNumber == motor.getPolePairNumber())) ok = false;
        
        if (!ok) {
            throw new ClampMotorConfigurationException("ERROR in hardware configuration of the motor", this.name,
                    readMotorType,readContinuousCurrentLimit,readMaxSpeedInCurrentMode,
                    readThermalTimeConstantWinding,readOutputCurrentLimit,readPolePairNumber);           
        }
        return ok;
    }
    
    public String displayMotorParameters() throws SDORequestException {
        StringBuilder sb = new StringBuilder("Read decimal values for motor parameters are :");
              
        //motor type
        int readMotorType = readMotorType();
        
        //position sensor type
        int readPositionSensorType = readPositionSensorType();
        
        //motor data
        int readContinuousCurrentLimit = readMotorData("01");      
        int readMaxSpeedInCurrentMode = readMotorData("04");
        int readThermalTimeConstantWinding = readMotorData("05");       
        int readOutputCurrentLimit = readMotorData("02");
        int readPolePairNumber = readMotorData("03");

        sb.append(" motor type ="); sb.append(readMotorType);
        sb.append(" position sensor type ="); sb.append(readPositionSensorType);
        sb.append(" continuousCurrentLimit ="); sb.append(readContinuousCurrentLimit);
        sb.append(" maxSpeedInCurrentMode ="); sb.append(readMaxSpeedInCurrentMode);
        sb.append(" thermalTimeConstantWinding ="); sb.append(readThermalTimeConstantWinding);
        sb.append(" outputCurrentLimit ="); sb.append(readOutputCurrentLimit);
        sb.append(" polePairNumber ="); sb.append(readPolePairNumber);
        return sb.toString();
    }
    
    /*Methods available in CURRENT mode*/ 
    /**
     * In current mode this methods send a current to the motor.
     * This make run the motor.
     * @param aValue UNIT=mA / FORMAT=decimal the value of the current to be sent.
     * @throws BadCommandException 
     * 
     */
    public void writeCurrent(int aValue) throws BadCommandException, SDORequestException {
        if (!isEnable()) 
            throw new BadCommandException(getName() + "is not ENABLE");
        if (!mode.equals(CanOpenEPOS.EposMode.CURRENT)) 
            throw new BadCommandException(getName() + "is not in CURRENT mode");       
        if (aValue > motor.getContinuousCurrentLimit())
            throw new IllegalArgumentException(aValue + " is greater than Continuous Current Limit");
        
        motor.setCurrentValue(aValue);
        String currentValueInHexa = Integer.toHexString(aValue);
        writeSDO("2030","0","2",currentValueInHexa);
            //TODO check the return code of writeSDO
    }
    
    /**
     * In Current Mode this methods returns the current actualy received by the motor.
     * @return the current UNIT=mA / FORMAT=decimal 
     * @throws BadCommandException 
     */
    public int readCurrent() throws BadCommandException, SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
            String currentInHexa = readSDO("6078","0");
            return Integer.parseInt(currentInHexa, 16);
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
    }
    
    /**
     * In current mode this methods set to zero the value of the current sent to
     * the motor.
     * This stops the motor.
     * @throws BadCommandException 
     */
    public void stopCurrent() throws BadCommandException, SDORequestException {
        if (mode.equals(CanOpenEPOS.EposMode.CURRENT)) {
            writeSDO("2030","00","2","0");
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
        
    }
    
        /*Methods available in VELOCITY mode*/ 
    public void setMaxiVelocity(int velocity) {
        if (velocity > this.motor.maxProfileVelocity) 
            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
        motor.maxProfileVelocity = velocity;
        //writeSDO("607F","00",velocity);
    }
    
    public void setMaxiAcceleration(int acceleration) {
        if (acceleration > this.motor.maxAcceleration ) 
            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
        motor.maxAcceleration = acceleration;
        //writeMaxonMotor(0x60C5,00,acceleration);
    }
    
    public void setVelocity() throws BadCommandException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            //writeMaxonMotor(0x206B,00,this.velocityValue);
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");

    }
    
    public int readVelocity()throws BadCommandException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            //String velocityInHexa = readSDO(0x2028,00);
            return 0;
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");
        
    } 
    
    public void stopVelocity() throws BadCommandException {
        if (mode.equals(CanOpenEPOS.EposMode.VELOCITY)) {
            //writeMaxonMotor(0x206B,00,0);
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");
        
    }
    
    
    
}
