/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

/**
 *
 * @author virieux
 */
public class CanOpenDevice extends Module implements PieceOfHardware{
    
    CanOpenProxy tcpProxy;
    
    String serialNB;
    String nodeID;
    boolean initialized;

    public CanOpenDevice(String aName, int aTickMillis,  String nodeID, String serialNB) {
        super(aName, aTickMillis);
        this.nodeID = nodeID;
        this.serialNB = serialNB;      
    }
    
    
    @Override
    public String getSerialNB() {
        return this.serialNB;
    }

    @Override
    public String getNodeID() {
        return this.nodeID;
    }
    
    
    @Override
    public boolean isNodeIdBootedAndOK() {
        return tcpProxy.isNodeIdBootedAndOK(this);
    }

    @Override
    public boolean isInitalized() {
        return this.initialized;
    }
    
    @Override
    public void initModule() {
        tcpProxy = (CanOpenProxy) this.getModule("tcpProxy");
        this.initialized = false;
    }

    /**
     * This methods has to be overridden if there is some initialization to do for the device.
     * @return a message
     * @throws Exception 
     */
    @Override
    public String initializeHardware() throws Exception {
        this.initialized = true;
        return getName() + ": no hardware initialization for this device to be done.";
    }
    
    public String writeSDO(String index, String subindex,String length, String newValue) throws SDORequestException {
        return tcpProxy.writeSDO(this.nodeID,index,subindex,length,newValue);
    }
    
    public String readSDO(String index, String subindex) throws SDORequestException {
        return tcpProxy.readSDO(this.nodeID, index, subindex);
    }
    
        
    
    /**************************************************************************/
    /*Methods to read or display the errors that have occured on the device   */
    /*The errors are stored in 2 places :
    /* error register at index 1001 
    /* error history at index 1003               
    /* The error register and the error history 
    /**************************************************************************/
    
        
    /**
     * At index 1001 there deviceErrorFile an error register for the device. 
     * The device maps internal errors in this byte.
     * @return error code in Hexa
     */
    public String readErrorRegister() throws SDORequestException {
        String error = (String) readSDO("1001", "0");
        return String.format("%02x", Integer.parseInt(error));
    }
    
    public String displayErrorRegister() throws SDORequestException {
        String errorInHexa = readErrorRegister();      
        String errorName = CanOpenErrorsTable.errorRegisterCodes.getProperty(errorInHexa);
        log.debug("error register=" + errorInHexa);
        return errorName;
    }
    
    /**
     * The error history holds errors that have occurred on the device and 
     * have been signalled via the emergency object.
     * This methods reads the Maxon Motor Error History and returns an array of 
     * error code.
     * @return errors : a list of device error code
     */
    public String[] readErrorHistory() throws SDORequestException {              
        int numberOfErrors = readNumberOfErrors();
        String[] errors = new String[numberOfErrors];
        for (int i=0; i < numberOfErrors;i++) {
            String subindex = Integer.toHexString(i+1);
            errors[i] = (String) readSDO("1003", subindex);
        }
        return errors;
    }
    

    public String displayErrorHistory() throws SDORequestException {
        String[] errorHistory = readErrorHistory();
        StringBuilder sb = new StringBuilder("Error history contains " + errorHistory.length + " errors.");
        if (!(errorHistory.length == 0)) {
            sb.append("List of errors in history : "
                    + "the newest is the first, the oldest the last\n");
            for (int ix=0; ix < errorHistory.length; ix++) {
                sb.append("Error code (in hexa)=");sb.append(errorHistory[ix]);
                sb.append("/ error name=");
                sb.append(CanOpenErrorsTable.deviceErrorCodes.getProperty(errorHistory[ix].toUpperCase()));
                sb.append("\n");
            }
        }
        return sb.toString();
    }
    

    
    /**
     * Extract from the Maxon Motor firmware specification :
     * Contains the number of actual errors that are recorded in the array starting 
     * at subindex 1. Writing a “0” deletes the error history (empties the array). 
     * Values higher then “0” (zero) are not allowed to write."
     * 
     * Read and returns the number of errors registred in the Error History 
     */
    public int readNumberOfErrors() throws SDORequestException {
        String errorNBinHEXA = (String) readSDO("1003", "0");
        return Integer.parseInt(errorNBinHEXA, 16);
    }
    
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());       
        sb.append("/SerialNumber=");
        sb.append(this.serialNB);
        sb.append("/NodeID_in_hexa=");
        sb.append(this.nodeID);
        int x = Integer.parseInt(this.nodeID,16);
        sb.append("/NodeID_in_decimal=");sb.append(x);sb.append("/");
        // because the configuration system can call toString before the object is completely built
        // for tracing purpose.
        if (tcpProxy == null ) return sb.toString();
        if (isNodeIdBootedAndOK())  {
            sb.append("is booted/");
            
        } else {
            sb.append("ERROR/");
        }       
        return sb.toString();
    }
}
