/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.concurrent.TimeoutException;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.state.State;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.PieceOfHardware;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.DeploymentException;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;

/**
 * This class gathers the description of the real hardware and the tools (methods)
 * to give access to this hardware.
 * The concrete classes which extends this abstract class will have to add the
 * fields for the pieces of hardware and to give the list of hardware in the method listHardware.
 * @author virieux
 */
public abstract class BridgeToCanOpenHardware extends Module implements BridgeToHardware {
    
    protected CanOpenProxy tcpProxy;

    private boolean hardwareInitialized = false;
    private boolean configurationChecked = false;

    public BridgeToCanOpenHardware(String aName, int aTickMillis, CanOpenProxy tcpProxy) {
        super(aName, aTickMillis);
        this.tcpProxy = tcpProxy;
    }

    
    /**
     * This method returns the list of hardware that this bridge will manage.
     * @return a list of PieceOfHardware
     * exemple :
     * public PieceOfHardware listHardware() {
     *      PieceOfHardware[] hardwareList = {clampActuatorXminus, clampActuatorXplus,  ai814, ao412, dio8n1, dio8n2};
     *      return hardwareList;
     * }
     * 
     */
    public abstract PieceOfHardware[] listHardware();

    public CanOpenProxy getTcpProxy() {
        return tcpProxy;
    }
    
    public void startThreadReader() {
        tcpProxy.startThreadReader();
    }
    
    
    
     /**
     * This method returns true if the C-Wrapper is connected to the tcp proxy.
     * The C-Wrapper has to send the tcpProxy clientName as a keyword before
     * any communication can begin. So if the C-Wrapper doesn't send the good
     * keyword or if it doesn't connect to the tcp proxy, this method returns 
     * false.
     */
    @Override
    public boolean isCWrapperConnected() {
        return tcpProxy.isReady(tcpProxy.getMyClientName());
    }
    
    
    /**
     * returns true if the hardware is booted, identified and initialized.
     */
    @Override
    public boolean isHardwareReady() {
        return tcpProxy.isHardwareReady();
    }
    
    
    
    @Override
    public void initModule() {
        
        log.info("INIT MODULE" + getName());       
        //PieceOfHardware[] hardwareList = {clampActuatorXminus, clampActuatorXplus,  ai814, ao412, dio8n1, dio8n2};
        PieceOfHardware[] hardwareList = listHardware();
        tcpProxy.hardwareList = hardwareList;
        int numberOfNodes = hardwareList.length;
        tcpProxy.setExpectedNodesNB(numberOfNodes);
        log.info("Number of pieces of hardware expected =" + numberOfNodes);
        
        
        //we have to read the errors table for the maxon motor and can open devices.
        try {
            CanOpenErrorsTable.loadDeviceErrorTable();
            CanOpenErrorsTable.loadErrorRegisterTable();
            CanOpenErrorsTable.loadCommunicationErrorTable();
        } catch (DeploymentException ex) {
            log.error(ex.getMessage() + ": Could not load CanOpen Error Tables");
            this.getSubsystem().updateState(State.InError, "Could not load CanOpen Error Tables");
        }
    }
    
    
     /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper. 
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     */
    public Object sendCanOpen(String command) throws InterruptedException, TimeoutException, BadCommandException {
        return tcpProxy.sendCanOpen(command);
    }

    @Override
    public PDOStorage readPDOs() throws HardwareException, BadCommandException {
        log.debug(name + "/readPDOS");
        return this.tcpProxy.readPDOs();
      
    }
    
    public String toString() {
        StringBuilder sb = new StringBuilder(this.name);
        sb.append("/hardware list={");
        for (PieceOfHardware p : this.listHardware() ) {
            sb.append(" ");
            sb.append(p.toString());sb.append(";");
        }
        sb.append("}");
        return sb.toString();
    }
}
