package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.concurrent.locks.Condition;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenProxy.PDOStorage;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;
import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * An abstract class which extends Module to model a clamp that holds a filter on the carousel.
 * It implements pattern clampState.
 * See <a href="CarouselClampState.png"> the clampState machine diagram for a carousel clamp.</a>
 * Each clamp on the carousel is coupled with a sensor which detects the presence of a filter.
 * This abstract class provides a method to lock, unlock or release the clamps, and to compute the clampState 
 * of the clamp. 
 * The clampState of the clamp is computed, updated and published on the status bus each tick of the timer.
 * 
 * 
 * @author virieux
 */
public class FilterClampModule extends MobileItemModule {
    
    //to be able to read sensors from PDO
    private BridgeToHardware bridge;
      
    /**
     * The actuator which controls this clamp.
     */
    //private ClampActuatorModule actuator;
    private Actuator actuator;
    
     /**
     * The filterPresenceSensor detects where is the filter within the clamp.
     * @See enum PresenceFilterStateOnClamp
     */
    private Sensor14bits filterPresenceSensor;
    
    
    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private Sensor14bits lockSensor;
    
   /**
     * The thermometer mesures the temperature inside the clamp.
    */
    private Thermometer thermometer;

    protected FilterClampState clampState;
    protected FilterPresenceStatus filterPresenceStatus;
    private LockStatus lockStatus;
    
    protected double temperature;
    
    /**
     * If the lock sensor returns a value between 0 and lockSensorValueA:
     * the sensor is unlocked.
     * If the lock sensor returns a value between lockSensorValueA and lockSensorValueB:
     * we don't know.
     * if the sensor returns a value between lockSensorValueB and ValueC:
     * the clamp is locked.
     * If the sensor returns a value greater than valueC the sensor is in ERROR.
     */
    private int lockSensorValueA;
    private int lockSensorValueB;
    private int lockSensorValueC;
    private int lockSensorOffset;
    
    /**
     * The min and max value for the sensors.
     * Needed by the GUI.
     */
    private static int lockSensorMinValue = Sensor14bits.minValue;
    private static int lockSensorMaxValue = Sensor14bits.maxValue;
    
    private static int filterPositionMinValue = Sensor14bits.minValue;
    private static int filterPositionMaxValue = Sensor14bits.maxValue;
    
    
    /**
     * if the filter presence sensor returns a value between 0 and filterPositionValueA : 
     * the sensor is in error.
     */
    //@ConfigChanger( doc= "if the filter presence sensor returns a value between 0 and filterPositionValueA : the sensor is in error.")
    private int filterPositionValueA;
    
    /**
     * if filter presence sensor returns a value between filterPositionValueA and filterPositionValueB : 
     * the filter is engaged and lockable.
     */
    private int filterPositionValueB;
    
     /**
     * if filter presence sensor returns a value between filterPositionValueB and filterPositionValueC : 
     * we don't know. And if the value is greater than valueC, there's no filter.
     */   
    private int filterPositionValueC;
    private int filterPositionOffset;
    

    
    
    public String publishedByClampOutputName = "publishedByClamp";
    
    private boolean initialized = false;
    final Condition stateUpdated = lock.newCondition();
    
     /* This is used when we update the clamp clampState with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    protected long timeoutForUnlocking = 4000;
    protected long timeoutForReleasing = 4000;

    public FilterClampModule(String aName, 
            int aTickMillis, 
            Sensor14bits filterPresenceSensor, 
            Sensor14bits lockSensor, 
            Thermometer thermometer, 
            int filterPositionValueA, int filterPositionValueB, int filterPositionValueC, int filterPositionOffset,
            int lockSensorValueA, int lockSensorValueB, int lockSensorValueC, int lockSensorOffset,
            int timeoutForUnlocking, int timeoutForReleasing
            ) {
        super(aName,aTickMillis);
        this.filterPresenceSensor = filterPresenceSensor;
        this.lockSensor = lockSensor;
        this.thermometer = thermometer;
        this.filterPositionValueA = filterPositionValueA;
        this.filterPositionValueB = filterPositionValueB;
        this.filterPositionValueC = filterPositionValueC;
        this.filterPositionOffset = filterPositionOffset;
        this.lockSensorValueA = lockSensorValueA;
        this.lockSensorValueB = lockSensorValueB;
        this.lockSensorValueC = lockSensorValueC;
        this.lockSensorOffset = lockSensorOffset;
        this.timeoutForUnlocking = timeoutForUnlocking;
        this.timeoutForReleasing = timeoutForReleasing;
    }






    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/


    //public ClampActuatorModule getActuator() {
    public Actuator getActuator() {
        return actuator;
    }

    //public void setActuator(ClampActuatorModule actuator) {
    public void setActuator(Actuator actuator) {
        this.actuator = actuator;
    }
    
    
    /**
     * @param filterPositionValueA the filterPositionValueA to set
     */
    @ConfigChanger
    public void setFilterPositionValueA(int valueA) {
        this.filterPositionValueA = valueA;
    }

    /**
     * @param filterPositionValueB the filterPositionValueB to set
     */
    @ConfigChanger
    public void setFilterPositionValueB(int valueB) {
        this.filterPositionValueB = valueB;
    }

    /**
     * @param filterPositionValueC the filterPositionValueC to set
     */
    @ConfigChanger
    public void setFilterPositionValueC(int valueC) {
        this.filterPositionValueC = valueC;
    }






    /**
     * @return the lockSensorValueA
     */
    public int getLockSensorValueA() {
        return lockSensorValueA;
    }

    /**
     * @param lockSensorValueA the lockSensorValueA to set
     */
    @ConfigChanger
    public void setLockSensorValueA(int aValue) {
        this.lockSensorValueA = aValue;
    }

    /**
     * @return the lockSensorValueB
     */
    public int getLockSensorValueB() {
        return lockSensorValueB;
    }

    /**
     * @param lockSensorValueB the lockSensorValueB to set
     */
    @ConfigChanger
    public void setLockSensorValueB(int aValue) {
        this.lockSensorValueB = aValue;
    }

    public int getLockSensorValueC() {
        return lockSensorValueC;
    }
    
    @ConfigChanger
    public void setLockSensorValueC(int lockSensorValueC) {
        this.lockSensorValueC = lockSensorValueC;
    }

    public int getFilterPositionMaxValue() {
        return filterPositionMaxValue;
    }



    public int getFilterPositionMinValue() {
        return filterPositionMinValue;
    }



    public int getLockSensorMaxValue() {
        return lockSensorMaxValue;
    }



    public int getLockSensorMinValue() {
        return lockSensorMinValue;
    }

    @ConfigChanger
    public void setLockSensorOffset(int lockSensorOffset) {
        this.lockSensorOffset = lockSensorOffset;
    }

    /**
     * @return the filterPositionValueA
     */
    public int getFilterPositionValueA() {
        return filterPositionValueA;
    }

    /**
     * @return the filterPositionValueB
     */
    public int getFilterPositionValueB() {
        return filterPositionValueB;
    }

    /**
     * @return the filterPositionValueC
     */
    public int getFilterPositionValueC() {
        return filterPositionValueC;
    }

    public int getLockSensorOffset() {
        return lockSensorOffset;
    }

    public int getFilterPositionOffset() {
        return filterPositionOffset;
    }

    public void setFilterPositionOffset(int filterPositionOffset) {
        this.filterPositionOffset = filterPositionOffset;
    }
    
    
    
     /**
     * For encapsulation : we want to access to filterPresenceSensor in sub-packages.
     * @return the filterPresenceSensor
     */
    public Sensor14bits getFilterPresenceSensor() {
        return filterPresenceSensor;
    }
    
        /**
     * @forSpring
     * @return the lockSensor
     */
    public Sensor14bits getLockSensor() {
        return lockSensor;
    }

    public Thermometer getThermometer() {
        return thermometer;
    }

    /**
     * @return the lockStatus
     */
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * @param lockStatus the lockStatus to set
     */
    public void setLockStatus(LockStatus lockStatus) {
        this.lockStatus = lockStatus;
    }



    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public double getTemperature() {
        return temperature;
    }

    public long getTimeoutRelease() {
        return timeoutForReleasing;
    }
    
    @ConfigChanger
    public void setTimeoutRelease(long timeoutRelease) {
        this.timeoutForReleasing = timeoutRelease;
    }

    public long getTimeoutUnlock() {
        return timeoutForUnlocking;
    }

    @ConfigChanger
    public void setTimeoutUnlock(long timeoutUnlock) {
        this.timeoutForUnlocking = timeoutUnlock;
    }
    
   

    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/
    
    @Override
    public void initModule() {
        this.bridge =  (BridgeToHardware) getModule("bridge");
        this.clampState = FilterClampState.UNDEFINED;
        this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
        this.lockStatus = LockStatus.UNKNOWN;
        this.temperature = 0;
    }
    
        /**
     * Returns the clampState of the clamp. 
     * If the clampState is being updated and waiting 
     * for a response from a sensor, this methods waits until the clampState is updated.
     * If the clampState is not being updated, it returns immediatly the clampState.
     * 
     * @return clampState
     * 
     */

    public FilterClampState getClampState() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return clampState;
            
        } finally {
            lock.unlock();
        }
    }

     /**
     * This methods returns true if the clamp is locked.
     * In the simulator, it is overriden.
     * @return 
     */
    public boolean isLocked() {
        return (this.getLockStatus().equals(LockStatus.LOCKED));
    }

    public synchronized boolean isFilterEngaged() {
        return (this.filterPresenceStatus.equals(FilterPresenceStatus.LOCKABLE));
    }


    
    /**
     * Update the field lockPresenceStatus in reading the filter presence sensor.
     * @throws HardwareError 
     */
    public void updateFilterPresenceStatus() throws HardwareException {
            //READ filter presence sensor
            this.filterPresenceSensor.updateValue();
            int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
            int mechaValue = newFilterPresenceSensorValue - this.filterPositionOffset;
            if (mechaValue < this.filterPositionValueA) {
                this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    log.error(getName() + " ERROR new read value FOR FILTER POSITION SENSOR  = " 
                            + newFilterPresenceSensorValue);
                    throw new SensorValueOutOfRangeException("FILTER POSITION SENSOR", 
                            this.filterPresenceSensor.getName(),FilterClampModule.filterPositionMinValue, 
                            this.filterPositionValueA, mechaValue);
                
            } else if (mechaValue < this.filterPositionValueB) {
                this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;     
                
            } else if (mechaValue < this.filterPositionValueC) {
                this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
                
            } else {
                this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            } 
    }
    
    private void updateFilterPresenceStatus(PDOStorage pdoStorage) throws HardwareException {
        int newFilterPresenceSensorValue = filterPresenceSensor.updateValue(pdoStorage);
            int mechaValue = newFilterPresenceSensorValue - this.filterPositionOffset;
            if (mechaValue < this.filterPositionValueA) {
                this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                    log.error(getName() + " ERROR new read value FOR FILTER POSITION SENSOR  = " 
                            + newFilterPresenceSensorValue);
                    throw new SensorValueOutOfRangeException("FILTER POSITION SENSOR", 
                            this.filterPresenceSensor.getName(),FilterClampModule.filterPositionMinValue, 
                            FilterClampModule.filterPositionMaxValue, mechaValue);
                
            } else if (mechaValue < this.filterPositionValueB) {
                this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;     
                
            } else if (mechaValue < this.filterPositionValueC) {
                this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
                
            } else {
                this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;
            } 
    }
    
    /**
     * Update the field lockStatus in reading the lock sensor.
     * @throws HardwareError 
     */
    public void updateLockStatus() throws HardwareException {
        
        //READ lock sensor
        this.lockSensor.updateValue();

        int newLockSensorValue = lockSensor.getValue();
        int mechaValue = newLockSensorValue - this.lockSensorOffset;

        //log.info(getName() + " NEWVALUE FOR LOCK SENSOR=" + newLockSensorValue);

        if ( mechaValue < lockSensorValueA) {
            this.lockStatus = LockStatus.UNLOCKED;

        } else if (mechaValue <= lockSensorValueB){
            this.lockStatus = LockStatus.UNKNOWN;


        } else if (mechaValue <= lockSensorValueC) {
            this.lockStatus = LockStatus.LOCKED;

        } else {
            this.lockStatus = LockStatus.ERROR;
                log.error(getName() + " ERROR new read value FOR LOCK SENSOR  = " + newLockSensorValue);
                throw new SensorValueOutOfRangeException("ERROR in reading LOCK SENSOR", 
                        this.lockSensor.getName(),FilterClampModule.lockSensorMinValue, FilterClampModule.lockSensorMaxValue, mechaValue);
        }
        
    }
    
    /**
     * Update lockStatus from parameter pdoStorage
     * @param pdoStorage
     * @throws HardwareError 
     */
    private void updateLockStatus(PDOStorage pdoStorage) throws HardwareException {
        int newLockSensorValue = lockSensor.updateValue(pdoStorage);
        int mechaValue = newLockSensorValue - this.lockSensorOffset;

        //log.info(getName() + " NEWVALUE FOR LOCK SENSOR=" + newLockSensorValue);

        if ( mechaValue < lockSensorValueA) {
            this.lockStatus = LockStatus.UNLOCKED;

        } else if (mechaValue <= lockSensorValueB){
            this.lockStatus = LockStatus.UNKNOWN;


        } else if (mechaValue <= lockSensorValueC) {
            this.lockStatus = LockStatus.LOCKED;

        } else {
            this.lockStatus = LockStatus.ERROR;
                log.error(getName() + " ERROR new read value FOR LOCK SENSOR  = " + newLockSensorValue);
                throw new SensorValueOutOfRangeException("ERROR in reading LOCK SENSOR", 
                        this.lockSensor.getName(),FilterClampModule.lockSensorMinValue, FilterClampModule.lockSensorMaxValue, mechaValue);
        }
    }
    
    
    /**
     * This method updates the clamp clampState regarding the value returned by the 
     * filter presence sensor and the value returned by the method isLocked().
     * When the update is completed, it sends a signal to threads waiting to get
     * the new value of clampState.
     */
    @Deprecated
    public void updateStateWithSensorsFromSDO() throws HardwareException {
        
        lock.lock();
        updatingState = true;   
        
        try {

            this.updateFilterPresenceStatus();
            
            this.updateLockStatus();

            this.clampState = this.computeClampState();
                  
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }
        
        //this.sendToStatus(this.getStatusData());
        StatusDataPublishedByClamp status = this.getStatusData();
        this.sendToStatus(status);
        this.publishTrendingData(status);

    }
    
     /**
     * This method updates the clamp clampState regarding the value returned by the 
     * filter presence sensor and the value returned by the method isLocked().
     * When the update is completed, it sends a signal to threads waiting to get
     * the new value of clampState.
     */

    public void updateStateWithSensors() throws HardwareException, BadCommandException {
        updateStateWithSensors(this.bridge.readPDOs());
    }
    
    
    //void updateStateWithSensorsFromSDO(HashMap<String, String> sensorsValues) throws HardwareError {
    void updateStateWithSensors(PDOStorage pdoStorage) throws HardwareException {
        lock.lock();
        updatingState = true;   
        
        try {

            this.updateFilterPresenceStatus(pdoStorage);
            
            this.updateLockStatus(pdoStorage);

            this.clampState = this.computeClampState();
                  
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }
        
        StatusDataPublishedByClamp status = this.getStatusData();
        this.sendToStatus(status);
        this.publishTrendingData(status);
    }
    
    /**
     * Broadcast the trending data on the status bus.
     * @param status 
     */
    public void publishTrendingData(StatusDataPublishedByClamp status) {
        
        long timeStamp = System.currentTimeMillis();
        ArrayList<ValueNotification> trendingValues
          = new ArrayList<>();
        trendingValues.add(new ValueNotification(getName() + "/clampState", status.getClampState(),
                                                 timeStamp));
        trendingValues.add(new ValueNotification(getName() + "/filterPositionInClamp", status.getFilterPositionInClamp(),
                                                 timeStamp));
        trendingValues.add(new ValueNotification(getName() + "/filterPositionSensorValue", status.getFilterPositionSensorValue(),
                                                 timeStamp));
        trendingValues.add(new ValueNotification(getName() + "/lockStatus", this.lockStatus,
                                                 timeStamp));
        trendingValues.add(new ValueNotification(getName() + "/lockSensorValue", status.getLockSensorValue(),
                                                 timeStamp));
        publishData(trendingValues);
        
    }

    
    
    
    /**
     * Compute the global state of the clamp given the lock sensor and the presence filter sensor state.
     * This has to be overriden for the clamp X-.
     * @return clamp state
     */
    public FilterClampState computeClampState() {
        
        
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.ERROR) 
                || (this.lockStatus.equals(LockStatus.ERROR))) 
            return  FilterClampState.ERROR;

        // a filter is in the socket
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.LOCKABLE)) {

            if (this.lockStatus.equals(LockStatus.LOCKED)) {
                return FilterClampState.CLAMPEDONFILTER;

            } else if (this.lockStatus.equals(LockStatus.UNLOCKED)) {
                return  FilterClampState.UNCLAMPEDONFILTER;
            } else {
                return FilterClampState.UNDEFINED;
            }
        }

        // no filter
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.NOFILTER)) {
            if (this.lockStatus.equals(LockStatus.LOCKED)) {
                return  FilterClampState.READYTOCLAMP;

            } else if (this.lockStatus.equals(LockStatus.UNLOCKED)) {
                return  FilterClampState.UNCLAMPEDEMPTY;

            } else {
                return  FilterClampState.ERROR;
            }
        }

        // NOT_LOCKABLE (UNDEFINED)
        if (this.filterPresenceStatus.equals(FilterPresenceStatus.NOT_LOCKABLE)) {
            return FilterClampState.UNDEFINED;
        }
        
        return FilterClampState.UNDEFINED;

    }
    

      
//    @Override
//    public void startTicking() {
//        //TODO when the carousel is not at standby position, it's not necessary to read the sensors.
//        updateStateWithSensors; // we can't do that before all the hardware has been connected and is OK
//        super.startTicking();
//    }
   

//    @Override
//    public void tick() {
//        if (initialized) {
//            this.updateStateWithSensors();
//            sendToStatus(getStatusData());
//        } else {
//            if (((BridgeToHardware) this.getModule("bridge")).isHardwareReady()) {
//                initialized = true;
//            }
//        }
//
//    }


    public StatusDataPublishedByClamp getStatusData() {
        StatusDataPublishedByClamp status = FcsUtils.createStatusDataPublishedByClamp(this);
        return status;
    }
    
    /**
     * The clamps on the carousel are locked automaticaly when the filter comes
     * at the standby position. To be able to lock automaticaly again, it has to be
     * released after each time it has been unlocked. 
     * @return
     * @throws HardwareError
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     */
    public String release() throws HardwareException, BadCommandException, ErrorInCommandExecutionException {
        log.info("Checking conditions for release clamp " + getName() + " on socket at standby position.");
        
        updateStateWithSensors();
        if (!getClampState().equals(FilterClampState.UNCLAMPEDEMPTY)) 
            throw new BadCommandException("Can't release a clamp if isn't unclamped and empty.");
        
        
        
        log.info("Releasing clamp " + getName() + " on socket at standby position.");
        
        return this.executeAction(MobileItemAction.RELEASE, timeoutForReleasing);
        
    }
    
     /**
     * Unlock the clamp when a filter is locked by the clamp.
     * This is used only in engineering mode.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */
    public String unlock() 
            throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        
        log.info(getName() + ": "
                + "UNLOCK State1 = " + clampState.toString());
        updateStateWithSensors();
        
        if (!getClampState().equals(FilterClampState.CLAMPEDONFILTER)) 
            throw new BadCommandException("Can't unlock a clamp if it is already unlocked.");
       
        
        
       
        BasicAutoChangerModule autochanger = (BasicAutoChangerModule) this.environment.getComponentByName("autochanger");
        if (autochanger == null) throw new BadCommandException("NO AUTOCHANGER");
        
        if (!autochanger.isHoldingFilterAtStandby()) throw new BadCommandException("CANNOT UNLOCK A CLAMP if FILTER is not HELD by autochanger.");
        
        return this.executeAction(MobileItemAction.UNLOCK, timeoutForUnlocking);
        
    } 
    
    
    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException {
        if (action.equals(MobileItemAction.UNLOCK)) {
            this.actuator.on();
        } else if (action.equals(MobileItemAction.RELEASE)) {
            this.actuator.off();
        } else throw new IllegalArgumentException("Action on clamp must be UNLOCK or RELEASE");
        
    }
    
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.UNLOCK)) {
            return this.getClampState().equals(FilterClampState.UNCLAMPEDONFILTER);
        } else if (action.equals(MobileItemAction.RELEASE)) {
            return this.getClampState().equals(FilterClampState.READYTOCLAMP);
        } else {
            throw new IllegalArgumentException("Action on clamp must be UNLOCK or RELEASE");
        }
    }
    
    @Override
    public void postAction(MobileItemAction action) {
    }
    
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws HardwareException, BadCommandException {
        this.updateStateWithSensors();
    }
    
    /**
     * This methods read the thermometer, update the field temperature and returns the value sent
     * by the thermometer;
     * @return 
     */
    public double updateTemperature() throws HardwareException {
        double temp = this.thermometer.readTemperature();
        this.temperature = temp; 
        return temp;
    }
    
    @Override
    public String toString() {
//        StringBuilder sb = new StringBuilder(super.toString());
//        sb.append("Name=");sb.append(this.getName());
        //sb.append("/actuator=");sb.append(String.valueOf(this.actuator.getName()));
        StringBuilder sb = new StringBuilder(this.getName());
        sb.append("/filterPresenceSensor=");sb.append(this.filterPresenceSensor.getName());
        sb.append("#valueA=");sb.append(this.filterPositionValueA);
        sb.append("#valueB=");sb.append(this.filterPositionValueB);
        sb.append("#valueC=");sb.append(this.filterPositionValueC);
        sb.append("/lockSensor=");sb.append(this.lockSensor.getName());
        sb.append("#valueA=");sb.append(this.lockSensorValueA);
        sb.append("#valueB=");sb.append(this.lockSensorValueB);
        sb.append("#valueC=");sb.append(this.lockSensorValueC);
        return sb.toString();
    }










    
	
}
