/*
 * To change this template, choose Tools | Templates
 * and on the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.concurrent.locks.Condition;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceInLatchStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.MobileItemModule;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for a latch. 
 * A latch is what holds a filter in the trucks of the autochanger.
 * There is 2 latches : one on each side of the filter.
 * The state of the  latches are monitoring by digital sensors :
 * one sensor to know if a filter is in the latch or not and 
 * 2 status sensors to know if the latch is locked or unlocked.
 * If the 2 status sensors are inconsistent, the latch is in error.
 * @author virieux
 */
public class FilterLatchModule extends MobileItemModule {






    private static class LatchError extends Exception {
        FilterLatchModule latch;
               
        public LatchError(String string, FilterLatchModule aThis) {
            super(string);
            latch = aThis;
        }
    }


    private boolean filterEngaged;
    private boolean locked;
    private boolean unlocked;
    private boolean inError;
    
    private FilterPresenceInLatchStatus presenceStatus;
    private LockStatus lockStatus;
    
    LatchActuatorModule latchActuator;
    protected NumericSensor filterPresenceSensor;
    protected NumericSensor lockSensor;
    protected NumericSensor unlockSensor;
    CompactIOModule filterSensorsDIO;
     /**
     * This is the name of the DIO on which are plugged the filter sensors.
     * Used in the groovy file and in initModule.
     * 
     */
    String filterSensorsDIOName;
    
    
    //Used because we have to wait for the update from the sensors to know the state
    //of the latch
    final Condition stateUpdated = lock.newCondition();
    
     /**
     * This is used when we update the latch state with the values returned 
     *  by the sensors.
     */
    protected volatile boolean updatingState = false;
    
    //private long timeToWaitForCompleteOpen;
    //private long timeToWaitForCompleteClose;
    private long timeoutForOpening;       
    private long timeoutForClosing;
   
    
    
    /**************************************************************************************************/
    /********************** SETTERS AND GETTERS  ******************************************************/
    /**************************************************************************************************/
    
    public LatchActuatorModule getLatchActuator() {
        return latchActuator;
    }

    public void setLatchActuator(LatchActuatorModule latchActuator) {
        this.latchActuator = latchActuator;
    }

    public NumericSensor getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    public void setFilterPresenceSensor(NumericSensor filterPresenceSensor) {
        this.filterPresenceSensor = filterPresenceSensor;
    }

    public NumericSensor getLockSensor() {
        return lockSensor;
    }

    public void setLockSensor(NumericSensor lockSensor) {
        this.lockSensor = lockSensor;
    }

    public NumericSensor getUnlockSensor() {
        return unlockSensor;
    }

    public void setUnlockSensor(NumericSensor unlockSensor) {
        this.unlockSensor = unlockSensor;
    }

    public CompactIOModule getFilterSensorsDIO() {
        return filterSensorsDIO;
    }

    public void setFilterSensorsDIO(CompactIOModule filterSensorsDIO) {
        this.filterSensorsDIO = filterSensorsDIO;
    }

    public String getFilterSensorsDIOName() {
        return filterSensorsDIOName;
    }

    public void setFilterSensorsDIOName(String filterSensorsDIOName) {
        this.filterSensorsDIOName = filterSensorsDIOName;
    }

//    public long getTimeToWaitForCompleteClose() {
//        return timeToWaitForCompleteClose;
//    }
//
//    public void setTimeToWaitForCompleteClose(long timeToWaitForCompleteClose) {
//        this.timeToWaitForCompleteClose = timeToWaitForCompleteClose;
//    }
//
//    public long getTimeToWaitForCompleteOpen() {
//        return timeToWaitForCompleteOpen;
//    }
//
//    public void setTimeToWaitForCompleteOpen(long timeToWaitForCompleteOpen) {
//        this.timeToWaitForCompleteOpen = timeToWaitForCompleteOpen;
//    }

    public long getTimeoutForClosing() {
        return timeoutForClosing;
    }

    public void setTimeoutForClosing(long timeoutForClosing) {
        this.timeoutForClosing = timeoutForClosing;
    }

    public long getTimeoutForOpening() {
        return timeoutForOpening;
    }

    public void setTimeoutForOpening(long timeoutForOpening) {
        this.timeoutForOpening = timeoutForOpening;
    }
    
    


    /**************************************************************************************************/
    /********************** END OF SETTERS AND GETTERS  ***********************************************/
    /**************************************************************************************************/

    @Override
    public void initModule() {
        this.filterSensorsDIO = (CompactIOModule) this.getModule(this.filterSensorsDIOName);
        this.lockStatus = LockStatus.UNKNOWN;
        this.presenceStatus = FilterPresenceInLatchStatus.UNKNOWN;
        this.filterEngaged = false;
    }
    
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        if (action.equals(MobileItemAction.OPEN)) {
            return this.getLockStatus().equals(LockStatus.UNLOCKED);
        } else if (action.equals(MobileItemAction.CLOSE)) {
            return this.getLockStatus().equals(LockStatus.LOCKED);
        } else {
            throw new IllegalArgumentException("Action on latch must be OPEN or CLOSE");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception {
        filterSensorsDIO.updateValue();
        
        String filterSensorsHexaValue = filterSensorsDIO.getHexaValue();

        this.updateState(filterSensorsHexaValue);
    }

    @Override
    public void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        if (action.equals(MobileItemAction.OPEN)) {
            this.latchActuator.open();
        } else if (action.equals(MobileItemAction.CLOSE)) {
            this.latchActuator.close();        
        } else throw new IllegalArgumentException(getName() + "Action on latch must be OPEN or CLOSE");
    }

    @Override
    public void postAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
    }

    public LockStatus getLockStatus() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return lockStatus;
            
        } finally {
            lock.unlock();
        }
        
    }

    public void setLockStatus(LockStatus lockStatus) {
        this.lockStatus = lockStatus;
    }
    
    public FilterPresenceInLatchStatus getPresenceStatus() {
        
        lock.lock();
        try {
            while(updatingState) {
                try {
                    this.stateUpdated.await();
                } catch (InterruptedException ex) {
                    Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
                }

            }
            return presenceStatus;
            
        } finally {
            lock.unlock();
        }
        
    }

    
    public StatusDataPublishedByLatch getStatusData() {
        StatusDataPublishedByLatch status = FcsUtils.createStatusDataPublishedByLatch(this);
        return status;
    }

    
    public  String open()  throws BadCommandException, ErrorInCommandExecutionException, HardwareException, Exception {
        
        this.updateStateWithSensorsToCheckIfActionIsCompleted();
        
        if (this.isUnlocked()) return getName() + "is already OPEN";

        CarouselModule carousel = (CarouselModule) this.environment.getComponentByName("carousel");
        
        if (!carousel.isHoldingFilterAtStandby()) throw new BadCommandException(getName() + "CANNOT OPEN LATCH at STANDBY : CAROUSEL is NOT HOLDING THE FILTER");
               
        this.executeAction(MobileItemAction.OPEN, this.timeoutForOpening);
       

        //We want to wait to be sure that the latches are opened.
        //Here we can't avoid having Thread.sleep
//        try {
//            log.info(getName() + ": Waiting for the latch to complete opening at standby");
//            Thread.sleep(timeToWaitForCompleteOpen);
//        } catch (InterruptedException ex) {
//            log.error(ex);
//        }
        
        this.latchActuator.powerOff();
    
        return getName() + " is OPEN";
        
    }

    public String close() throws BadCommandException, ErrorInCommandExecutionException, HardwareException, Exception {
        
        this.updateStateWithSensorsToCheckIfActionIsCompleted();
        
        if (this.isLocked()) return getName() + "is already CLOSED";
               
        this.executeAction(MobileItemAction.CLOSE, this.timeoutForClosing);

//        
//        //We want to wait to be sure that the latches are opened.
//        //Here we can't avoid having Thread.sleep
//        try {
//            log.info(getName() + ": Waiting for the latch to close at standby");
//            Thread.sleep(timeToWaitForCompleteClose);
//        } catch (InterruptedException ex) {
//            log.error(ex);
//        }
        
        this.latchActuator.powerOff();
    
        return getName() + " is CLOSED";
        
    }

    //TODO : wait for the update of the state
    public boolean isLocked() {       
        return locked;
    }


    public boolean isUnlocked() {
        return unlocked;
    }
    

    public boolean isInError() {
        return inError;
    }

    
    
    public void updateState(String hexaValue) {
        lock.lock();
        try {
            updatingState = true; 
            this.filterPresenceSensor.updateValue(hexaValue);
            this.lockSensor.updateValue(hexaValue);
            this.unlockSensor.updateValue(hexaValue);
            
            if (this.filterPresenceSensor.getDigitalValue() == 1) {
                filterEngaged = true;
                this.presenceStatus = FilterPresenceInLatchStatus.ENGAGED;
            } else {
                filterEngaged = false;
                this.presenceStatus = FilterPresenceInLatchStatus.NOFILTER;
            }
            if (this.lockSensor.getDigitalValue() == 1) {
                locked = true;
            } else {
                locked = false;
            }
            if (this.unlockSensor.getDigitalValue() == 1) {
                unlocked = true;
            } else {
                unlocked = false;
            }
            if (locked && unlocked) {
                inError = true;
            } else {
                inError = false;
            }
            
            if (inError) {
                lockStatus = LockStatus.ERROR;
            } else if (locked) {
                lockStatus = LockStatus.LOCKED;
            } else if (unlocked) {
                lockStatus = LockStatus.UNLOCKED;
            } else {
                lockStatus = LockStatus.UNKNOWN;
            }

            

            
        } finally {
            updatingState = false;
            stateUpdated.signal();
            lock.unlock();
        }
        
        StatusDataPublishedByLatch status = this.getStatusData();
        this.sendToStatus(status); 
        publishTrendingData(status);

    }
    
     /**
     * Broadcast the trending data on the status bus.
     * @param status 
     */
    public void publishTrendingData(StatusDataPublishedByLatch status) {
        
        long timeStamp = System.currentTimeMillis();
        ArrayList<ValueNotification> trendingValues
          = new ArrayList<>();
        trendingValues.add(new ValueNotification(getName() + "/lockStatus", status.getLockStatus(),
                                                 timeStamp));
        trendingValues.add(new ValueNotification(getName() + "/lockSensorValue", status.getLockSensorValue(),
                                                 timeStamp));
        trendingValues.add(new ValueNotification(getName() + "/unlockSensorValue", status.getUnlockSensorValue(),
                                                 timeStamp));
        trendingValues.add(new ValueNotification(getName() + "/filterPositionSensorValue", status.getFilterPositionSensorValue(),
                                                 timeStamp));
        publishData(trendingValues);
        
    }


    
}
