/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.common;

import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.Alarm;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.state.State;
import org.lsst.ccs.subsystems.fcs.FcsAlarm;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.errors.HardwareException;

/**
 * This represents a Mechanical Item which can be moved by a motor.
 * Generaly we want to execute an action on this mechanical item :
 * - start the action,
 * - read sensors to know if the action is completed, 
 * - wait for the end of the action. 
 * This module provides a general way to start an action and wait until the action is completed.
 * It factorises some code that otherwise would have to be duplicated in each Module that needs such fonctionnalities.
 * @author virieux
 */
public abstract class MobileItemModule extends Module {

    protected boolean itemMoving = false;
    //Used because we have to wait for the update from the sensors to know if the action is completed
    public final Lock lock = new ReentrantLock();
    public final Condition motionCompleted = lock.newCondition();

    ScheduledFuture<?> readSensorsHandle;
    final ScheduledThreadPoolExecutor scheduler = new ScheduledThreadPoolExecutor(1);

    /**
     * This stops the reading of the sensors.
     */
    public void cancelReadingSensors() {
        lock.lock();
        try {
            log.debug(" => Stop reading sensors");
            motionCompleted.signal();
        } finally {
            lock.unlock();
        }
        this.readSensorsHandle.cancel(true);
        log.debug("readingSensors canceled");
    }

    /*
     * This method should returns true if the action is completed, false otherwise.
     * It's abstract because it'up to the sub-classes to know when the action is completed.
     */
    public abstract boolean isActionCompleted(MobileItemAction action);
    
    public abstract void updateStateWithSensorsToCheckIfActionIsCompleted() throws Exception;
    
    public abstract void startAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException;
    
    public abstract void postAction(MobileItemAction action) throws BadCommandException, ErrorInCommandExecutionException, HardwareException;
    
    /**
     * This executes an action which moves the MobileItem and waits for the end od this action.
     * The MobileItemAction is an ENUM from the fcs.bus package.
     * @param action, timeout
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException 
     */
    public String executeAction(MobileItemAction action, long timeoutForAction) throws BadCommandException, ErrorInCommandExecutionException, HardwareException {
        
        lock.lock();
        this.itemMoving = true;
        
        try {
            startAction(action);
            
            readSensorsUntilActionIsCompleted(action, System.currentTimeMillis(), timeoutForAction);
            
            waitForEndOfAction(action);
            
            if (isActionCompleted(action)) {
                //TODO postAction() exemple :
                //this.actuator.maintainUnclamped();              
                postAction(action);
                return getName() + " " + action.doneString();
            
            } else {
                throw new ErrorInCommandExecutionException(getName() + ": " +  action.getFailureMsg());
            }
            
        } finally {
            this.itemMoving = false;
            this.motionCompleted.signal();
            lock.unlock();
        }
    }
    
    /**
     * Start reading the sensors at a fixed rate (scheduler.scheduleAtFixedRate) until the action is completed or the timeout for this action is past.
     * @param action
     * @param beginTime
     * 
     * @param timeout 
     */
    public void readSensorsUntilActionIsCompleted(final MobileItemAction action, final long beginTime, final long timeout) {
        final Runnable readSensors = new Runnable() {

            @Override
            public void run() {

                try {
                    log.debug("Socket at standby : Reading sensors");
                    long duration = System.currentTimeMillis() - beginTime;
                    updateStateWithSensorsToCheckIfActionIsCompleted();
                    boolean actionCompleted = isActionCompleted(action);
                    if (!actionCompleted && (duration < timeout)) {
                        log.debug(action.toString() + " not completed.....");
                    } else if (actionCompleted) {
                        itemMoving = false;
                        log.debug(action.toString() + " ACTION COMPLETED");
                        cancelReadingSensors();
                    } else if (duration >= timeout) {
                        itemMoving = false;
                        log.debug(action.toString() + " ACTION NOT COMPLETED during allocated time");
                        cancelReadingSensors();
                        throw new TimeoutException(action.toString() + " clamps at standby was greater than timeout for this task: duration=" + duration + ",timeout=" + timeout);
                    }
                } catch (Exception ex) {
                    log.error("ERROR during action:" + ex.getMessage());
                    //TODO just for debugging 
                    ex.printStackTrace();
                    Alarm alarm = new FcsAlarm(ex.toString());
                    sendToStatus(alarm);
                    getSubsystem().updateState(State.InError, "ERROR DURING ACTION " + action.toString() + " "  + ex.toString());
                    cancelReadingSensors();
                }
            }
        };
        this.readSensorsHandle = scheduler.scheduleAtFixedRate(readSensors, 500, 500, TimeUnit.MILLISECONDS);
    }

    /**
     * This method waits until the action is completed.
     */
    public void waitForEndOfAction(MobileItemAction action) {
        lock.lock();
        try {
            while (itemMoving) {
                try {
                    log.debug("Waiting for end of " + action.toString());
                    motionCompleted.await();
                } catch (InterruptedException ex) {
                }
            }
            log.debug("STOP WAITING");
            log.info(action.toString() + " IS COMPLETED NOW");
        } finally {
            lock.unlock();
        }
    }
    
    /**
     * This shutdowns the scheduler.
     */
    @Override
    public void shutdownNow() {
        super.shutdownNow();
        scheduler.shutdown();
    }




    
}
